<?php
// This file handles the settings page functionality for the token-based plugin
// It's included in the main plugin file

if (!defined('ABSPATH')) exit;

// The settings page is handled in admin-pages.php
// This file now also contains access helpers used across the plugin.

/**
 * Access levels we support (ordered).
 */
function aialttext_token_get_access_levels() {
    // Highest to lowest
    return array(
        'full'     => __('Full Access', 'aialttext-token'),
        'partial'  => __('Partial Access', 'aialttext-token'),
        'generate' => __('Generate Only', 'aialttext-token'),
        'none'     => __('No Access', 'aialttext-token'),
    );
}

/**
 * Fetch role → access map. If unset, default admins to full, everyone else none.
 */
function aialttext_token_get_role_access_map() {
    $map = get_option('aialttext_token_role_access', array());
    if (!is_array($map)) $map = array();

    // Seed sane defaults if empty (don’t auto-grant beyond admins)
    if (empty($map) && function_exists('get_editable_roles')) {
        $roles = get_editable_roles();
        foreach ($roles as $role_key => $role) {
            $map[$role_key] = ($role_key === 'administrator') ? 'full' : 'none';
        }
    }
    return $map;
}

/**
 * Highest effective level for the current (or given) user across all their roles.
 * Ensures site admins can’t lock themselves out.
 */
function aialttext_token_get_user_access_level($user_id = 0) {
    if (!$user_id) {
        $user = wp_get_current_user();
    } else {
        $user = get_user_by('id', (int) $user_id);
    }
    if (!$user || empty($user->roles)) return 'none';

    // Admins always at least full.
    if (user_can($user, 'manage_options')) return 'full';

    $map = aialttext_token_get_role_access_map();
    $order = array('none' => 0, 'generate' => 1, 'partial' => 2, 'full' => 3);
    $best  = 'none';

    foreach ((array) $user->roles as $role) {
        $level = isset($map[$role]) ? $map[$role] : 'none';
        if (!isset($order[$level])) $level = 'none';
        if ($order[$level] > $order[$best]) $best = $level;
        if ($best === 'full') break;
    }
    return $best;
}

/**
 * Capability check by feature.
 *  - settings/account  → full
 *  - bulk               → partial or full
 *  - generate           → generate or partial or full
 */
function aialttext_token_user_can($feature, $user_id = 0) {
    $level = aialttext_token_get_user_access_level($user_id);

    switch ($feature) {
        case 'settings':
        case 'account':
            // Only Full access can manage account & settings
            return ($level === 'full');

        // Usage History – Full only (so Partial can no longer see it)
        case 'usage':
            return ($level === 'full');

        // Bulk processing (and anything treated as “bulk-level”) – Partial + Full
        case 'bulk':
            return ($level === 'partial' || $level === 'full');

        // Support – also Partial + Full
        case 'support':
            return ($level === 'partial' || $level === 'full');

        // Single-image generation – Generate / Partial / Full
        case 'generate':
            return ($level === 'generate' || $level === 'partial' || $level === 'full');

        default:
            return false;
    }
}


/**
 * Get default prompt template
 */
function aialttext_token_get_default_prompt_template() {
    return "Describe the image clearly for accessibility. Title: {title}. Context: {context}. Return only the alt text.";
}

/**
 * Sanitize prompt template
 */
function aialttext_token_sanitize_prompt_template($template) {
    // Allow basic HTML tags and placeholders
    $allowed_tags = array();
    return wp_kses($template, $allowed_tags);
}

/**
 * Validate batch size
 */
function aialttext_token_validate_batch_size($size) {
    $size = intval($size);
    if ($size < 1) $size = 1;
    if ($size > 200) $size = 200;
    return $size;
}

/**
 * Validate max images
 */
function aialttext_token_validate_max_images($max) {
    $max = intval($max);
    if ($max < 0) $max = 0;
    return $max;
}

/**
 * Get available integrations
 */
function aialttext_token_get_available_integrations() {
    $integrations = array();
    
    if (class_exists('WooCommerce')) {
        $integrations['woocommerce'] = array(
            'name' => 'WooCommerce',
            'description' => 'Include product information in alt text generation',
            'active' => get_option('aialttext_token_woocommerce_integration', 1)
        );
    }
    
    if (defined('WPSEO_VERSION') || class_exists('RankMath') || class_exists('AIOSEO\Plugin\Common\Main') || defined('SEOPRESS_VERSION')) {
        $integrations['seo'] = array(
            'name' => 'SEO Plugins',
            'description' => 'Include SEO keywords and meta descriptions in context',
            'active' => get_option('aialttext_token_seo_integration', 1)
        );
    }
    
    return $integrations;
}

/**
 * Check if token system is properly configured
 */
function aialttext_token_is_configured() {
    $email = get_option('aialttext_token_account_email', '');
    $access_key = get_option('aialttext_token_access_key', '');
    
    return !empty($email) && !empty($access_key);
}

/**
 * Get configuration status for admin notices
 */
function aialttext_token_get_config_status() {
    if (!aialttext_token_is_configured()) {
        return array(
            'status'     => 'incomplete',
            'message'    => __('Img Alt Gen Pro: Please login or sign-up to your account in the Account & Tokens page.', 'aialttext-token'),
            'action_url' => admin_url('admin.php?page=aialttext-token'),
            'action_text'=> __('Go to Account Page', 'aialttext-token'),
        );
    }
    
        // Check balance
        $balance = aialttext_token_check_balance();

        // If the balance check determined the key is invalid and cleared it,
        // treat this as "not configured" rather than "0 tokens".
        if ( ! aialttext_token_is_configured() ) {
            return array(
                'status'     => 'incomplete',
                'message'    => __('Img Alt Gen Pro: Please login or link your account in the Account & Tokens page.', 'aialttext-token'),
                'action_url' => admin_url('admin.php?page=aialttext-token'),
                'action_text'=> __('Go to Account Page', 'aialttext-token'),
            );
        }
    
        if ($balance <= 0) {
            return array(
                'status' => 'no_tokens',
                'message' => __('You have no tokens remaining. Purchase more tokens to continue generating alt text.', 'aialttext-token'),
                'action_url' => admin_url('admin.php?page=aialttext-token'),
                'action_text' => __('Purchase Tokens', 'aialttext-token'),
            );
        }
    
    if ($balance <= 5) {
        return array(
            'status' => 'low_tokens',
            'message' => sprintf(
                _n(
                    'You have %s token remaining. Consider purchasing more tokens soon.',
                    'You have %s tokens remaining. Consider purchasing more tokens soon.',
                    (int) $balance,
                    'aialttext-token'
                ),
                number_format_i18n( (int) $balance )
            ),
            'action_url' => admin_url('admin.php?page=aialttext-token'),
            'action_text' => __('Purchase Tokens', 'aialttext-token'),
        );
    }
    
    return array(
        'status' => 'configured',
        'message' => sprintf(
            __('Token system is configured. You have %s tokens available.', 'aialttext-token'),
            number_format_i18n( (int) $balance )
        ),
        'balance' => (int) $balance,
    );
}

// Add admin notices for configuration status
add_action('admin_notices', 'aialttext_token_admin_notices');

function aialttext_token_admin_notices() {
    // Only show on our plugin pages
    $screen = get_current_screen();
    if (!$screen || strpos($screen->id, 'aialttext-token') === false) {
        return;
    }

    // Don’t show the “go to account page” banner if we are already on it.
    $page      = isset($_GET['page']) ? sanitize_key((string) $_GET['page']) : '';
    $on_account= ($page === 'aialttext-token');

    $status = aialttext_token_get_config_status();

    if ($on_account && $status['status'] === 'incomplete') {
        return;
    }

    add_action('wp_ajax_aialttext_token_dismiss_incomplete', 'aialttext_token_dismiss_incomplete');
function aialttext_token_dismiss_incomplete() {
    aialttext_token_check_nonce();
    update_user_meta( get_current_user_id(), 'aialttext_token_dismiss_incomplete', '1' );
    wp_send_json_success();
}
    
    if ($status['status'] === 'incomplete') {
        echo '<div class="notice notice-warning is-dismissible aialttext-incomplete-notice">';
        echo '<p><strong>' . esc_html__('Img Alt Gen Pro:', 'aialttext-token') . '</strong> ' . esc_html($status['message']);
        echo ' <a href="' . esc_url($status['action_url']) . '" class="button button-secondary">' . esc_html($status['action_text']) . '</a></p>';
        echo '</div>';
    } elseif ($status['status'] === 'no_tokens') {
        echo '<div class="notice notice-error">';
        echo '<p><strong>' . esc_html__('AI Alt Text Generator:', 'aialttext-token') . '</strong> ' . esc_html($status['message']);
        echo ' <a href="' . esc_url($status['action_url']) . '" class="button button-primary">' . esc_html($status['action_text']) . '</a></p>';
        echo '</div>';
    } elseif ($status['status'] === 'low_tokens') {
        echo '<div class="notice notice-warning">';
        echo '<p><strong>' . esc_html__('AI Alt Text Generator:', 'aialttext-token') . '</strong> ' . esc_html($status['message']);
        echo ' <a href="' . esc_url($status['action_url']) . '" class="button button-secondary">' . esc_html($status['action_text']) . '</a></p>';
        echo '</div>';
    }
}

// Add settings link to plugins page
add_filter('plugin_action_links_' . AIALTTEXT_TOKEN_BASENAME, 'aialttext_token_plugin_action_links');

function aialttext_token_plugin_action_links($links) {
    $status = aialttext_token_get_config_status();
    
    if ($status['status'] === 'incomplete') {
        $setup_link = '<a href="' . admin_url('admin.php?page=aialttext-token') . '" style="color: #d63638; font-weight: bold;">' . __('Setup Required', 'aialttext-token') . '</a>';
        array_unshift($links, $setup_link);
    } elseif ($status['status'] === 'no_tokens') {
        $tokens_link = '<a href="' . admin_url('admin.php?page=aialttext-token') . '" style="color: #d63638; font-weight: bold;">' . __('No Tokens', 'aialttext-token') . '</a>';
        array_unshift($links, $tokens_link);
    } elseif (isset($status['balance'])) {
        $balance_link = '<span style="color: #46b450; font-weight: bold;">' . sprintf(__('%d Tokens', 'aialttext-token'), $status['balance']) . '</span>';
        array_unshift($links, $balance_link);
    }
    
    return $links;
}

/**
 * Supported languages as a label=>label list (keeps storage human-friendly).
 * The model instructions will use the label directly (e.g., "Spanish").
 */
function aialttext_token_get_language_options() {
    return array(
        'Afrikaans'=>'Afrikaans','Albanian'=>'Albanian','Amharic'=>'Amharic','Arabic'=>'Arabic','Armenian'=>'Armenian',
        'Assamese'=>'Assamese','Azerbaijani'=>'Azerbaijani','Belarusian'=>'Belarusian','Bengali'=>'Bengali','Bhojpuri'=>'Bhojpuri',
        'Bosnian'=>'Bosnian','Bulgarian'=>'Bulgarian','Burmese'=>'Burmese','Cantonese'=>'Cantonese','Cebuano'=>'Cebuano',
        'Chichewa'=>'Chichewa','Chittagonian'=>'Chittagonian','Czech'=>'Czech','Danish'=>'Danish','Deccan'=>'Deccan',
        'Dutch'=>'Dutch','English'=>'English','Finnish'=>'Finnish','French'=>'French','Fula'=>'Fula',
        'Gan Chinese'=>'Gan Chinese','Georgian'=>'Georgian','German'=>'German','Greek'=>'Greek','Gujarati'=>'Gujarati',
        'Haitian Creole'=>'Haitian Creole','Hakka'=>'Hakka','Hausa'=>'Hausa','Hebrew'=>'Hebrew','Hindi'=>'Hindi',
        'Hmong'=>'Hmong','Hungarian'=>'Hungarian','Igbo'=>'Igbo','Ilocano'=>'Ilocano','Indonesian'=>'Indonesian',
        'Italian'=>'Italian','Japanese'=>'Japanese','Javanese'=>'Javanese','Kannada'=>'Kannada','Kazakh'=>'Kazakh',
        'Khmer'=>'Khmer','Kinyarwanda'=>'Kinyarwanda','Kirundi'=>'Kirundi','Korean'=>'Korean','Kurdish'=>'Kurdish',
        'Lao'=>'Lao','Madurese'=>'Madurese','Maithili'=>'Maithili','Malagasy'=>'Malagasy','Malay'=>'Malay',
        'Malayalam'=>'Malayalam','Mandarin Chinese'=>'Mandarin Chinese','Marathi'=>'Marathi','Min Nan'=>'Min Nan',
        'Nepali'=>'Nepali','Oriya (Odia)'=>'Oriya (Odia)','Oromo'=>'Oromo','Pashto'=>'Pashto','Persian (Farsi)'=>'Persian (Farsi)',
        'Polish'=>'Polish','Portuguese'=>'Portuguese','Punjabi'=>'Punjabi','Quechua'=>'Quechua','Romanian'=>'Romanian',
        'Russian'=>'Russian','Serbo-Croatian'=>'Serbo-Croatian','Shona'=>'Shona','Sindhi'=>'Sindhi','Sinhala'=>'Sinhala',
        'Slovak'=>'Slovak','Somali'=>'Somali','Southern Min'=>'Southern Min','Spanish'=>'Spanish','Sundanese'=>'Sundanese',
        'Swahili'=>'Swahili','Swedish'=>'Swedish','Sylheti'=>'Sylheti','Tagalog (Filipino)'=>'Tagalog (Filipino)',
        'Tamil'=>'Tamil','Telugu'=>'Telugu','Thai'=>'Thai','Tigrinya'=>'Tigrinya','Turkish'=>'Turkish',
        'Turkmen'=>'Turkmen','Ukrainian'=>'Ukrainian','Urdu'=>'Urdu','Uyghur'=>'Uyghur','Uzbek'=>'Uzbek',
        'Vietnamese'=>'Vietnamese','Wu Chinese'=>'Wu Chinese','Xhosa'=>'Xhosa','Xiang Chinese'=>'Xiang Chinese','Yoruba'=>'Yoruba',
        'Zhuang'=>'Zhuang','Zulu'=>'Zulu',
    );
}

/**
 * Guess a sensible default output language from the site locale.
 * Falls back to English.
 */
function aialttext_token_guess_default_language() {
    $locale = function_exists('determine_locale') ? determine_locale() : get_locale();
    $map = array(
        'af'=>'Afrikaans','sq'=>'Albanian','am'=>'Amharic','ar'=>'Arabic','hy'=>'Armenian','as'=>'Assamese','az'=>'Azerbaijani',
        'be'=>'Belarusian','bn'=>'Bengali','bs'=>'Bosnian','bg'=>'Bulgarian','my'=>'Burmese','yue'=>'Cantonese','ceb'=>'Cebuano',
        'ny'=>'Chichewa','cs'=>'Czech','da'=>'Danish','nl'=>'Dutch','en'=>'English','fi'=>'Finnish','fr'=>'French','ff'=>'Fula',
        'ka'=>'Georgian','de'=>'German','el'=>'Greek','gu'=>'Gujarati','ht'=>'Haitian Creole','ha'=>'Hausa','he'=>'Hebrew',
        'hi'=>'Hindi','hmn'=>'Hmong','hu'=>'Hungarian','ig'=>'Igbo','ilo'=>'Ilocano','id'=>'Indonesian','it'=>'Italian',
        'ja'=>'Japanese','jv'=>'Javanese','kn'=>'Kannada','kk'=>'Kazakh','km'=>'Khmer','rw'=>'Kinyarwanda','rn'=>'Kirundi',
        'ko'=>'Korean','ku'=>'Kurdish','lo'=>'Lao','mr'=>'Marathi','ml'=>'Malayalam','ms'=>'Malay','ne'=>'Nepali','or'=>'Oriya (Odia)',
        'om'=>'Oromo','ps'=>'Pashto','fa'=>'Persian (Farsi)','pl'=>'Polish','pt'=>'Portuguese','pa'=>'Punjabi','ro'=>'Romanian',
        'ru'=>'Russian','sd'=>'Sindhi','si'=>'Sinhala','sk'=>'Slovak','so'=>'Somali','es'=>'Spanish','su'=>'Sundanese',
        'sw'=>'Swahili','sv'=>'Swedish','tl'=>'Tagalog (Filipino)','ta'=>'Tamil','te'=>'Telugu','th'=>'Thai','ti'=>'Tigrinya',
        'tr'=>'Turkish','tk'=>'Turkmen','uk'=>'Ukrainian','ur'=>'Urdu','ug'=>'Uyghur','uz'=>'Uzbek','vi'=>'Vietnamese',
        'xh'=>'Xhosa','yo'=>'Yoruba','za'=>'Zhuang','zu'=>'Zulu',
        // Chinese variants → default to Mandarin Chinese for UI locale zh*
        'zh'=>'Mandarin Chinese','zh_CN'=>'Mandarin Chinese','zh_TW'=>'Mandarin Chinese'
    );
    $key = strtolower((string)$locale);
    $key = strtok($key, '_'); // use language part first
    return isset($map[$key]) ? $map[$key] : 'English';
}