<?php
if (!defined('ABSPATH')) exit;

/**
 * Minimal queue using postmeta flags + cron for scalability without new tables.
 * - Enqueue by marking attachments with meta _aialttext_token_pending = 1
 * - Worker pulls up to batch size, generates, writes alt text, clears the flag
 */

add_filter('cron_schedules', function($s){
    // Every 2 minutes is a good default; change if needed
    $s['aialttext_every_two_minutes'] = array('interval' => 120, 'display' => 'Every 2 minutes (AIALTTEXT)');
    return $s;
});

add_action('aialttext_token_process_queue', 'aialttext_token_process_queue');
function aialttext_token_process_queue() {
    // Concurrency lock
    if ( get_transient('aialttext_token_queue_lock') ) return;
    set_transient('aialttext_token_queue_lock', 1, 5 * MINUTE_IN_SECONDS);

    $batch_size = max(1, intval(get_option('aialttext_token_batch_size', 25)));
    $args = array(
        'post_type'      => 'attachment',
        'post_status'    => 'inherit',
        'posts_per_page' => $batch_size,
        'meta_query'     => array(
            array(
                'key'   => '_aialttext_token_pending',
                'value' => '1',
            ),
        ),
        'fields' => 'ids',
    );

    $q = new WP_Query($args);
    if ($q->have_posts()) {
        require_once AIALTTEXT_TOKEN_PATH . 'includes/api-client.php';
        require_once AIALTTEXT_TOKEN_PATH . 'includes/image-processor.php';

        foreach ($q->posts as $attachment_id) {
            $context = '';
            if ( function_exists('aialttext_token_build_context_for_image') ) {
                $context = aialttext_token_build_context_for_image((int)$attachment_id);
            }
            $img_src  = wp_get_attachment_image_url($attachment_id, 'full');
            $title    = get_the_title($attachment_id);

            // Fallback if no URL
            if (!$img_src) {
                delete_post_meta($attachment_id, '_aialttext_token_pending');
                continue;
            }

            $client  = new AiAltText_Token_API_Client();
            $payload_alt = $client->generate($img_src, $title, $context);

            if (is_array($payload_alt) && !empty($payload_alt['success']) && !empty($payload_alt['alt'])) {
                // Write alt text (core field)
                update_post_meta(
                    $attachment_id,
                    '_wp_attachment_image_alt',
                    wp_strip_all_tags($payload_alt['alt'])
                );

                // Invalidate the cached set of published attachments missing alt text
                delete_transient('aialttext_token_pub_ids');

                delete_post_meta($attachment_id, '_aialttext_token_pending');
            } else {
                // Leave pending; worker will retry next tick (or you can set a failure flag)
                // Optional: mark attempt count
                $attempts = (int) get_post_meta($attachment_id, '_aialttext_token_attempts', true);
                update_post_meta($attachment_id, '_aialttext_token_attempts', $attempts + 1);
            }
        }
    }

    delete_transient('aialttext_token_queue_lock');
}

/**
 * Public helper: enqueue a specific attachment id for background processing.
 */
function aialttext_token_enqueue_attachment($attachment_id) {
    $attachment_id = (int)$attachment_id;
    if ($attachment_id > 0) {
        update_post_meta($attachment_id, '_aialttext_token_pending', '1');
        // Ensure cron is scheduled
        if (!wp_next_scheduled('aialttext_token_process_queue')) {
            wp_schedule_event(time() + 60, 'aialttext_every_two_minutes', 'aialttext_token_process_queue');
        }
        return true;
    }
    return false;
}

/**
 * Optional: bulk enqueue for attachments missing alt text.
 * Safe for very large libraries because actual work is in the cron worker.
 */
function aialttext_token_enqueue_missing_alts($limit = 500) {
    $limit = max(1, (int)$limit);
    $q = new WP_Query(array(
        'post_type'      => 'attachment',
        'post_status'    => 'inherit',
        'posts_per_page' => $limit,
        'meta_query'     => array(
            array(
                'key'     => '_wp_attachment_image_alt',
                'compare' => 'NOT EXISTS',
            ),
        ),
        'fields' => 'ids',
    ));
    foreach ($q->posts as $attachment_id) {
        aialttext_token_enqueue_attachment($attachment_id);
    }
}