<?php
if (!defined('ABSPATH')) exit;

/**
 * Treat checkbox-style options ('1', 1, true, 'true', 'on', 'yes') as enabled.
 */
if (!function_exists('aialttext_token_is_enabled')) {
    function aialttext_token_is_enabled(string $option_name, $default = 0): bool {
        $v = get_option($option_name, $default);
        return in_array($v, array(1, '1', true, 'true', 'on', 'yes'), true);
    }
}

// Build contextual hints (WooCommerce, Yoast, RankMath)
function aialttext_token_build_context_for_attachment( $attachment_id ) {
    $parts  = array();
    $parent = wp_get_post_parent_id( $attachment_id );

    if ( $parent && 'product' === get_post_type( $parent ) && function_exists( 'wc_get_product' ) ) {
        $product = wc_get_product( $parent );
        if ( $product ) {
            $p_title = wp_strip_all_tags( $product->get_name() );
            if ( $p_title !== '' ) {
                $parts[] = 'PRODUCT_TITLE: ' . $p_title . '.';
            }
            $desc_raw = $product->get_short_description();
            if ( ! $desc_raw ) { $desc_raw = $product->get_description(); }
            $desc = wp_strip_all_tags( $desc_raw );
            if ( $desc !== '' ) {
                $parts[] = 'PRODUCT_DESC: ' . wp_html_excerpt( $desc, 300, '' ) . '.';
            }
        }
    } elseif ( $parent && defined( 'WPSEO_VERSION' ) ) {
        // SEO keyphrase when available (normalized)
        $kw = (string) get_post_meta( $parent, '_yoast_wpseo_focuskw', true );
        if ( $kw !== '' ) {
            $parts[] = 'SEO_KEYPHRASE: ' . rtrim( wp_strip_all_tags( $kw ), " .\t\n\r\0\x0B" );
        }
    }

    // Optional: include a descriptive filename in the attachment context as well.
    // Uses the same setting and helper as the image-processor path so behaviour is consistent.
    if ( function_exists( 'wp_get_attachment_url' ) ) {
        $use_filename = (int) get_option( 'aialttext_token_use_filename_context', 1 ) === 1;

        if ( $use_filename ) {
            $url = wp_get_attachment_url( $attachment_id );

            if ( $url ) {
                $path = parse_url( $url, PHP_URL_PATH );

                if ( $path ) {
                    $basename = basename( $path );

                    if (
                        function_exists( 'aialttext_token_is_informative_filename' )
                        && aialttext_token_is_informative_filename( $basename )
                    ) {
                        $parts[] = 'Filename: ' . $basename;
                    }
                }
            }
        }
    }

    return implode( "\n", array_filter( $parts ) );
}




/**
 * Detect the current parent post ID when an image is uploaded from edit screens.
 * Works in Classic Editor, Gutenberg (REST), and WooCommerce product editor.
 *
 * @param int $attachment_id Optional. Attachment ID as a fallback to read post_parent.
 * @return int Post ID or 0 when unknown.
 */
function aialttext_token_detect_current_post_id( $attachment_id = 0 ) {
    $pid = 0;

        // 0) Explicit context hints from our JS and core upload flows.
    // These are cheap to check and help in odd upload contexts.
    if ( ! $pid && isset( $_REQUEST['context_post_id'] ) ) {
        $pid = absint( $_REQUEST['context_post_id'] );
    }

    // Some flows still send post_ID instead of post_id.
    if ( ! $pid && isset( $_REQUEST['post_ID'] ) ) {
        $pid = absint( $_REQUEST['post_ID'] );
    }

    // 1) Standard upload params (media modal / REST)
    if ( isset( $_REQUEST['post_id'] ) ) {
        $pid = absint( $_REQUEST['post_id'] );
    }
    if ( ! $pid && isset( $_REQUEST['post'] ) ) {
        $pid = absint( $_REQUEST['post'] );
    }

    // 2) Classic editor screen (edit.php / post.php)
    if ( ! $pid && is_admin() && function_exists( 'get_current_screen' ) ) {
        $screen = get_current_screen();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( $screen && ! empty( $_GET['post'] ) ) {
            $pid = absint( $_GET['post'] );
        }
    }

    // 3) Referer patterns (Classic, Gutenberg, media modal, WooCommerce wc-admin)
    if ( ! $pid && ! empty( $_SERVER['HTTP_REFERER'] ) ) {
        $ref = (string) $_SERVER['HTTP_REFERER'];

        // Classic/Gutenberg edit (?post=123) and REST parent hint
        if ( preg_match( '/[?&]post=([0-9]+)/', $ref, $m ) ) {
            $pid = (int) $m[1];

        } elseif ( preg_match( '#/wp-json/wp/v2/media.*[?&]parent=([0-9]+)#', $ref, $m ) ) {
            $pid = (int) $m[1];

        } elseif ( preg_match(
            '#wp-admin/(?:post|post-new|post\.php|post-new\.php).*?[?&]post=([0-9]+)#',
            $ref,
            $m
        ) ) {
            $pid = (int) $m[1];

        // WooCommerce block product editor (wc-admin) – multiple URL shapes
        } elseif ( preg_match(
            '#[?&]page=wc-admin[^#]*[?&]path=(?:%2F|/)?product(?:%2F|/)([0-9]+)(?:%2F|/)?(?:edit)?#',
            $ref,
            $m
        ) ) {
            // e.g. /wp-admin/admin.php?page=wc-admin&path=/product/360 or /product/360/edit
            $pid = (int) $m[1];

        } elseif ( preg_match(
            '#[?&]path=(?:%2F|/)?product(?:%2F|/)([0-9]+)(?:%2F|/)?(?:edit)?#',
            $ref,
            $m
        ) ) {
            // Fallback: product editor path without explicit page param
            $pid = (int) $m[1];

        } elseif ( preg_match( '#/product/([0-9]+)(?:/edit)?#', $ref, $m ) ) {
            // Very defensive fallback
            $pid = (int) $m[1];
        }
    }

    // 4) Session/transient fallback (populated by admin_init tracking in this file)
    if ( ! $pid ) {
        if ( ! session_id() ) {
            @session_start();
        }

        if (
            ! empty( $_SESSION['aialttext_last_edited_post'] )
            && ! empty( $_SESSION['aialttext_edit_timestamp'] )
        ) {
            // Only trust session data for the last hour
            if ( time() - (int) $_SESSION['aialttext_edit_timestamp'] < HOUR_IN_SECONDS ) {
                $pid = absint( $_SESSION['aialttext_last_edited_post'] );
            }
        }

        if ( ! $pid && function_exists( 'get_current_user_id' ) ) {
            $t = get_transient(
                'aialttext_current_editing_post_' . get_current_user_id()
            );
            if ( $t ) {
                $pid = absint( $t );
            }
        }
    }

    // 5) Final fallback: attachment parent
    if ( ! $pid && $attachment_id ) {
        $pid = (int) wp_get_post_parent_id( $attachment_id );
    }

    if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
        error_log( '[AI Alt Text] detect_current_post_id => ' . ( $pid ?: 0 ) );
    }

    return $pid > 0 ? $pid : 0;
}



/**
 * Build high-quality context for the AI prompt, pulling from SEO plugins and WooCommerce
 * depending on where the image was uploaded.
 *
 * For scalability and consistency, this now delegates to the richer
 * post-based helper whenever we can detect a parent post/product.
 */
function aialttext_token_get_image_context( $image_id, $post_id = 0 ) {
    // Detect parent post if not provided (Classic, Gutenberg, wc-admin)
    if ( ! $post_id ) {
        $post_id = aialttext_token_detect_current_post_id( $image_id );
    }

    // If we have a parent post, reuse the consolidated context builder.
    // This already merges WooCommerce + SEO context and respects the
    // plugin's integration toggles. We then defensively ensure that
    // PRODUCT_* markers exist for WooCommerce products so the account
    // plugin can always see product + brand context.
    if ( $post_id && function_exists( 'aialttext_token_build_context_for_post' ) ) {
        $context = aialttext_token_build_context_for_post( (int) $post_id );

        if ( $context !== '' && function_exists( 'get_post_type' ) ) {
            $post_type = get_post_type( $post_id );

            // Treat both simple products and variations as products
            if ( in_array( $post_type, array( 'product', 'product_variation' ), true ) ) {
                // Do we already have PRODUCT_* markers in the context?
                $has_product_markers =
                    ( false !== stripos( $context, 'PRODUCT_TITLE:' ) ) ||
                    ( false !== stripos( $context, 'PRODUCT_DESC:' ) )  ||
                    ( false !== stripos( $context, 'PRODUCT_BRAND:' ) );

                // Honor the WooCommerce integration toggle
                if ( function_exists( 'aialttext_token_is_enabled' ) ) {
                    $woo_enabled = aialttext_token_is_enabled( 'aialttext_token_woocommerce_integration', 1 );
                } else {
                    $woo_enabled = ( (int) get_option( 'aialttext_token_woocommerce_integration', 1 ) === 1 );
                }

                // If Woo integration is enabled but no PRODUCT_* markers made it
                // through (which is what the admin plugin relies on), add a
                // defensive WooCommerce context block here.
                if ( $woo_enabled && ! $has_product_markers && function_exists( 'aialttext_token_get_woocommerce_context' ) ) {
                    $woo_ctx = aialttext_token_get_woocommerce_context( $post_id );
                    if ( $woo_ctx !== '' ) {
                        $context = trim( $context . "\n" . $woo_ctx );

                        if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
                            error_log(
                                sprintf(
                                    '[AI Alt Text] Fallback WooCommerce context appended for product %d',
                                    $post_id
                                )
                            );
                        }
                    }
                }
            }
        }

        // Append descriptive filename into the context when enabled.
        // This mirrors the behaviour in aialttext_token_build_context_for_image()
        // so all generation paths (bulk, single, auto-upload) see the same hints.
        if ( function_exists( 'wp_get_attachment_url' ) ) {
            if ( function_exists( 'aialttext_token_is_enabled' ) ) {
                $use_filename = aialttext_token_is_enabled( 'aialttext_token_use_filename_context', 1 );
            } else {
                $use_filename = (int) get_option( 'aialttext_token_use_filename_context', 1 ) === 1;
            }


            if ( $use_filename ) {
                $url = wp_get_attachment_url( $image_id );

                if ( $url ) {
                    $path = parse_url( $url, PHP_URL_PATH );

                    if ( $path ) {
                        $basename = basename( $path );

                        if (
                            function_exists( 'aialttext_token_is_informative_filename' )
                            && aialttext_token_is_informative_filename( $basename )
                        ) {
                            // Avoid duplicating the filename if something upstream already added it.
                            if ( strpos( $context, $basename ) === false ) {
                                $context = trim( $context . "\nFilename: " . $basename );
                            }
                        }
                    }
                }
            }
        }

        return $context;
    }


    // Legacy fallbacks for unattached images – keep behaviour compatible
    // while still giving the model something useful.
    if ( function_exists( 'aialttext_token_build_context_for_attachment' ) ) {
        return aialttext_token_build_context_for_attachment( (int) $image_id );
    }

    if ( function_exists( 'aialttext_token_build_context_for_image' ) ) {
        // Defined in image-processor.php – lightweight context without Woo/SEO hard deps.
        return aialttext_token_build_context_for_image( (int) $image_id );
    }

    return '';
}

/**
 * Extract SEO intent for the current post from major SEO plugins.
 * Rule: If a Yoast/Rank Math/AIOSEO/SEOPress keyphrase exists, include it.
 * Otherwise, include the SEO title + meta description as context.
 */
function aialttext_token_get_seo_context( $post_id ) {
    if ( ! $post_id ) {
        return '';
    }

    $parts = array();

    // Yoast SEO
    // Initialize variables
$focus = '';
$seo_title = '';
$seo_desc = '';

// Yoast SEO
if ( defined( 'WPSEO_VERSION' ) ) {
    // Prefer Yoast’s own API, then fall back to raw post meta.
    // This keeps behaviour compatible across Yoast versions and setups.

    if ( class_exists( 'WPSEO_Meta' ) && is_callable( array( 'WPSEO_Meta', 'get_value' ) ) ) {
        // Primary internal field (still used by Yoast for the Focus keyphrase)
        $focus = (string) WPSEO_Meta::get_value( 'focuskw', $post_id );
    }

    // Database meta fallbacks – handle both underscored and non-underscored keys
    // so we still work if something has written directly to meta.
    if ( $focus === '' ) {
        $focus = (string) get_post_meta( $post_id, '_yoast_wpseo_focuskw', true );
    }
    if ( $focus === '' ) {
        $focus = (string) get_post_meta( $post_id, 'yoast_wpseo_focuskw', true );
    }

    // Title + meta description are already standard and safe to keep as-is
    $seo_title = (string) get_post_meta( $post_id, '_yoast_wpseo_title', true );
    $seo_desc  = (string) get_post_meta( $post_id, '_yoast_wpseo_metadesc', true );
}


// RankMath SEO
elseif ( class_exists( 'RankMath' ) || defined( 'RANK_MATH_VERSION' ) ) {
    $focus = (string) get_post_meta( $post_id, 'rank_math_focus_keyword', true );
    $seo_title = (string) get_post_meta( $post_id, 'rank_math_title', true );
    $seo_desc = (string) get_post_meta( $post_id, 'rank_math_description', true );
}

// All in One SEO (AIOSEO)
elseif ( function_exists( 'aioseo' ) && aioseo()->core->db->start() ) {
    // AIOSEO stores data differently - in a custom table
    global $wpdb;
    $aioseo_data = $wpdb->get_row( $wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}aioseo_posts WHERE post_id = %d",
        $post_id
    ) );
    
    if ( $aioseo_data ) {
        // AIOSEO stores keyphrases as JSON
        if ( !empty( $aioseo_data->keyphrases ) ) {
            $keyphrases = json_decode( $aioseo_data->keyphrases, true );
            if ( is_array( $keyphrases ) && !empty( $keyphrases['focus']['keyphrase'] ) ) {
                $focus = (string) $keyphrases['focus']['keyphrase'];
            }
        }
        $seo_title = !empty( $aioseo_data->title ) ? (string) $aioseo_data->title : '';
        $seo_desc = !empty( $aioseo_data->description ) ? (string) $aioseo_data->description : '';
    }
}

// SEOPress
elseif ( function_exists( 'seopress_get_service' ) ) {
    $focus = (string) get_post_meta( $post_id, '_seopress_analysis_target_kw', true );
    $seo_title = (string) get_post_meta( $post_id, '_seopress_titles_title', true );
    $seo_desc = (string) get_post_meta( $post_id, '_seopress_titles_desc', true );
}

// Add the SEO data to parts
if ( $focus !== '' ) {
    $parts[] = 'SEO_KEYPHRASE: ' . wp_strip_all_tags( $focus ) . '.';
} else {
    if ( $seo_title !== '' ) {
        $parts[] = 'SEO_TITLE: ' . wp_strip_all_tags( $seo_title ) . '.';
    }
    if ( $seo_desc !== '' ) {
        $parts[] = 'SEO_META: ' . wp_strip_all_tags( $seo_desc ) . '.';
    }

        // Add improved instructions for natural SEO integration
$has_focus = false;
foreach ( $parts as $p ) {
    if (strpos($p, 'SEO_KEYPHRASE:') === 0) { $has_focus = true; break; }
}
if ( $has_focus ) {
    $parts[] = 'INSTRUCTION: Naturally incorporate the SEO_KEYPHRASE into the alt text when it accurately describes the image. Focus on creating descriptive, natural-sounding alt text that includes the keyphrase organically rather than forcing it in verbatim.';
} else {
    $parts[] = 'INSTRUCTION: No SEO_KEYPHRASE found; use SEO_TITLE and SEO_META as contextual guidance for creating natural, descriptive alt text.';
}


    }

    // Fallback to core post data if no SEO meta found
    if ( empty( $parts ) ) {
        $post = get_post( $post_id );
        if ( $post ) {
            $parts[] = 'SEO_TITLE: ' . wp_strip_all_tags( get_the_title( $post_id ) ) . '.';
            $excerpt = $post->post_excerpt ? $post->post_excerpt : wp_trim_words( $post->post_content, 30, '' );
            if ( $excerpt ) {
                $parts[] = 'SEO_META: ' . wp_strip_all_tags( $excerpt ) . '.';
            }
        }
    }



    return trim( implode( ' ', $parts ) );
}


function aialttext_token_get_woocommerce_context( $post_id ) {
    // Only require the WooCommerce helpers we actually use here.
    // On some admin-ajax / REST requests the main WooCommerce class
    // may not be loaded, but wc_get_product() is still available.
    if ( ! $post_id || ! function_exists( 'wc_get_product' ) ) {
        return '';
    }

    $product = wc_get_product( $post_id );
    if ( ! $product ) {
        return '';
    }

    // For variations, also look at the parent product for shared data
    $parent_product = null;
    $parent_id      = 0;

    if ( method_exists( $product, 'is_type' ) && $product->is_type( 'variation' ) ) {
        $parent_id = (int) $product->get_parent_id();
        if ( $parent_id > 0 ) {
            $maybe_parent = wc_get_product( $parent_id );
            if ( $maybe_parent ) {
                $parent_product = $maybe_parent;
            }
        }
    }

    $parts = array();

    // Title (keep as PRODUCT_TITLE which the server already parses)
    // Use the variation name when available (it usually includes attributes),
    // otherwise fall back to the parent product title.
    $p_title = wp_strip_all_tags( $product->get_name() );
    if ( $p_title === '' && $parent_product ) {
        $p_title = wp_strip_all_tags( $parent_product->get_name() );
    }
    if ( $p_title !== '' ) {
        $parts[] = 'PRODUCT_TITLE: ' . $p_title . '.';
    }

    // Brand (explicit) — supports common brand setups
    $brand = '';

    // 1. Check common brand taxonomies on both product and parent
    $brand_taxes = array(
        'product_brand',        // WooCommerce Brands
        'product_brands',       // Some themes/plugins use plural
        'pwb-brand',            // Perfect WooCommerce Brands
        'yith_product_brand',   // YITH WooCommerce Brands Add-on
        'yith_product_brands',
        'brand',                // Generic "brand" taxonomy
        'pa_brand',             // Attribute-based brand
    );

    $ids_to_check = array( (int) $product->get_id() );
    if ( $parent_product ) {
        $ids_to_check[] = (int) $parent_product->get_id();
    }
    $ids_to_check = array_values( array_unique( array_filter( $ids_to_check ) ) );

    foreach ( $brand_taxes as $tax ) {
        if ( ! taxonomy_exists( $tax ) ) {
            continue;
        }
        foreach ( $ids_to_check as $pid ) {
            $terms = wp_get_post_terms( $pid, $tax, array( 'fields' => 'names' ) );
            if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                $brand = $terms[0];
                break 2; // stop both loops
            }
        }
    }

    // 2. Check product / parent attributes if no brand taxonomy found
    if ( $brand === '' ) {
        $attribute_sources = array( $product );
        if ( $parent_product ) {
            $attribute_sources[] = $parent_product;
        }

        foreach ( $attribute_sources as $attr_product ) {
            if ( ! $attr_product || ! method_exists( $attr_product, 'get_attributes' ) ) {
                continue;
            }

            foreach ( $attr_product->get_attributes() as $attribute ) {
                $name  = $attribute->get_name();
                $label = wc_attribute_label( $name );

                // Check both name and label for "brand"
                if ( stripos( $name, 'brand' ) === false && stripos( $label, 'brand' ) === false ) {
                    continue;
                }

                if ( $attribute->is_taxonomy() ) {
                    $terms = wp_get_post_terms( $attr_product->get_id(), $name, array( 'fields' => 'names' ) );
                    if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                        $brand = $terms[0];
                        break 2;
                    }
                } else {
                    $options = (array) $attribute->get_options();
                    if ( ! empty( $options ) ) {
                        $brand = reset( $options );
                        break 2;
                    }
                }
            }
        }
    }

    // 3. Final fallback: heuristically derive brand from the product title itself
    if ( $brand === '' && $p_title !== '' ) {
        // Common brand patterns in titles (e.g., "Nike Running Shoes" or "Adidas - Sport Jacket")
        if ( preg_match( '/^([A-Z][a-zA-Z]+(?:\s+[A-Z][a-zA-Z]+)?)\s+[-–]/u', $p_title, $m ) ) {
            $brand = trim( $m[1] );
        } elseif ( preg_match( '/^([A-Z][a-zA-Z]+(?:\s+[A-Z][a-zA-Z]+)?)\s+[A-Z]/u', $p_title, $m ) ) {
            $candidate = trim( $m[1] );
            if ( strlen( $candidate ) >= 3 && strlen( $candidate ) <= 20 ) {
                $brand = $candidate;
            }
        }
    }

    if ( $brand !== '' ) {
        $parts[] = 'PRODUCT_BRAND: ' . wp_strip_all_tags( $brand ) . '.';
    }

    // Single normalized description field (PRODUCT_DESC) – trimmed for signal/noise
    $desc_raw = $product->get_short_description();
    if ( ! $desc_raw && $parent_product ) {
        $desc_raw = $parent_product->get_short_description();
    }
    if ( ! $desc_raw ) {
        $desc_raw = $product->get_description();
    }
    if ( ! $desc_raw && $parent_product ) {
        $desc_raw = $parent_product->get_description();
    }

    $desc = wp_strip_all_tags( $desc_raw );
    if ( $desc !== '' ) {
        // Keep it short; models do better with concise hints
        $parts[] = 'PRODUCT_DESC: ' . wp_html_excerpt( $desc, 300, '' ) . '.';
    }

    // Attributes: limit to a few, and a few values each → PRODUCT_ATTRS
    $attrs_out    = array();
    $seen_labels  = array();
    $attr_sources = array( $product );
    if ( $parent_product ) {
        $attr_sources[] = $parent_product;
    }

    foreach ( $attr_sources as $attr_product ) {
        if ( ! $attr_product || ! method_exists( $attr_product, 'get_attributes' ) ) {
            continue;
        }

        foreach ( $attr_product->get_attributes() as $attribute ) {
            if ( ! $attribute->get_visible() ) {
                continue;
            }

            $label = wc_attribute_label( $attribute->get_name() );
            $key   = strtolower( trim( $label ) );

            // Avoid duplicate labels across product/parent
            if ( isset( $seen_labels[ $key ] ) ) {
                continue;
            }

            if ( $attribute->is_taxonomy() ) {
                $terms = wp_get_post_terms( $attr_product->get_id(), $attribute->get_name(), array( 'fields' => 'names' ) );
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                    $attrs_out[]          = $label . ': ' . implode( ', ', array_slice( $terms, 0, 3 ) );
                    $seen_labels[ $key ] = true;
                }
            } else {
                $options = (array) $attribute->get_options();
                if ( ! empty( $options ) ) {
                    $attrs_out[]          = $label . ': ' . implode( ', ', array_slice( $options, 0, 3 ) );
                    $seen_labels[ $key ] = true;
                }
            }

            if ( count( $attrs_out ) >= 4 ) {
                break 2; // cap total attributes overall
            }
        }
    }

    if ( ! empty( $attrs_out ) ) {
        $parts[] = 'PRODUCT_ATTRS: ' . implode( '; ', $attrs_out ) . '.';
    }

    // Optional debug logging so you can verify Woo context is being built
    if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
        $debug_summary = implode( ' | ', array_filter( $parts ) );
        if ( strlen( $debug_summary ) > 300 ) {
            $debug_summary = substr( $debug_summary, 0, 300 ) . '...';
        }
        error_log( sprintf(
            '[AI Alt Text] Woo context for post %d (parent %d): %s',
            $post_id,
            $parent_id,
            $debug_summary
        ) );
    }

    // Each PRODUCT_* marker on its own line so the server-side parser
    // can reliably pick up PRODUCT_TITLE / PRODUCT_DESC / PRODUCT_BRAND / PRODUCT_ATTRS.
    return trim( implode( "\n", array_filter( $parts ) ) );
}



if (!function_exists('aialttext_token_build_context_for_post')) {
    function aialttext_token_build_context_for_post(int $post_id): string {
        // Safety check at the very top
        $post = get_post($post_id);
        if (!$post || is_wp_error($post)) {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] aialttext_token_build_context_for_post: Invalid post ID ' . $post_id);
            }
            return ''; // EXPLICIT RETURN
        }
    
        $parts = array();
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] Building context for post ID: ' . $post_id . ', post type: ' . $post->post_type);
        }
    
        // Always include a simple anchor for the AI
        $title = get_the_title($post);
        if (!empty($title)) {
            $parts[] = 'This image appears on: "' . wp_strip_all_tags($title) . '".';
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Added post title to context: ' . $title);
            }
        }
    
        // WooCommerce product context
        $is_product = ($post->post_type === 'product' || $post->post_type === 'product_variation');

        if (function_exists('aialttext_token_is_enabled')) {
            $woo_enabled = aialttext_token_is_enabled('aialttext_token_woocommerce_integration', 1);
        } else {
            $woo_enabled = ((int) get_option('aialttext_token_woocommerce_integration', 1) === 1);
        }

        $woo_has_context = false;

        if ($is_product && $woo_enabled) {
            if (function_exists('aialttext_token_get_woocommerce_context')) {
                $woo_ctx = aialttext_token_get_woocommerce_context($post_id);
                if (!empty($woo_ctx)) {
                    $parts[] = $woo_ctx;
                    $woo_has_context = true;
                }
            }

            if (!$woo_has_context && function_exists('wc_get_product')) {
                $product = wc_get_product($post_id);
                if ($product) {
                    $p_title = wp_strip_all_tags($product->get_name());
                    if ($p_title !== '') {
                        $parts[] = 'PRODUCT_TITLE: ' . $p_title . '.';
                    }

                    $desc = $product->get_short_description();
                    if (empty($desc)) {
                        $desc = $product->get_description();
                    }

                    if (!empty($desc)) {
                        $parts[] = 'PRODUCT_DESC: ' . wp_html_excerpt(wp_strip_all_tags($desc), 300, '') . '.';
                        $woo_has_context = true;
                    }
                }
            }
        }
    
        // SEO context
        if (function_exists('aialttext_token_is_enabled')) {
            $seo_enabled = aialttext_token_is_enabled('aialttext_token_seo_integration', 1);
        } else {
            $seo_enabled = ((int) get_option('aialttext_token_seo_integration', 1) === 1);
        }
        
        if ($seo_enabled && (!$is_product || !$woo_has_context)) {
            if (function_exists('aialttext_token_get_seo_context')) {
                $seo_ctx = aialttext_token_get_seo_context($post_id);
                if (!empty($seo_ctx)) {
                    $parts[] = $seo_ctx;
                }
            }
        }
        
        // ALWAYS return a string - this is OUTSIDE all if blocks
        return trim(implode("\n", array_filter($parts)));
    }

// Media Library column indicator - REMOVED TO PREVENT DUPLICATES
// The main plugin file handles this functionality
// Metabox removed - auto-generation on upload now handles SEO and WooCommerce context automatically

/* ---------------------------------------
 * Enhanced tracking for better context detection  
 * --------------------------------------- */
add_action( 'admin_init', function() {
    // Classic/Gutenberg post editor – explicitly opened edit screen.
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if ( is_admin() && isset( $_GET['action'], $_GET['post'] ) && $_GET['action'] === 'edit' ) {
        $post_id = (int) $_GET['post'];
        if ( $post_id > 0 && function_exists( 'get_current_user_id' ) ) {
            set_transient(
                'aialttext_current_editing_post_' . get_current_user_id(),
                $post_id,
                HOUR_IN_SECONDS
            );
            if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
                error_log( '[AI Alt Text] TRACKING (classic/gutenberg): post ' . $post_id );
            }
        }
    }

    // Keep transient fresh whenever a screen with a post ID is loaded.
    if ( is_admin() && function_exists( 'get_current_screen' ) && function_exists( 'get_current_user_id' ) ) {
        $screen = get_current_screen();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( $screen && isset( $_GET['post'] ) ) {
            $post_id = (int) $_GET['post'];
            if ( $post_id > 0 ) {
                set_transient(
                    'aialttext_current_editing_post_' . get_current_user_id(),
                    $post_id,
                    HOUR_IN_SECONDS
                );
                if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
                    error_log(
                        '[AI Alt Text] TRACKING (screen): post ' .
                        $post_id .
                        ' on ' .
                        $screen->id
                    );
                }
            }
        }
    }

    // WooCommerce block product editor (wc-admin) – parse product ID from ?path=/product/123
    // Example URL: /wp-admin/admin.php?page=wc-admin&path=/product/360 or /product/360/edit
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (
        is_admin()
        && isset( $_GET['page'], $_GET['path'] )
        && $_GET['page'] === 'wc-admin'
        && function_exists( 'get_current_user_id' )
    ) {
        $decoded = rawurldecode( (string) $_GET['path'] ); // e.g. /product/360 or /product/360/edit
        if ( preg_match( '#/product/([0-9]+)#', $decoded, $m ) ) {
            $post_id = (int) $m[1];
            if ( $post_id > 0 ) {
                set_transient(
                    'aialttext_current_editing_post_' . get_current_user_id(),
                    $post_id,
                    HOUR_IN_SECONDS
                );
                if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
                    error_log(
                        '[AI Alt Text] TRACKING (wc-admin): product ' .
                        $post_id .
                        ' from path=' .
                        $decoded
                    );
                }
            }
        }
    }
} );


/* ---------------------------------------
 * Enhanced JavaScript for admin pages to pass context
 * --------------------------------------- */
add_action('admin_enqueue_scripts', function($hook_suffix) {
    $post_id = 0;

    // Classic/Gutenberg
    if (in_array($hook_suffix, array('post.php','post-new.php'), true) && isset($_GET['post'])) {
        $post_id = (int) $_GET['post'];
    }
    // WooCommerce block product editor (wc-admin)
    elseif (
        ( $hook_suffix === 'woocommerce_page_wc-admin' || $hook_suffix === 'toplevel_page_wc-admin' )
        && isset($_GET['path'])
    ) {
        // wc-admin product editor, e.g. /wp-admin/admin.php?page=wc-admin&path=/product/360 or /product/360/edit
        $decoded = rawurldecode((string) $_GET['path']);
        if (preg_match('#/product/([0-9]+)#', $decoded, $m)) {
            $post_id = (int) $m[1];
        }
    }
    if ($post_id > 0) {
        if (function_exists('wp_enqueue_media')) {
            wp_enqueue_media();
        }

        // Primary injection via media-views when available
        wp_add_inline_script('media-views', "
            window.aialttext_current_post_id = {$post_id};
            (function(){
                if (typeof wp !== 'undefined' && wp) {
                    if (wp.media && wp.media.events) {
                        wp.media.events.on('open', function(){ window.aialttext_current_post_id = {$post_id}; });
                    }
                    if (wp.hooks && typeof wp.hooks.addFilter === 'function') {
                        wp.hooks.addFilter(
                            'media.upload.params',
                            'aialttext-token/add-post-context',
                            function (params) {
                                params = params || {};
                                var id = {$post_id};

                                // Ensure WordPress still sees the parent for attachment.
                                if (!params.post_id) {
                                    params.post_id = id;
                                }

                                // Our detection helpers can read both post and context_post_id.
                                params.post            = id;
                                params.context_post_id = id;

                                return params;
                            }
                        );
                    
            })();
        ");

        // Footer fallback for wc-admin when media-views isn’t present
        add_action('admin_print_footer_scripts', function() use ($post_id) {
            echo '<script>
                window.aialttext_current_post_id = '.intval($post_id).';
                (function(){
                    if (typeof wp !== "undefined" && wp && wp.hooks && typeof wp.hooks.addFilter === "function") {
                        wp.hooks.addFilter("media.upload.params","aialttext-token/add-post-context-fallback", function(params){
                            params = params || {};
                            params.post = '.intval($post_id).';
                            params.context_post_id = '.intval($post_id).';
                            return params;
                        });
                    }
                })();
            </script>';
        }, 99);
    }
});
}



/* ---------------------------------------
 * Additional Gutenberg/Block Editor Support
 * --------------------------------------- */
add_action('enqueue_block_editor_assets', function() {
    // Only if auto-generation is enabled
    if (!get_option('aialttext_token_auto_generate', 0)) {
        return;
    }
    
    global $post;
    if ($post && $post->ID) {
        wp_add_inline_script('wp-editor', "
            // Store current post ID for Gutenberg media uploads
            window.aialttext_current_post_id = {$post->ID};
            
            // Hook into media upload for Gutenberg
            if (typeof wp !== 'undefined' && wp.media) {
                // Hook into media upload for Gutenberg
                if (typeof wp !== 'undefined' && wp.media && wp.hooks && typeof wp.hooks.addFilter === 'function') {
                    wp.hooks.addFilter(
                        'media.upload.params',
                        'aialttext/add-post-context',
                        function (params) {
                            params = params || {};
                            var id = {$post->ID};
    
                            // Maintain WP's own parent attachment semantics.
                            if (!params.post_id) {
                                params.post_id = id;
                            }
    
                            // Leave the older `post` hint for our detection helpers.
                            params.post = id;
    
                            return params;
                        }
                    );
                }
            }
        ");
    }
});

/* ---------------------------------------
 * Clean up expired session data
 * --------------------------------------- */
add_action('wp_loaded', function () {
    // Only touch sessions in the admin to avoid interfering with front-end/REST.
    if ( ! is_admin() ) {
        return;
    }

    // Never interfere with REST API responses (including media uploads).
    if ( defined('REST_REQUEST') && REST_REQUEST ) {
        return;
    }

    // Start the session safely if needed and headers are still open.
    if ( ! session_id() && ! headers_sent() ) {
        @session_start();
    }

    if ( empty($_SESSION) || ! isset($_SESSION['aialttext_edit_timestamp']) ) {
        return;
    }

    // Clean up old edit session data (older than 1 hour)
    if ( time() - (int) $_SESSION['aialttext_edit_timestamp'] > HOUR_IN_SECONDS ) {
        unset(
            $_SESSION['aialttext_last_edited_post'],
            $_SESSION['aialttext_edit_timestamp']
        );
    }
});


// Return normalized WooCommerce product context (title/brand/desc/attrs) for a post_id
add_action('wp_ajax_aialttext_token_get_product_context', 'aialttext_token_ajax_get_product_context');
function aialttext_token_ajax_get_product_context() {
    if (!current_user_can('upload_files')) {
        wp_send_json_error(array('message' => 'Permission denied'), 403);
    }

    // Nonce is optional here if you prefer — mirror the same nonce used elsewhere:
    $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['nonce'])) : '';
    if ($nonce && !wp_verify_nonce($nonce, 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Invalid nonce'), 400);
    }

    $post_id = isset($_REQUEST['post_id']) ? absint($_REQUEST['post_id']) : 0;
    if (!$post_id) {
        wp_send_json_error(array('message' => 'Missing post_id'), 400);
    }

    $ctx = '';
    if (function_exists('aialttext_token_get_woocommerce_context')) {
        $ctx = aialttext_token_get_woocommerce_context($post_id);
    }
    if ($ctx === '') {
        // Fall back to your generic post context builder if Woo isn’t present or yields nothing
        if (function_exists('aialttext_token_build_context_for_post')) {
            $ctx = aialttext_token_build_context_for_post($post_id);
        }
    }

    wp_send_json_success(array(
        'post_id' => $post_id,
        'context' => (string) $ctx,
    ));
}
