<?php
if (!defined('ABSPATH')) exit;

/**
 * Build lightweight context for an attachment to help the model.
 * Safe on any site: no Woo/SEO hard dependencies.
 */
if (!function_exists('aialttext_token_build_context_for_image')) {
    function aialttext_token_build_context_for_image(int $attachment_id): string {
        $bits = [];

        // Parent post context
        $parent_id = (int) wp_get_post_parent_id($attachment_id);
        if ($parent_id) {
            $parent = get_post($parent_id);
            if ($parent && !is_wp_error($parent)) {
                $title = wp_strip_all_tags($parent->post_title);
                if ($title !== '') {
                    $bits[] = 'Parent title: ' . $title;
                }

                // Common taxonomies if present
                $taxes = array('category','post_tag','product_cat','product_tag');
                foreach ($taxes as $tx) {
                    $terms = get_the_terms($parent_id, $tx);
                    if (!empty($terms) && !is_wp_error($terms)) {
                        $names = array_filter(array_map('wp_strip_all_tags', wp_list_pluck($terms, 'name')));
                        if ($names) {
                            $bits[] = ucfirst(str_replace('_',' ',$tx)) . ': ' . implode(', ', $names);
                        }
                    }
                }
            }
                    // Optional: SEO plugin context based on the parent post (Yoast, RankMath, AIOSEO, SEOPress)
        // This mirrors the behaviour of aialttext_token_build_context_for_post(), so fallback paths
        // still expose SEO_KEYPHRASE / SEO_TITLE / SEO_META to the account plugin.
        if ($parent_id && function_exists('aialttext_token_get_seo_context')) {
            // Respect the SEO integration toggle
            if (function_exists('aialttext_token_is_enabled')) {
                $seo_enabled = aialttext_token_is_enabled('aialttext_token_seo_integration', 1);
            } else {
                $seo_enabled = ((int) get_option('aialttext_token_seo_integration', 1) === 1);
            }

            if ($seo_enabled) {
                $seo_ctx = aialttext_token_get_seo_context($parent_id);
                if (!empty($seo_ctx)) {
                    $bits[] = $seo_ctx;
                }
            }
        }

        }

        // Attachment title & filename
        $title = get_the_title($attachment_id);
        if (!empty($title)) {
            $bits[] = 'Attachment title: ' . wp_strip_all_tags($title);
        }
        $url = wp_get_attachment_url($attachment_id);
        if ($url) {
            $path = parse_url($url, PHP_URL_PATH);

            // Respect the setting (default: enabled)
            // Use the shared helper so legacy truthy values ("1", "on", "yes", etc.) are honoured.
            if ( function_exists( 'aialttext_token_is_enabled' ) ) {
                $use_filename = aialttext_token_is_enabled( 'aialttext_token_use_filename_context', 1 );
            } else {
                $use_filename = (int) get_option( 'aialttext_token_use_filename_context', 1 ) === 1;
            }


            if ($path && $use_filename) {
                $basename = basename($path);

                if (function_exists('aialttext_token_is_informative_filename')
                    && aialttext_token_is_informative_filename($basename)
                ) {
                    $bits[] = 'Filename: ' . $basename;
                }
            }
        }

        // Site hint
        $sitename = get_bloginfo('name');
        if ($sitename) $bits[] = 'Site: ' . wp_strip_all_tags($sitename);

        return implode(' | ', array_filter($bits));
    }
}

if (!function_exists('aialttext_token_is_informative_filename')) {
    /**
     * Decide whether a filename looks semantically useful for alt generation.
     * Examples:
     *  - "albany.jpeg"      => true
     *  - "nike-airmax.svg"  => true
     *  - "image.jpg"        => false
     *  - "IMG_1234.JPG"     => false
     *  - "13204i.png"       => false
     */
    function aialttext_token_is_informative_filename(string $filename): bool {
        // Strip directory and extension
        $basename = pathinfo($filename, PATHINFO_FILENAME);
        $name     = strtolower((string) $basename);

        // Keep letters only for generic checks
        $letters = preg_replace('/[^a-z]/', '', $name);

        // If there are fewer than 3 letters, it's almost certainly noise (e.g. "13204i")
        if ($letters === '' || strlen($letters) < 3) {
            return false;
        }

        // Generic/unhelpful names we want to ignore
        $generic = array(
            'image',
            'img',
            'photo',
            'picture',
            'pic',
            'screenshot',
            'dsc',
            'untitled'
        );

        if (in_array($letters, $generic, true)) {
            return false;
        }

        return true;
    }
}

/**
 * Enhanced context detection function for image uploads
 * Detects the post/page/product context from various sources
 */
function aialttext_token_detect_upload_context($attachment_id) {
    $context_post_id = 0;

    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Starting context detection for attachment ' . $attachment_id);
    }

    // 1) Standard params (we also accept our own 'context_post_id')
    $param_candidates = array('post_id','post_ID','post','parent','uploadedTo','uploaded_to','context_post_id');
    foreach ($param_candidates as $k) {
        if (isset($_REQUEST[$k])) {
            $maybe = (int) $_REQUEST[$k];
            if ($maybe > 0) {
                $context_post_id = $maybe;
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] Found context from $_REQUEST['.$k.']: '.$context_post_id);
                }
                break;
            }
        }
    }

    // 2) Attachment parent if it’s already set
    if ($context_post_id <= 0) {
        $parent = (int) wp_get_post_parent_id($attachment_id);
        if ($parent > 0) {
            $context_post_id = $parent;
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from attachment parent: ' . $context_post_id);
            }
        }
    }

    // 3) Referer patterns — Classic, Gutenberg, and WooCommerce wc-admin
    if ($context_post_id <= 0 && !empty($_SERVER['HTTP_REFERER'])) {
        $referer = (string) $_SERVER['HTTP_REFERER'];

        // Classic/Gutenberg edit (?post=123)
        if (preg_match('/[?&]post=([0-9]+)/', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from referer (?post=): '.$context_post_id);
            }
        }
        // Explicit post.php?post=123
        elseif (preg_match('#post\.php.*[?&]post=([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from referer (post.php): '.$context_post_id);
            }
        }
        // Gutenberg edit for other types
        elseif (preg_match('#wp-admin/(post|page|product)\.php.*[?&]post=([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[2];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from referer (gutenberg): '.$context_post_id);
            }
        }
        // WooCommerce block product editor (wc-admin), encoded path
        elseif (preg_match('#[?&]page=wc-admin[^#]*[?&]path=(?:%2F|/)product%2F([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from wc-admin (encoded path): '.$context_post_id);
            }
        }
        // WooCommerce block product editor (wc-admin), unencoded path
        elseif (preg_match('#[?&]path=(?:%2F|/)product/([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from wc-admin (path): '.$context_post_id);
            }
        }
        // wc-admin hash/other shapes
        elseif (preg_match('#wc-admin.+(?:%2F|/)?product(?:%2F|/)([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from wc-admin (hash): '.$context_post_id);
            }
        }
    }

    // 4) Current screen (admin)
    if ($context_post_id <= 0 && is_admin()) {
        global $post;
        if ($post && is_object($post) && $post->ID) {
            $context_post_id = (int) $post->ID;
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from global $post: ' . $context_post_id);
            }
        }
    }

    // 5) User-scoped transient fallback (no PHP sessions)
if ($context_post_id <= 0 && function_exists('get_current_user_id')) {
    $t = get_transient( 'aialttext_current_editing_post_' . get_current_user_id() );
    if ( $t ) {
        $context_post_id = (int) $t;
        if ( defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG ) {
            error_log('[AI Alt Text] Found context from transient: ' . $context_post_id);
        }
    }
}

    // 6) Transient per user
    if ($context_post_id <= 0 && function_exists('get_current_user_id')) {
        $t = get_transient('aialttext_current_editing_post_' . get_current_user_id());
        if ($t) {
            $context_post_id = (int) $t;
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from transient: ' . $context_post_id);
            }
        }
    }

    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Final context detection result: ' . ($context_post_id ?: 'none'));
    }
    return $context_post_id;
}


/**
 * Core image processor using the token-based API client.
 */
class AiAltText_Token_Image_Processor {
    /**
     * @param int   $attachment_id
     * @param array $args ['context' => string, 'mime' => string, 'post_id' => int]
     * @return array|WP_Error array('alt' => '...') on success
     */

     /**
 * Return the first captured group for regex or empty string.
 */
private static function preg_first($haystack, $pattern) {
    if (preg_match($pattern, (string) $haystack, $m) && !empty($m[2])) {
        return trim($m[2]);
    }
    return '';
}

/**
 * Build a short, high-priority block that *forces* brand/title/details
 * to be considered by the model (without keyword stuffing).
 * Pulls PRODUCT_* markers from existing $context, and fills gaps using
 * aialttext_token_get_woocommerce_context($post_id) when available.
 *
 * @param int    $post_id
 * @param string $existing_context
 * @return string  Enforced block or empty string
 */
private static function build_enforced_product_block($post_id, $existing_context) {
    $existing_context = (string) $existing_context;

    // Parse what we already have
    $title = self::preg_first($existing_context, '/(^|\\n)\\s*PRODUCT_TITLE:\\s*([^\\n]+)/i');
    $brand = self::preg_first($existing_context, '/(^|\\n)\\s*PRODUCT_BRAND:\\s*([^\\n]+)/i');
    $desc  = self::preg_first($existing_context, '/(^|\\n)\\s*PRODUCT_DESC:\\s*([^\\n]+)/i');
    $attrs = self::preg_first($existing_context, '/(^|\\n)\\s*PRODUCT_ATTRS:\\s*([^\\n]+)/i');

    // Fill gaps from Woo helper if present
    if (function_exists('aialttext_token_get_woocommerce_context')) {
        if ($title === '' || $brand === '' || $desc === '' || $attrs === '') {
            $woo_ctx = aialttext_token_get_woocommerce_context((int) $post_id);
            if (is_string($woo_ctx) && $woo_ctx !== '') {
                if ($title === '') $title = self::preg_first($woo_ctx, '/(^|\\n)\\s*PRODUCT_TITLE:\\s*([^\\n]+)/i');
                if ($brand === '') $brand = self::preg_first($woo_ctx, '/(^|\\n)\\s*PRODUCT_BRAND:\\s*([^\\n]+)/i');
                if ($desc  === '') $desc  = self::preg_first($woo_ctx, '/(^|\\n)\\s*PRODUCT_DESC:\\s*([^\\n]+)/i');
                if ($attrs === '') $attrs = self::preg_first($woo_ctx, '/(^|\\n)\\s*PRODUCT_ATTRS:\\s*([^\\n]+)/i');
            }
        }
    }

    // If absolutely nothing product-specific, skip
    if ($title === '' && $brand === '' && $desc === '' && $attrs === '') {
        return '';
    }

    // Compact, explicit instruction + canonical fields
    $lines = array();
    $lines[] = 'IMPORTANT: You are generating ALT TEXT for an e-commerce product image.';
    $lines[] = 'Write ONE natural sentence (≤ 140 chars), specific and truthful.';
    $lines[] = 'You MUST naturally include (if visually plausible): Brand, product name, and key details (e.g., color, material, style, size/model).';
    $lines[] = 'Avoid keyword stuffing and do not invent details.';

    if ($title !== '') $lines[] = 'PRODUCT_TITLE: ' . $title;
    if ($brand !== '') $lines[] = 'PRODUCT_BRAND: ' . $brand;
    if ($desc  !== '') $lines[] = 'PRODUCT_DESC: '  . mb_substr($desc, 0, 500);
    if ($attrs !== '') $lines[] = 'PRODUCT_ATTRS: ' . mb_substr($attrs, 0, 300);

    return implode("\n", $lines) . "\n---";
}

    public static function process_attachment($attachment_id, $args = array()) {
        $attachment_id = (int) $attachment_id;

        // Validate attachment
        $post = get_post($attachment_id);
        if (!$post || $post->post_type !== 'attachment') {
            return new WP_Error('invalid_attachment', 'Invalid attachment.');
        }
        $mime = !empty($args['mime']) ? (string) $args['mime'] : get_post_mime_type($attachment_id);
        if (strpos((string) $mime, 'image/') !== 0) {
            return new WP_Error('not_image', 'Attachment is not an image.');
        }

        // URL & title
        $url = wp_get_attachment_url($attachment_id);
        if (!$url) return new WP_Error('no_url', 'Missing attachment URL');

        $title = get_the_title($attachment_id);
        if (!$title) {
            $path = parse_url($url, PHP_URL_PATH);
            $title = $path ? basename($path) : ('attachment-' . $attachment_id);
        }

        // SVG: try to rasterize locally for richer AI results; otherwise continue.
        // The account service can also handle SVG (Imagick/data-URL or text-only).
        if ($mime === 'image/svg+xml') {
            $maybe_png = aialttext_token_rasterize_svg_to_png_url($url);
            if ($maybe_png) {
                $url  = $maybe_png;          // hand AI a real bitmap
                $mime = 'image/png';
            }
            // If rasterization fails, keep $url as-is; the account service will still produce AI alt text.
        }

                // Enhanced context building with better SEO/WooCommerce integration
                $context   = '';
                $parent_id = !empty($args['post_id']) ? (int) $args['post_id'] : 0;
        
                // 1) If the caller passed a context string (e.g. auto-generate on upload),
                //    trust it as the canonical Woo/SEO-aware context and avoid duplicating
                //    markers here.
                if (isset($args['context']) && is_string($args['context']) && $args['context'] !== '') {
                    $context = $args['context'];

                    // Merge missing Woo markers (TITLE/DESC/BRAND) if we know the product post
if ($parent_id > 0 && function_exists('aialttext_token_get_woocommerce_context')) {
    $need_title = (stripos($context, 'PRODUCT_TITLE:') === false);
    $need_desc  = (stripos($context, 'PRODUCT_DESC:')  === false);
    $need_brand = (stripos($context, 'PRODUCT_BRAND:') === false);

    if ($need_title || $need_desc || $need_brand) {
        $woo_ctx = aialttext_token_get_woocommerce_context($parent_id);
        if (!empty($woo_ctx)) {
            // Only graft the lines we’re missing
            $grafts = array();

            if ($need_title && preg_match('/(^|\n)PRODUCT_TITLE:\s*([^\n]+)/i', $woo_ctx, $m) && !empty($m[2])) {
                $grafts[] = 'PRODUCT_TITLE: ' . trim($m[2]);
            }
            if ($need_desc  && preg_match('/(^|\n)PRODUCT_DESC:\s*([^\n]+)/i',  $woo_ctx, $m) && !empty($m[2])) {
                $grafts[] = 'PRODUCT_DESC: ' . trim($m[2]);
            }
            if ($need_brand && preg_match('/(^|\n)PRODUCT_BRAND:\s*([^\n]+)/i', $woo_ctx, $m) && !empty($m[2])) {
                $grafts[] = 'PRODUCT_BRAND: ' . trim($m[2]);
            }

            if ($grafts) {
                $context = trim($context . "\n" . implode("\n", $grafts));
            }
        }
    }
}

        
                } else {
                    // 2) Resolve the parent post/product ID as robustly as possible
                    if ($parent_id <= 0) {
                        $parent_id = (int) wp_get_post_parent_id($attachment_id);
                    }
                    if ($parent_id <= 0 && function_exists('aialttext_token_detect_current_post_id')) {
                        $parent_id = (int) aialttext_token_detect_current_post_id($attachment_id);
                    }
                    if ($parent_id <= 0 && function_exists('aialttext_token_detect_upload_context')) {
                        $parent_id = (int) aialttext_token_detect_upload_context($attachment_id);
                    }
        
                    // 3) Build a consolidated context. Prefer the higher-level helper
                    //    so WooCommerce + SEO settings are respected centrally.
                    if ($parent_id > 0 && function_exists('aialttext_token_get_image_context')) {
                        $context = aialttext_token_get_image_context($attachment_id, $parent_id);
                    } elseif ($parent_id > 0 && function_exists('aialttext_token_build_context_for_post')) {
                        $context = aialttext_token_build_context_for_post($parent_id);
                    } elseif (function_exists('aialttext_token_build_context_for_image')) {
                        $context = aialttext_token_build_context_for_image($attachment_id);
                    }
                }
        
                // If this is a product, prepend a short "IMPORTANT" block to enforce brand/details.
if ($parent_id > 0 && get_post_type($parent_id) === 'product') {
    $enforced = self::build_enforced_product_block($parent_id, (string) $context);
    if ($enforced !== '') {
        $context = $enforced . "\n" . (string) $context;
    }
}

// Debug logging
if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
    error_log(sprintf(
        '[AI Alt Text] Processing attachment %d, parent %d, final context length: %d chars',
        $attachment_id,
        $parent_id,
        strlen($context)
    ));
    error_log('[AI Alt Text] Final context being sent to API: ' . $context);
}


        // Call token API
        if (!class_exists('AiAltText_Token_API_Client')) {
            return new WP_Error('missing_client', 'API client not available.');
        }
        
        $api = new AiAltText_Token_API_Client();
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] About to call API with: URL=' . $url . ', Title=' . $title . ', Context=' . substr($context, 0, 200) . (strlen($context) > 200 ? '...' : ''));
        }

// Prefer parent post/product title as the "page title" for the AI
$page_title_for_api = $title;
if ($parent_id > 0) {
    $parent_title = get_the_title($parent_id);
    if (!empty($parent_title)) {
        $page_title_for_api = $parent_title;
    }
}

// CRITICAL: Prefer sending bytes so Gemini/Claude can work even if the admin server can't fetch client media
// This is essential for the tier system to work correctly - Gemini REQUIRES inline bytes
$image_b64 = null;
$image_mime = null;

$max_bytes = (int) apply_filters('aialttext_token_max_inline_bytes', 15 * 1024 * 1024); // Increased from 10MB to 15MB

$file_path = get_attached_file($attachment_id);
$bytes = false;

if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
    error_log('[AI Alt Text] Starting to fetch image bytes for attachment ' . $attachment_id);
    error_log('[AI Alt Text] File path: ' . ($file_path ?: 'none'));
    error_log('[AI Alt Text] Max bytes: ' . $max_bytes);
}

// 1) Try local file first
if (is_string($file_path) && file_exists($file_path) && is_readable($file_path)) {
    $size = @filesize($file_path);
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Local file size: ' . ($size !== false ? $size : 'unknown'));
    }
    
    if ($size !== false && $size <= $max_bytes) {
        $bytes = @file_get_contents($file_path);
        if ($bytes !== false) {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Successfully read local file: ' . strlen($bytes) . ' bytes');
            }
        }
    } elseif ($size !== false && $size > $max_bytes && function_exists('wp_get_image_editor')) {
        // File too large - resize it
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] File too large (' . $size . ' bytes), attempting to resize');
        }
        
        $tmp = wp_tempnam($file_path);
        if ($tmp) {
            @copy($file_path, $tmp);
            $ed = wp_get_image_editor($tmp);
            if (!is_wp_error($ed)) {
                $ed->resize(1600, 1600, false);
                $saved = $ed->save($tmp);
                if (!is_wp_error($saved) && !empty($saved['path'])) {
                    $bytes = @file_get_contents($saved['path']);
                    // Prefer saved mime if supplied
                    if (!empty($saved['mime-type'])) {
                        $image_mime = $saved['mime-type'];
                    }
                    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                        error_log('[AI Alt Text] Successfully resized large file to: ' . strlen($bytes) . ' bytes');
                    }
                }
            }
            @unlink($tmp);
        }
    }
}

// 2) CRITICAL fallback: If local file failed, fetch from public URL
// This is essential for Gemini tiers to work
if ($bytes === false || $bytes === null || $bytes === '') {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Local file read failed, attempting to fetch from URL: ' . $url);
    }
    
    $remote = wp_remote_get($url, array(
        'timeout'     => 20, // Increased timeout
        'redirection' => 3,
        'sslverify'   => true,
        'headers'     => array('Accept' => 'image/*'),
    ));
    
    if (!is_wp_error($remote)) {
        $ctype = wp_remote_retrieve_header($remote, 'content-type');
        $body  = wp_remote_retrieve_body($remote);

        if (is_string($body) && $body !== '') {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Successfully fetched from URL: ' . strlen($body) . ' bytes');
            }
            
            // Keep under cap; if too big, try re-encode/resize
            if (strlen($body) > $max_bytes && function_exists('wp_get_image_editor')) {
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] Downloaded image too large, attempting to resize');
                }
                
                $tmp = wp_tempnam($url);
                if ($tmp) {
                    @file_put_contents($tmp, $body);
                    $ed = wp_get_image_editor($tmp);
                    if (!is_wp_error($ed)) {
                        $ed->resize(1600, 1600, false);
                        $saved = $ed->save($tmp);
                        if (!is_wp_error($saved) && !empty($saved['path'])) {
                            $body = @file_get_contents($saved['path']);
                            if (!empty($saved['mime-type'])) {
                                $ctype = $saved['mime-type'];
                            }
                            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                                error_log('[AI Alt Text] Successfully resized downloaded image to: ' . strlen($body) . ' bytes');
                            }
                        }
                    }
                    @unlink($tmp);
                }
            }
            
            if (is_string($body) && strlen($body) <= $max_bytes) {
                $bytes = $body;
                if (!$image_mime) {
                    $image_mime = $ctype ?: get_post_mime_type($attachment_id);
                    if (!$image_mime) {
                        $ft = wp_check_filetype(parse_url($url, PHP_URL_PATH));
                        if (!empty($ft['type'])) $image_mime = $ft['type'];
                    }
                    if (!$image_mime) $image_mime = 'image/jpeg';
                }
            }
        }
    } else {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] Failed to fetch from URL: ' . $remote->get_error_message());
        }
    }
}

// 3) Finalize base64 if we have bytes
if ($bytes !== false && $bytes !== null && $bytes !== '') {
    $image_b64 = base64_encode($bytes);
    if (!$image_mime) {
        $image_mime = get_post_mime_type($attachment_id);
        if (!$image_mime) {
            $fp = $file_path ?: parse_url($url, PHP_URL_PATH);
            $ft = $fp ? wp_check_filetype($fp) : array();
            if (!empty($ft['type'])) $image_mime = $ft['type'];
        }
        if (!$image_mime) $image_mime = 'image/jpeg';
    }
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Successfully prepared image bytes for API: ' . strlen($image_b64) . ' base64 chars, mime: ' . $image_mime);
    }
} else {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] WARNING: Could not get image bytes for attachment ' . $attachment_id . ' - Gemini tiers may be skipped on the server!');
    }
}

// Send bytes when available; fall back to URL-only (will skip Gemini tiers if no bytes)
if ($image_b64 && $image_mime) {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] ✓ Sending API request WITH image bytes');
        error_log('[AI Alt Text]   - Base64 length: ' . strlen($image_b64) . ' chars');
        error_log('[AI Alt Text]   - MIME type: ' . $image_mime);
        error_log('[AI Alt Text]   - URL: ' . $url);
    }
    $result = $api->generate($url, $page_title_for_api, $context, $image_b64, $image_mime);
    
    // Log the API response
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        if (!empty($result['debug_logs'])) {
            error_log('[AI Alt Text] Debug logs from server:');
            foreach ($result['debug_logs'] as $log) {
                error_log('[AI Alt Text]   - ' . $log['message']);
                if (!empty($log['data'])) {
                    error_log('[AI Alt Text]     Data: ' . print_r($log['data'], true));
                }
            }
        }
    }
} else {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] ⚠ WARNING: Sending API request WITHOUT image bytes - Gemini will be skipped!');
        error_log('[AI Alt Text]   - has_b64: ' . (is_string($image_b64) && $image_b64 !== '' ? 'yes' : 'NO'));
        error_log('[AI Alt Text]   - has_mime: ' . (is_string($image_mime) && $image_mime !== '' ? 'yes' : 'NO'));
    }
    $result = $api->generate($url, $page_title_for_api, $context);
}


if (empty($result) || empty($result['success'])) {
    $msg = isset($result['error']['message']) ? (string) $result['error']['message'] : 'API error';
    $http_code = isset($result['error']['code']) ? (int) $result['error']['code'] : 0;

    // Normalize token exhaustion so batch processing can reliably detect it
    if ($http_code === 402 || stripos($msg, 'insufficient') !== false) {
        return new WP_Error('insufficient_tokens', ($msg !== '' ? $msg : 'Insufficient tokens'));
    }
    return new WP_Error('api_error', $msg !== '' ? $msg : 'API error');
}

$alt = isset($result['alt_text']) ? trim((string) $result['alt_text']) : '';
if ($alt === '') {
    return new WP_Error('empty_alt', 'No alt text returned.');
}

// Safety net: collapse obviously duplicated sentences coming back from the API
$parts = preg_split('/[\.!?]+/u', (string) $alt);
if (is_array($parts) && count($parts) > 1) {
    $seen = array();
    $kept = array();

    foreach ($parts as $p) {
        $p = trim($p);
        if ($p === '') {
            continue;
        }

        if (function_exists('mb_strtolower')) {
            $key = mb_strtolower($p, 'UTF-8');
        } else {
            $key = strtolower($p);
        }

        if (!in_array($key, $seen, true)) {
            $seen[] = $key;
            $kept[] = $p;
        }
    }

    if (!empty($kept)) {
        $alt = implode('. ', $kept);
    }
}

        // Persist
        update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);

        // Invalidate the published-attachments cache so re-scans reflect the new alt
                delete_transient('aialttext_token_pub_ids');
        
                return array('alt' => $alt);        
    }
}

/**
 * If possible, rasterize an SVG URL to a temporary PNG in uploads and return its public URL.
 * Returns null on failure (no Imagick, download error, or no delegate).
 */
function aialttext_token_rasterize_svg_to_png_url(string $svg_url): ?string {
    if (!class_exists('Imagick')) return null;

    $res = wp_remote_get($svg_url);
    if (is_wp_error($res)) return null;
    $svg = wp_remote_retrieve_body($res);
    if (!$svg) return null;

    try {
        $im = new Imagick();
        $im->setBackgroundColor(new ImagickPixel('transparent'));
        $im->readImageBlob($svg);           // needs librsvg/inkscape delegate on server
        $im->setImageFormat('png32');
        $blob = $im->getImageBlob();

        $upload = wp_upload_dir();
        $dir    = trailingslashit($upload['basedir']) . 'aialttext-svg-cache/';
        if (!file_exists($dir)) wp_mkdir_p($dir);

        $fname = 'svg-' . md5($svg_url) . '.png';
        $path  = $dir . $fname;
        file_put_contents($path, $blob);

        return trailingslashit($upload['baseurl']) . 'aialttext-svg-cache/' . $fname;
    } catch (Throwable $e) {
        return null;
    }
}

/**
 * Scan next slice of the media library for images lacking alt text.
 * Supports excluding IDs we already attempted in this run.
 */
function aialttext_token_scan_media_library(int $batch_size = 10, int $offset = 0, array $exclude_ids = array()) {
    // IMPORTANT: here $offset is treated as a CURSOR (the last processed attachment ID),
    // not an SQL OFFSET. This avoids skipping items when earlier ones drop out after being processed.
    global $wpdb;

    $exclude_ids = array_filter(array_map('intval', (array) $exclude_ids));
    $exclude_sql = $exclude_ids ? ' AND p.ID NOT IN (' . implode(',', $exclude_ids) . ')' : '';

    $sql = $wpdb->prepare(
        "SELECT p.ID, p.post_title, p.guid, p.post_mime_type
         FROM $wpdb->posts p
         WHERE p.post_type = 'attachment'
           AND p.post_mime_type IN ('image/jpeg','image/png','image/gif','image/webp','image/svg+xml')
           $exclude_sql
           AND p.ID > %d
           AND p.ID NOT IN (
               SELECT post_id FROM $wpdb->postmeta
               WHERE meta_key = '_wp_attachment_image_alt' AND meta_value != ''
           )
         ORDER BY p.ID ASC
         LIMIT %d",
        $offset, $batch_size
    );

    return $wpdb->get_results($sql);
}


/** Simple helper to update alt text directly. */
function aialttext_token_update_alt_text($image_id, $alt_text) {
    $updated = update_post_meta(
        (int) $image_id,
        '_wp_attachment_image_alt',
        (string) $alt_text
    );

    if ($updated) {
        // Keep "published only" scans accurate
        delete_transient('aialttext_token_pub_ids');
    }

    return $updated;
}

// Treat timeouts and temporary upstream issues as transient (worth a retry)
if (!function_exists('aialttext_token_is_transient_error')) {
    function aialttext_token_is_transient_error($message): bool {
        $m = strtolower((string)$message);
        $needles = array(
            'timeout',                 // “Timeout while downloading …”, “request timed out”
            'timed out',
            'operation timed out',
            'connection timed out',
            'cURL error 28',           // WP HTTP API cURL timeout
            '429',                     // rate limits
            '502', '503', '504',       // gateway / service interruptions
            'temporarily unavailable',
            'rate limit'
        );
        foreach ($needles as $n) {
            if (strpos($m, $n) !== false) return true;
        }
        return false;
    }
}

/**
 * Update alt text inside post content for all published posts/pages that reference this attachment.
 * - Adds or fills the alt="" attribute on <img> tags referencing the attachment
 * - Updates the Gutenberg wp:image block JSON's "alt" attribute when present
 * - Does NOT overwrite non-empty alt attributes in existing HTML
 *
 * @param int    $attachment_id
 * @param string $alt_text
 */
function aialttext_token_update_alt_in_referencing_posts( $attachment_id, $alt_text ) {
    global $wpdb;
    $attachment_id = (int) $attachment_id;
    $alt_text = (string) $alt_text;
    if ( $attachment_id <= 0 || $alt_text === '' ) {
        return;
    }

    $guid         = (string) get_the_guid( $attachment_id );
    $url          = (string) wp_get_attachment_url( $attachment_id );
    $file_rel     = (string) get_post_meta( $attachment_id, '_wp_attached_file', true );
    $upload_base  = wp_get_upload_dir();
    $file_path    = $file_rel ? trailingslashit( $upload_base['baseurl'] ) . ltrim( $file_rel, '/' ) : '';

    // Mirrors the logic used in aialttext_token_scan_media_library_published_only()
    $cpts   = (array) apply_filters( 'aialttext_token_published_post_types', array( 'post', 'page' ) );
    $cpts_in = implode("','", array_map('esc_sql', $cpts));

    $post_ids = $wpdb->get_col( $wpdb->prepare(
        "SELECT DISTINCT c.ID
           FROM {$wpdb->posts} p
           JOIN {$wpdb->posts} c ON c.post_status='publish' AND c.post_type IN ('$cpts_in')
          WHERE p.ID = %d
            AND (
                  c.post_content LIKE %s
               OR c.post_content LIKE %s
               OR c.post_content LIKE %s
               OR c.post_content LIKE %s
               OR c.post_content LIKE %s
               OR c.post_content LIKE %s
               OR c.post_content LIKE %s
            )",
        $attachment_id,
        '%wp-image-' . $attachment_id . '%',
        '%attachment_' . $attachment_id . '%',
        '%data-id=\"' . $attachment_id . '\"%',
        '%\"id\":' . $attachment_id . '%',
        $guid !== '' ? '%' . $guid . '%' : '___AIALTTEXT_NO_MATCH___',
        $url  !== '' ? '%' . $url  . '%' : '___AIALTTEXT_NO_MATCH___',
        $file_rel !== '' ? '%' . $file_rel . '%' : '___AIALTTEXT_NO_MATCH___'
    ) );

    if ( empty( $post_ids ) ) {
        return;
    }

    foreach ( $post_ids as $pid ) {
        $pid = (int) $pid;
        $content = (string) get_post_field( 'post_content', $pid );

        $updated_content = aialttext_token_update_img_tags_in_content( $content, $attachment_id, $alt_text, array_filter( array( $url, $guid, $file_path ) ) );
        $updated_content = aialttext_token_update_wp_image_block_json( $updated_content, $attachment_id, $alt_text );

        if ( $updated_content !== $content ) {
            // Save without disturbing other fields
            $post_update = array(
                'ID'           => $pid,
                'post_content' => $updated_content,
            );
            // WordPress expects slashed data in wp_update_post.
            wp_update_post( wp_slash( $post_update ) );
            clean_post_cache( $pid );
        }
    }
}

/**
 * Ensure <img> tags that reference the given attachment have a non-empty alt attribute.
 * Only fills alt when missing or empty; leaves non-empty alts intact.
 */
function aialttext_token_update_img_tags_in_content( $html, $attachment_id, $alt_text, $src_candidates = array() ) {
    $alt_attr = esc_attr( $alt_text );
    $id       = (int) $attachment_id;
    if ( $id <= 0 ) return $html;

    // Identify <img> tags referencing this attachment by class, data-id, id or src variants.
    $patterns = array();
    $patterns[] = 'class=\"[^\"]*wp-image-' . $id . '[^\"]*\"';
    $patterns[] = 'data-id=[\"\\\']' . $id . '[\"\\\']';
    $patterns[] = 'id=[\"\\\']attachment_' . $id . '[\"\\\']';

    foreach ( (array) $src_candidates as $src ) {
        if ( ! $src ) continue;
        $esc = preg_quote( $src, '/' );
        $patterns[] = 'src=[\"\\\'][^\"\\\']*' . $esc . '[^\"\\\']*[\"\\\']';
    }

    $img_rx = '/<img\b([^>]*(' . implode('|', $patterns) . ')[^>]*)>/i';

    $html = preg_replace_callback( $img_rx, function( $m ) use ( $alt_attr ) {
        $tag = $m[0];
        // If alt exists and is non-empty, keep it.
        if ( preg_match( '/\balt\s*=\s*([\"\'])(.*?)\1/i', $tag, $am ) ) {
            $existing = trim( html_entity_decode( $am[2], ENT_QUOTES ) );
            if ( $existing !== '' ) {
                return $tag;
            }
            // Replace empty alt
            $new = preg_replace( '/\balt\s*=\s*([\"\'])(.*?)\1/i', 'alt="'.$alt_attr.'"', $tag, 1 );
            return $new;
        }
        // Insert a fresh alt before closing bracket
        $tag = rtrim( $tag );
        $tag = preg_replace( '/\s*>$/', ' alt="'.$alt_attr.'">', $tag, 1 );
        return $tag;
    }, $html );

    return $html;
}

/**
 * Update Gutenberg wp:image block JSON to include alt for matching attachment IDs.
 * Keeps other attributes intact and never overwrites a non-empty alt.
 */
function aialttext_token_update_wp_image_block_json( $content, $attachment_id, $alt_text ) {
    $id  = (int) $attachment_id;
    $alt = (string) $alt_text;
    if ( $id <= 0 || $alt === '' ) return $content;

    if ( ! preg_match_all( '/<!--\s*wp:image\s+(\{.*?\})\s*-->/', $content, $matches, PREG_OFFSET_CAPTURE ) ) {
        return $content;
    }

    $offset_delta = 0;
    foreach ( $matches[1] as $match ) {
        $json_str = $match[0];
        $json_pos = $match[1] + $offset_delta;

        $attrs = json_decode( $json_str, true );
        if ( ! is_array( $attrs ) || empty( $attrs['id'] ) ) {
            continue;
        }
        if ( (int) $attrs['id'] !== $id ) {
            continue;
        }

        // Respect existing non-empty alt
        if ( empty( $attrs['alt'] ) ) {
            $attrs['alt'] = $alt;
        } else {
            continue;
        }

        $new_json = wp_json_encode( $attrs );
        if ( $new_json && $new_json !== $json_str ) {
            $before = substr( $content, 0, $json_pos );
            $after  = substr( $content, $json_pos + strlen( $json_str ) );
            $content = $before . $new_json . $after;
            $offset_delta += strlen( $new_json ) - strlen( $json_str );
        }
    }

    return $content;
}


/**
 * Legacy processor (kept for compatibility). Uses the newer processor internally.
 * NOTE: The current UI calls aialttext_token_process_images_with_params().
 */
function aialttext_token_process_images($offset, $batch_size) {
    $res = aialttext_token_process_images_with_params((int) $offset, (int) $batch_size, 0, array());
if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
    error_log(sprintf('[AI Alt Text] run summary: processed=%d last_id=%s next_offset=%d insufficient_tokens=%s',
        (int) (isset($res['processed']) ? $res['processed'] : 0),
        !empty($res['processed_ids']) ? max(array_map('intval', (array) $res['processed_ids'])) : 'none',
        (int) (isset($res['offset']) ? $res['offset'] : 0),
        !empty($res['insufficient_tokens']) ? 'yes' : 'no'
    ));
}

    
    // Map to legacy response shape
    return array(
        'success'        => $res['success'],
        'processed'      => $res['processed'],
        'errors'         => $res['errors'],
        'error_messages' => $res['error_messages'],
        'offset'         => $res['offset'],
        'total_images'   => $res['total_images'],
        'progress'       => $res['progress'],
        'completed'      => $res['completed'],
    );
}

/** Count images missing alt text. */
function aialttext_token_get_total_images_without_alt() {
    global $wpdb;
    return (int) $wpdb->get_var(
        "SELECT COUNT(p.ID)
         FROM $wpdb->posts p
         LEFT JOIN $wpdb->postmeta pm
           ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
         WHERE p.post_type = 'attachment'
           AND p.post_mime_type IN ('image/jpeg','image/png','image/gif','image/webp','image/svg+xml')
           AND (pm.meta_value IS NULL OR TRIM(pm.meta_value) = '')"
    );
}

/**
 * Count images missing alt text that are actually used on published posts/pages.
 * Matches either the robust "wp-image-{ID}" marker OR the direct URL (fallback).
 */
function aialttext_token_get_total_images_without_alt_on_published() {
    return (int) count( aialttext_token_get_published_attachment_ids_cached() );
}

/**
 * Build/cache the set of attachment IDs (images) that are missing alt text
 * AND are referenced by published content. Cached for 10 minutes and
 * invalidated on relevant content changes.
 */
if ( ! function_exists( 'aialttext_token_get_published_attachment_ids_cached' ) ) {
    function aialttext_token_get_published_attachment_ids_cached(): array {
        $cache_key = 'aialttext_token_pub_ids';
        $ids = get_transient( $cache_key );
        if ( is_array( $ids ) ) {
            return array_map( 'intval', $ids );
        }

        global $wpdb;
        $cpts    = (array) apply_filters( 'aialttext_token_published_post_types', array( 'post', 'page' ) );
        $cpts_in = implode( "','", array_map( 'esc_sql', $cpts ) );

        $sql = "
            SELECT DISTINCT p.ID
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm_alt
                   ON pm_alt.post_id = p.ID
                  AND pm_alt.meta_key = '_wp_attachment_image_alt'
            LEFT JOIN {$wpdb->postmeta} pm_file
                   ON pm_file.post_id = p.ID
                  AND pm_file.meta_key = '_wp_attached_file'
            WHERE p.post_type = 'attachment'
              AND p.post_mime_type LIKE 'image/%'
              AND (pm_alt.meta_value IS NULL OR TRIM(pm_alt.meta_value) = '')
              AND (
                    /* Featured image on a published post/page */
                    EXISTS (
                        SELECT 1
                        FROM {$wpdb->postmeta} pmf
                        JOIN {$wpdb->posts} c ON c.ID = pmf.post_id
                        WHERE pmf.meta_key = '_thumbnail_id'
                          AND pmf.meta_value = p.ID
                          AND c.post_status = 'publish'
                          AND c.post_type IN ('{$cpts_in}')
                    )
                    OR
                    /* Referenced in published post/page content */
                    EXISTS (
                        SELECT 1
                        FROM {$wpdb->posts} c
                        WHERE c.post_status = 'publish'
                          AND c.post_type IN ('{$cpts_in}')
                          AND (
                                c.post_content LIKE CONCAT('%wp-image-', p.ID, '%')
                             OR c.post_content LIKE CONCAT('%attachment_', p.ID, '%')
                             OR c.post_content LIKE CONCAT('%data-id=\"', p.ID, '\"%')
                             OR c.post_content LIKE CONCAT('%\"id\":', p.ID, '%')
                             OR c.post_content LIKE CONCAT('%', p.guid, '%')
                             OR (pm_file.meta_value IS NOT NULL AND c.post_content LIKE CONCAT('%', pm_file.meta_value, '%'))
                          )
                    )
              )
            ORDER BY p.ID ASC
        ";

        $ids = array_map( 'intval', (array) $wpdb->get_col( $sql ) );
        set_transient( $cache_key, $ids, 10 * MINUTE_IN_SECONDS );
        return $ids;
    }
}

/** Invalidate the cache when relevant published content changes. */
if ( ! function_exists( 'aialttext_token_invalidate_pub_cache_on_content_change' ) ) {
    function aialttext_token_invalidate_pub_cache_on_content_change( $new, $old, $post ) {
        if ( ! $post || ! is_object( $post ) ) return;
        $cpts = (array) apply_filters( 'aialttext_token_published_post_types', array( 'post', 'page' ) );
        if ( in_array( $post->post_type, $cpts, true ) && ( $new === 'publish' || $old === 'publish' ) ) {
            delete_transient( 'aialttext_token_pub_ids' );
        }
    }
    add_action( 'transition_post_status', 'aialttext_token_invalidate_pub_cache_on_content_change', 10, 3 );
    add_action( 'deleted_post', function(){ delete_transient( 'aialttext_token_pub_ids' ); } );
    add_action( 'save_post', function( $post_id, $post ){
        if ( $post && is_object( $post ) ) {
            $cpts = (array) apply_filters( 'aialttext_token_published_post_types', array( 'post', 'page' ) );
            if ( in_array( $post->post_type, $cpts, true ) ) {
                delete_transient( 'aialttext_token_pub_ids' );
            }
        }
    }, 10, 2 );
    // Also invalidate when the featured image or alt text meta changes
add_action('added_post_meta', function($object_id, $meta_key, $_){
    if ($meta_key === '_thumbnail_id' || $meta_key === '_wp_attachment_image_alt') {
        delete_transient('aialttext_token_pub_ids');
    }
}, 10, 3);

add_action('updated_postmeta', function($meta_id, $object_id, $meta_key, $_){
    if ($meta_key === '_thumbnail_id' || $meta_key === '_wp_attachment_image_alt') {
        delete_transient('aialttext_token_pub_ids');
    }
}, 10, 4);

add_action('deleted_post_meta', function($meta_ids, $object_id, $meta_key, $_){
    if ($meta_key === '_thumbnail_id' || $meta_key === '_wp_attachment_image_alt') {
        delete_transient('aialttext_token_pub_ids');
    }
}, 10, 4);
}

/**
 * Batch scan for attachments missing alt that are used on published posts/pages.
 * Cursor semantics: $offset is the last processed attachment ID (NOT SQL OFFSET).
 */
function aialttext_token_scan_media_library_published_only(
    int $batch_size = 10,
    int $offset = 0,
    array $exclude_ids = array()
) {
    global $wpdb;

    $ids_all = aialttext_token_get_published_attachment_ids_cached();
    $exclude = array_flip( array_map( 'intval', (array) $exclude_ids ) );

    // Respect cursor semantics: pick next IDs strictly greater than $offset
    $slice = array();
    foreach ( $ids_all as $id ) {
        $id = (int) $id;
        if ( $id <= (int) $offset ) continue;
        if ( isset( $exclude[ $id ] ) ) continue;
        $slice[] = $id;
        if ( count( $slice ) >= $batch_size ) break;
    }

    if ( empty( $slice ) ) {
        return array(); // signals "empty slice": wrap/finish handled by caller
    }

    // Fetch the small row set for the chosen IDs
    $in = implode( ',', array_map( 'intval', $slice ) );
    $rows = $wpdb->get_results( "
        SELECT p.ID, p.post_title, p.guid, p.post_mime_type
        FROM {$wpdb->posts} p
        WHERE p.ID IN ({$in})
        ORDER BY p.ID ASC
    " );

    return $rows;
}

/**
 * Batch processor with exclude list and rich return payload.
 * Returns:
 *  - processed (int), processed_ids (int[]), errors (int), failed ([['id'=>int,'message'=>string], ...])
 *  - error_messages (string[]), offset (int), total_images (int), completed (bool), latest_image (array|null)
 */
function aialttext_token_process_images_with_params(int $offset, int $batch_size, int $max_images = 0, array $exclude_ids = array(), bool $only_published = false) {
    $offset      = max(0, $offset);
    $batch_size  = max(1, $batch_size);
    $max_images  = max(0, $max_images);
    $exclude_ids = array_filter(array_map('intval', (array) $exclude_ids));

    // Fetch next candidates (front of the queue)
    $images = $only_published
    ? aialttext_token_scan_media_library_published_only($batch_size, $offset, $exclude_ids)
    : aialttext_token_scan_media_library($batch_size, $offset, $exclude_ids);

    $processed      = 0;
    $processed_ids  = array();
    $processed_details = array(); // NEW: id+alt+url+title for UI
    $errors         = 0;
    $failed         = array();   // each: ['id'=>int, 'message'=>string]
    $error_messages = array();
    $latest_image   = null;
    $out_of_tokens  = false; // stop the batch immediately when balance hits zero

// Process up to the requested batch size exactly.
// We intentionally avoid time-slicing so the UI’s batch size is honored.
// Hosts with strict timeouts will still be protected by the outer AJAX loop.

foreach ($images as $img) {
    // Accept rows as objects (default) or arrays (defensive)
    $id   = (int) (is_array($img) ? ($img['ID'] ?? 0) : ($img->ID ?? 0));
    $mime = (string) (is_array($img) ? ($img['post_mime_type'] ?? '') : ($img->post_mime_type ?? ''));

    if ($id <= 0) {
        // Skip silently if a bad row ever creeps in
        continue;
    }

        try {
            $context      = aialttext_token_build_context_for_image($id);
            $max_attempts = 3;
            $attempts     = 0;
            $result       = null;
            $last_error   = '';
        
            while ($attempts < $max_attempts) {
                $attempts++;
        
                $result = AiAltText_Token_Image_Processor::process_attachment($id, array(
                    'context' => $context,
                    'mime'    => $mime,
                ));
        
                // Success
                if (!is_wp_error($result)) {
                    break;
                }
        
                // Error path
                $last_error = $result->get_error_message();
        
                // If we hit a token exhaustion condition, mark and stop immediately (not transient)
// Prefer structured signal (our normalized WP_Error code), then fall back to message/HTTP hints.
$err_code = ($result instanceof WP_Error) ? $result->get_error_code() : '';
if (
    $err_code === 'insufficient_tokens'
    || preg_match('/\b402\b/', (string) $last_error)            // e.g., "HTTP 402"
    || (stripos($last_error, 'insufficient') !== false && stripos($last_error, 'token') !== false)
) {
    $out_of_tokens = true;
    break;
}

        
                // Retry only on transient network-ish issues
                if (!aialttext_token_is_transient_error($last_error)) {
                    break;
                }
        
                // simple back-off: ~250ms, ~500ms
                usleep(250000 * $attempts);
            }
        
            if (!is_wp_error($result)) {
                $processed++;
                $processed_ids[] = $id;
            
                // NEW: collect details per processed image (safe, additive)
                $edit_link = get_edit_post_link($id, 'raw');
if (!$edit_link) {
    $edit_link = admin_url('post.php?post=' . $id . '&action=edit');
}

$processed_details[] = array(
    'id'        => $id,
    'title'     => get_the_title($id),
    'alt'       => isset($result['alt']) ? $result['alt'] : '',
    'url'       => wp_get_attachment_url($id),
    'edit_link' => $edit_link,
);

// If we are filtering to published posts/pages, also backfill the alt="" inside those posts.
// This preserves existing non-empty alt attributes and only fills missing/empty alts.
if ( $only_published && ! empty( $result['alt'] ) ) {
    aialttext_token_update_alt_in_referencing_posts( $id, $result['alt'] );
}
            
                if (!$latest_image && !empty($result['alt'])) {
                    $latest_image = array(
                        'id'        => $id,
                        'title'     => get_the_title($id),
                        'alt_text'  => $result['alt'],
                        'url'       => wp_get_attachment_url($id),
                        'edit_link' => get_edit_post_link($id, 'raw') ?: admin_url('post.php?post=' . $id . '&action=edit'),
                    );
                }
            } else {
                $msg = $last_error !== '' ? $last_error : $result->get_error_message();
        
                // Record the failure once, annotating attempts so the UI error panel is informative
                $failed[] = array(
                    'id'      => $id,
                    'message' => ($attempts > 1 ? "[{$attempts} attempts] " : "") . $msg,
                );
                $error_messages[] = "Failed to generate alt text for image {$id}: {$msg}";
        
                if ($out_of_tokens) {
                    break; // whole batch pauses for top-up
                }
            }
        } catch (\Throwable $e) {
            $errors++;
            $failed[] = array('id' => $id, 'message' => $e->getMessage());
            $error_messages[] = "Error processing image {$id}: " . $e->getMessage();
        }
        
    }

    $total_images = $only_published
    ? aialttext_token_get_total_images_without_alt_on_published()
    : aialttext_token_get_total_images_without_alt();

    $empty_slice = empty($images);

    // Defaults: assume not completed and keep current cursor
    $completed   = false;
    $next_offset = $offset;

    // If we processed anything, advance cursor to the last processed ID
    if (!empty($processed_ids)) {
        $next_offset = (int) max(array_map('intval', (array) $processed_ids));
    }

    // If we ran out of tokens, pause exactly where we are (overrides advance)
    if (!empty($out_of_tokens)) {
        $next_offset = $offset;
    }

    // If this slice had no candidates, decide whether to wrap or finish
    if ($empty_slice) {
        if ((int) $total_images > 0) {
            // Work remains elsewhere (e.g., after wrapping or excluding attempted IDs)
            if (count($exclude_ids) >= (int) $total_images) {
                // Everything remaining is excluded for this run — stop politely
                $completed   = true;
                $next_offset = 0;
            } else {
                // Wrap to the front to continue scanning
                $next_offset = 0;
                // completed stays false
            }
        } else {
            // Truly done
            $completed   = true;
            $next_offset = 0;
        }
    }

    return array(
        'success'              => true,
        'processed'            => (int) $processed,
        'processed_ids'        => array_values(array_map('intval', (array) $processed_ids)),
        'processed_details'    => $processed_details, // NEW (id,title,alt,url)
        'errors'               => (int) $errors,
        'failed'               => $failed,
        'error_messages'       => $error_messages,
        'offset'               => (int) $next_offset,
        'total_images'         => (int) $total_images,
        'progress'             => 0,         // UI computes %
        'completed'            => (bool) $completed,
        'latest_image'         => $latest_image ?: null,
        'insufficient_tokens'  => !empty($out_of_tokens),
        'top_up_url'           => admin_url('admin.php?page=aialttext-token'),
    );
}

/* ---------------------------------------
 * Debug and test endpoints
 * --------------------------------------- */

// Add debug endpoint for testing context detection
add_action('wp_ajax_aialttext_token_test_context', function() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(
            array( 'message' => __( 'Permission denied', 'aialttext-token' ) )
        );
    }
    
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(
            array( 'message' => __( 'Security check failed', 'aialttext-token' ) )
        );
    }
    
    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
    
    if (!$post_id) {
        wp_send_json_error(
            array( 'message' => __( 'Post ID required', 'aialttext-token' ) )
        );
    }
    
    $results = array();
    
    // Test basic context detection
    if ($attachment_id) {
        $detected_context = aialttext_token_detect_upload_context($attachment_id);
        $results['detected_context'] = $detected_context;
    }
    
    // Test post context building
    $post_context = '';
    if (function_exists('aialttext_token_build_context_for_post')) {
        $post_context = aialttext_token_build_context_for_post($post_id);
    }
    $results['post_context'] = $post_context;
    
    // Test SEO context
    $seo_context = '';
    if (function_exists('aialttext_token_get_seo_context')) {
        $seo_context = aialttext_token_get_seo_context($post_id);
    }
    $results['seo_context'] = $seo_context;
    
    // Test WooCommerce context
    $woo_context = '';
    if (function_exists('aialttext_token_get_woocommerce_context')) {
        $woo_context = aialttext_token_get_woocommerce_context($post_id);
    }
    $results['woo_context'] = $woo_context;
    
    // Get post details
    $post = get_post($post_id);
    if ($post) {
        $results['post_details'] = array(
            'ID' => $post->ID,
            'post_type' => $post->post_type,
            'post_title' => $post->post_title,
            'post_status' => $post->post_status
        );
    } else {
        $results['post_details'] = array('error' => 'Post not found');
    }
    
    // Check settings
    $results['settings'] = array(
        'woocommerce_integration' => get_option('aialttext_token_woocommerce_integration', 1),
        'seo_integration' => get_option('aialttext_token_seo_integration', 1),
        'auto_generate' => get_option('aialttext_token_auto_generate', 0),
        'woocommerce_active' => class_exists('WooCommerce'),
        'yoast_active' => defined('WPSEO_VERSION')
    );
    
    // Check if Yoast has focus keyphrase for this post
    if (defined('WPSEO_VERSION')) {
        $yoast_focus = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        $results['yoast_focus_keyphrase'] = $yoast_focus;
    }
    
    wp_send_json_success($results);
});

/* ---------------------------------------
 * Enhanced Auto-generate on upload (respects setting)
 * Improved context detection for SEO and WooCommerce integration
 * --------------------------------------- */
add_action('add_attachment', function($attachment_id){
    if (!get_option('aialttext_token_auto_generate', 0)) return;
    if (get_post_type($attachment_id) !== 'attachment') return;
    $mime = get_post_mime_type($attachment_id);
    if (strpos((string) $mime, 'image/') !== 0) return;

    // Enhanced context detection for better SEO and WooCommerce integration
    $context_post_id = 0;
if (function_exists('aialttext_token_detect_current_post_id')) {
    $context_post_id = (int) aialttext_token_detect_current_post_id($attachment_id);
}
if ($context_post_id <= 0) {
    $context_post_id = aialttext_token_detect_upload_context($attachment_id);
}

    
    // Enhanced debugging for upload context
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] UPLOAD EVENT - attachment: ' . $attachment_id . ', detected context: ' . ($context_post_id ?: 'none'));
        error_log('[AI Alt Text] UPLOAD $_REQUEST data: ' . print_r($_REQUEST, true));
        error_log('[AI Alt Text] UPLOAD HTTP_REFERER: ' . (isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'none'));
    }

    $args = array('mime' => $mime);
    if ($context_post_id > 0) {
        $args['post_id'] = $context_post_id;
        
        // Log context detection for debugging
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log(sprintf('AI Alt Text: Detected context post %d for attachment %d', $context_post_id, $attachment_id));
            
            // Test if we can build context for this post
            if (function_exists('aialttext_token_build_context_for_post')) {
                $test_context = aialttext_token_build_context_for_post($context_post_id);
                error_log('[AI Alt Text] UPLOAD Test context build result: ' . $test_context);
            }
        }
    } else {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] UPLOAD WARNING: No context detected for attachment ' . $attachment_id);
        }
    }
    // If an alt exists but we're now on a product, allow overwrite to add product context
if (!empty(get_post_meta($attachment_id, '_wp_attachment_image_alt', true))) {
    if ($context_post_id > 0 && get_post_type($context_post_id) === 'product') {
        delete_post_meta($attachment_id, '_wp_attachment_image_alt');
    }
}

    // Best-effort; ignore failures to avoid blocking uploads
    try {
        
        AiAltText_Token_Image_Processor::process_attachment($attachment_id, $args);
    } catch (\Throwable $e) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('AI Alt Text auto-generation error: ' . $e->getMessage());
        }
    }
}, 10, 1);

/* ---------------------------------------
 * Enhanced Auto-generate on REST media upload (Gutenberg, Woo block editor)
 * --------------------------------------- */
add_action('rest_after_insert_attachment', function ( $attachment, $request, $creating ) {
    // Belt-and-suspenders: only run inside REST requests.
    if ( ! ( defined('REST_REQUEST') && REST_REQUEST ) ) {
        return;
    }

    try {
        // Only new uploads, respect the setting
        $auto    = get_option('aialttext_token_auto_generate', 0);
        $enabled = in_array($auto, array(1, '1', true, 'true', 'on', 'yes'), true);
        if ( ! $creating || ! $enabled ) {
            return;
        }

        $attachment_id = (int) $attachment->ID;
        if ( get_post_type($attachment_id) !== 'attachment' ) {
            return;
        }

        $mime = get_post_mime_type($attachment_id);
        if ( strpos((string) $mime, 'image/') !== 0 ) {
            return;
        }


    // --- Detect context FIRST (so we can decide whether to overwrite existing alt) ---
    $context_post_id = 0;

    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] REST UPLOAD EVENT - attachment: ' . $attachment_id);
        if ($request) {
            $params = $request->get_params();
            error_log('[AI Alt Text] REST request params: ' . print_r($params, true));
        }
        error_log('[AI Alt Text] REST HTTP_REFERER: ' . (isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'none'));
    }

    if ($request) {
        foreach (array('post','parent','uploadedTo','uploaded_to','context_post_id') as $key) {
            $val = (int) $request->get_param($key);
            if ($val > 0) {
                $context_post_id = $val;
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] REST found context from request param "' . $key . '": ' . $context_post_id);
                }
                break;
            }
        }
        if (!$context_post_id) {
            $ctx = $request->get_param('context');
            if (is_array($ctx)) {
                foreach (array('post','postId','parent','productId') as $key) {
                    if (!empty($ctx[$key]) && (int) $ctx[$key] > 0) {
                        $context_post_id = (int) $ctx[$key];
                        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                            error_log('[AI Alt Text] REST found context from nested context.' . $key . ': ' . $context_post_id);
                        }
                        break;
                    }
                }
            }
        }
    }

    // Fallbacks
    if ($context_post_id <= 0 && function_exists('aialttext_token_detect_current_post_id')) {
        $context_post_id = (int) aialttext_token_detect_current_post_id($attachment_id);
    }
    if ($context_post_id <= 0) {
        $context_post_id = aialttext_token_detect_upload_context($attachment_id);
    }
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] REST fallback context detection result: ' . ($context_post_id ?: 'none'));
    }

    // --- Decide whether to keep or overwrite existing alt text ---
    $existing = (string) get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if ($existing !== '') {
        $allow_overwrite = false;

        // Always overwrite on product pages to inject product context
        if ($context_post_id > 0 && get_post_type($context_post_id) === 'product') {
            $allow_overwrite = true;
        }

        // Or overwrite if it's just a placeholder (filename/title/generic)
        if (!$allow_overwrite && function_exists('aialttext_token_is_placeholder_alt')) {
            $allow_overwrite = aialttext_token_is_placeholder_alt($attachment_id, $existing);
        }

        if ($allow_overwrite) {
            delete_post_meta($attachment_id, '_wp_attachment_image_alt');
        } else {
            // Respect user-provided alt
            return;
        }
    }

    // Build args for processor
    $args = array('mime' => $mime);
    if ($context_post_id > 0) {
        $args['post_id'] = $context_post_id;
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log(sprintf('AI Alt Text REST: Detected context post %d for attachment %d', $context_post_id, $attachment_id));
        }
    }

    // Defer heavy API work to the existing background worker so that
    // REST uploads (Gutenberg/media) stay fast and always return valid JSON.
    if ( ! wp_next_scheduled('aialttext_token_delayed_generate_alt', array($attachment_id)) ) {
        wp_schedule_single_event(
            time() + 10,
            'aialttext_token_delayed_generate_alt',
            array($attachment_id)
        );
    }
    } catch (\Throwable $e) {
        // Critical safety net: never let REST upload failures break the JSON response.
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('AI Alt Text REST handler failure: ' . $e->getMessage());
        }
    }
}, 10, 3);


/* ---------------------------------------
 * Delayed context retry for missed uploads
 * --------------------------------------- */
add_action('wp_generate_attachment_metadata', function($metadata, $attachment_id) {
    // This runs later in the upload process, good for catching missed uploads
    if (!get_option('aialttext_token_auto_generate', 0)) return $metadata;
    
    $mime = get_post_mime_type($attachment_id);
    if (strpos((string) $mime, 'image/') !== 0) return $metadata;
    
    // Check if we already have alt text
    $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if (!empty($existing_alt)) return $metadata;
    
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] DELAYED RETRY for attachment ' . $attachment_id);
    }
    
    // Try context detection again
    $context_post_id = aialttext_token_detect_upload_context($attachment_id);
    
    if ($context_post_id > 0) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] DELAYED RETRY found context: ' . $context_post_id);
        }
        
        $args = array('mime' => $mime, 'post_id' => $context_post_id);
        
        try {
            AiAltText_Token_Image_Processor::process_attachment($attachment_id, $args);
        } catch (\Throwable $e) {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] DELAYED RETRY error: ' . $e->getMessage());
            }
        }
    }
    
    return $metadata;
}, 10, 2);

/* ---------------------------------------
 * AJAX: single generate (used by admin.js / attachment.js)
 * Enhanced with better context detection
 * --------------------------------------- */
add_action('wp_ajax_aialttext_token_generate_single', function () {
    // Checkpoint 1
    error_log('CHECKPOINT 1: Handler started');
    
    if ( ! ob_get_level() ) {
        ob_start();
    }

    // Checkpoint 2
    error_log('CHECKPOINT 2: Before permission check');
    
    if (!current_user_can('upload_files') || !aialttext_token_user_can('generate')) {
        error_log('CHECKPOINT 2.1: Permission denied');
        if ( ob_get_length() ) {
            ob_clean();
        }
        wp_send_json_error(array( 'message' => 'Permission denied' ));
        die();
    }

    // Checkpoint 3
    error_log('CHECKPOINT 3: Before nonce check');
    
    $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
    if ( ! $nonce || ! wp_verify_nonce( $nonce, 'aialttext_token_nonce' ) ) {
        error_log('CHECKPOINT 3.1: Nonce failed');
        if ( ob_get_length() ) {
            ob_clean();
        }
        wp_send_json_error(array( 'message' => 'Invalid nonce' ), 403);
        die();
    }

    // Checkpoint 4
    error_log('CHECKPOINT 4: Getting attachment ID');
    
    $id      = isset($_POST['attachment_id']) ? absint($_POST['attachment_id']) : 0;
    $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
    
    error_log('CHECKPOINT 4.1: ID=' . $id . ', post_id=' . $post_id);

    if (!$id) {
        error_log('CHECKPOINT 4.2: Missing attachment_id');
        if ( ob_get_length() ) {
            ob_clean();
        }
        wp_send_json_error(array( 'message' => 'Missing attachment_id' ));
        die();
    }

    // Checkpoint 5
    error_log('CHECKPOINT 5: Building context');
    
    $mime = get_post_mime_type($id);
    $args = array('mime' => $mime);

    if ($post_id <= 0 && function_exists('aialttext_token_detect_current_post_id')) {
        $post_id = (int) aialttext_token_detect_current_post_id($id);
    }
    if ($post_id <= 0 && function_exists('aialttext_token_detect_upload_context')) {
        $post_id = (int) aialttext_token_detect_upload_context($id);
    }

    // Checkpoint 6
    error_log('CHECKPOINT 6: Before get_image_context');

    if ($post_id > 0) {
        $args['post_id'] = $post_id;
        if (function_exists('aialttext_token_get_image_context')) {
            try {
                $args['context'] = aialttext_token_get_image_context($id, $post_id);
                error_log('CHECKPOINT 6.1: Context built, length=' . strlen($args['context']));
            } catch (\Throwable $e) {
                error_log('CHECKPOINT 6.2: Context build FAILED: ' . $e->getMessage());
            }
        }
    }

    // Checkpoint 7
    error_log('CHECKPOINT 7: Before process_attachment');

    try {
        $result = AiAltText_Token_Image_Processor::process_attachment($id, $args);
        error_log('CHECKPOINT 8: process_attachment completed');
    } catch ( \Throwable $e ) {
        error_log('CHECKPOINT 8.1: process_attachment FAILED: ' . $e->getMessage());
        
        if ( ob_get_length() ) {
            ob_clean();
        }
        wp_send_json_error(array(
            'message' => 'Error: ' . $e->getMessage()
        ), 500);
        die();
    }

    // Checkpoint 9
    error_log('CHECKPOINT 9: Checking result');

    if (is_wp_error($result)) {
        error_log('CHECKPOINT 9.1: WP_Error: ' . $result->get_error_message());
        
        $msg  = $result->get_error_message();
        $code = $result->get_error_code();

        // Special-case unsupported / non-image file types so we can show a clearer message in the UI.
        // We treat anything not in the core supported set as unsupported.
        $mime_for_error = ! empty( $mime ) ? $mime : get_post_mime_type( $id );
        $allowed_mimes  = array( 'image/jpeg', 'image/png', 'image/webp', 'image/gif' );
        $is_image       = ( strpos( (string) $mime_for_error, 'image/' ) === 0 );

        if ( $code === 'not_image' || ( $is_image && ! in_array( $mime_for_error, $allowed_mimes, true ) ) ) {
            if ( ob_get_length() ) {
                ob_clean();
            }

            wp_send_json_error( array(
                'message'           => 'There was a technical issue generating alt text. Img Alt Gen Pro supports: PNG, JPG, webP, and GIF.',
                'technical_issue'   => true,
                'unsupported_type'  => true,
            ) );

            die();
        }

        $is_out =
            $code === 'insufficient_tokens' ||
            preg_match('/\b402\b/', (string) $msg) ||
            (stripos($msg, 'insufficient') !== false && stripos($msg, 'token') !== false);

        if ($is_out) {
            if ( ob_get_length() ) {
                ob_clean();
            }
            wp_send_json_error(array(
                'message'              => 'You have no tokens remaining.',
                'insufficient_tokens'  => true,
                'top_up_url'           => admin_url('admin.php?page=aialttext-token'),
            ));
            die();
        }

        if ( ob_get_length() ) {
            ob_clean();
        }
        wp_send_json_error(array(
            'message'         => $msg,
            'technical_issue' => true,
        ));
        die();
    }

    // Checkpoint 10
    error_log('CHECKPOINT 10: Sending success response');

    if ( ob_get_length() ) {
        ob_clean();
    }

    wp_send_json_success(array(
        'attachment_id' => $id,
        'alt_text'      => isset($result['alt_text']) ? $result['alt_text'] : (isset($result['alt']) ? $result['alt'] : ''),
        'model'         => isset($result['model']) ? $result['model'] : null,
        'debug_logs'    => isset($result['debug_logs']) ? $result['debug_logs'] : null,
    ));
    
    error_log('CHECKPOINT 11: Response sent');
    die();
});

// Diagnostic endpoint
add_action('wp_ajax_aialttext_token_diagnostic', function() {
    wp_send_json_success(array(
        'message' => 'AJAX handler is working',
        'time' => current_time('mysql'),
        'user_id' => get_current_user_id(),
        'can_upload' => current_user_can('upload_files'),
        'function_exists' => function_exists('aialttext_token_user_can'),
        'class_exists' => class_exists('AiAltText_Token_Image_Processor'),
    ));
});
// TEMPORARY TEST HANDLER - Remove after debugging
add_action('wp_ajax_aialttext_token_test_simple', function() {
    error_log('TEST: Simple handler called');
    wp_send_json_success(array('message' => 'Test handler works!'));
    die();
});