<?php
if (!defined('ABSPATH')) exit;

class AiAltText_Token_API_Client {
    private $api_base; 
    private $access_key;
    
    public function __construct(){
        $this->api_base = defined('AIALTTEXT_TOKEN_API_BASE') ? AIALTTEXT_TOKEN_API_BASE : admin_url('admin-ajax.php');
        $this->access_key = trim(get_option('aialttext_token_access_key',''));
    }
    
    
    private function request($action, $method = 'POST', $data = array()) {
        $url  = $this->api_base;

        // Remember whether this call was made with a stored access key.
    $had_access_key = ! empty( $this->access_key );
    
        // Enforce HTTPS for remote endpoints (local admin-ajax remains allowed).
        $parsed = wp_parse_url($url);
        if ( ! empty($parsed['host']) ) {
            $is_local_admin = (untrailingslashit($url) === untrailingslashit(admin_url('admin-ajax.php')));
            if ( ! $is_local_admin && (!isset($parsed['scheme']) || strtolower($parsed['scheme']) !== 'https') ) {
                return array('success' => false, 'error' => array('code' => 'insecure_endpoint', 'message' => __( 'Refusing non-HTTPS API base.', 'aialttext-token' )));
            }
        }
    
        $body = array_merge((array) $data, array('action' => $action));

        /**
         * Send the access key both:
         *  - in the Authorization header (preferred), and
         *  - in the POST body (for environments that strip Authorization).
         *
         * The portal authenticator explicitly supports both, so this is safe
         * and improves robustness without changing server-side behaviour.
         */
        if ( ! empty( $this->access_key ) ) {
            // Provide the key in the POST body so aialttext_account_authenticate_request()
            // can always read $_POST['access_key'], even if Authorization is stripped.
            $body['access_key'] = $this->access_key;
        }

        $headers = array(
            'Accept'        => 'application/json',
            'Cache-Control' => 'no-cache',
            // Keep default WP form encoding; do not force JSON content-type here
            'User-Agent'    => 'IMG-Alt-Gen-Pro/' . (defined('AIALTTEXT_TOKEN_VERSION') ? AIALTTEXT_TOKEN_VERSION : 'unknown') . '; ' . home_url(),
        );

        if ( ! empty( $this->access_key ) ) {
            $headers['Authorization'] = 'Bearer ' . $this->access_key;
        }

        // Ensure there is an idempotency key for any mutating call
        if ( ! isset($body['idempotency_key']) ) {
            if (function_exists('wp_generate_uuid4')) {
                $body['idempotency_key'] = wp_generate_uuid4();
            } else {
                $body['idempotency_key'] = md5(uniqid('', true));
            }
        }
        
        // Sign the canonical JSON of the body so the admin service can verify integrity quickly
        $canonical_json = wp_json_encode($body);
        $headers['X-Idempotency-Key'] = $body['idempotency_key'];
        if ( ! empty($this->access_key) ) {
            $headers['X-Signature'] = hash_hmac('sha256', $canonical_json, $this->access_key);
        }
    
        // Slightly shorter timeouts for quick endpoints; keep others at 20s
$quick_endpoints = array(
    'aialttext_account_exists',
    'aialttext_account_password_forgot',
);

$timeout = in_array($action, $quick_endpoints, true) ? 12 : 20;

$args = array(
    'method'      => $method,
    'timeout'     => $timeout,
    'redirection' => 3,
    'headers'     => $headers,
    'sslverify'   => true,
    'user-agent'  => 'IMG-Alt-Gen-Pro/' . (defined('AIALTTEXT_TOKEN_VERSION') ? AIALTTEXT_TOKEN_VERSION : 'unknown'),
);
    
        if (strtoupper($method) !== 'GET') {
            $args['body'] = $body;
        } else {
            $url = add_query_arg($body, $url);
        }
    
        // Simple bounded retry on transient 429/5xx
        $attempts = 0; $max = 4; $last_error = '';
        while ($attempts++ < $max) {
            $res = wp_remote_request($url, $args);
            if (is_wp_error($res)) {
                $last_error = $res->get_error_message();
            } else {
                $code = wp_remote_retrieve_response_code($res);
                if ($code >= 200 && $code < 300) {
                    $json = json_decode(wp_remote_retrieve_body($res), true);
                    return is_array($json) ? $json : array('success' => false, 'error' => array('code' => 'bad_json', 'message' => __( 'Bad JSON', 'aialttext-token' )));
                }
                // transient?
                if ($code == 429 || ($code >= 500 && $code <= 599)) {
                    usleep(200000 * $attempts); // linear backoff 200ms, 400ms, ...
                    continue;
                }
                $body_text = wp_remote_retrieve_body($res);
                $json      = json_decode($body_text, true);
                if (is_array($json) && array_key_exists('success', $json)) {
                    // Auto-logout locally only when we are sure the stored access key is no longer valid.
                    if (empty($json['success'])) {
                        $err_code = '';
                        $err_msg  = '';
        
                        if (!empty($json['error']['code'])) {
                            $err_code = strtolower((string) $json['error']['code']);
                        }
                        if (!empty($json['error']['message'])) {
                            $err_msg = strtolower((string) $json['error']['message']);
                        }
        
                        $has_key = ! empty($this->access_key);
        
                        /**
                         * Determine whether this is really an authentication / key error.
                         *
                         * - 'auth', 'unauthorized', 'invalid_key' are explicit auth failures.
                         * - 'not_found' is only treated as auth-related if the message clearly refers
                         *   to the access/API key (e.g. "Access key not found").
                         *
                         * This avoids unlinking the account for unrelated "not found" errors
                         * such as "Account not found for this email" or "Ticket not found".
                         */
                        $is_auth_error =
                            in_array($err_code, array('auth', 'unauthorized', 'invalid_key'), true)
                            || (
                                $err_code === 'not_found'
                                && (strpos($err_msg, 'access key not found') !== false
                                    || strpos($err_msg, 'api key') !== false)
                            );
        
                            if ($had_access_key && $has_key && $is_auth_error) {
                                // How recently was this key issued?
                                $issued_at      = (int) get_option('aialttext_token_access_key_issued_at', 0);
                                $now            = time();
                                // Default: treat keys as "recent" for the first 120 seconds.
                                $recent_window  = (int) apply_filters('aialttext_token_recent_key_window', 120);
                                $is_recent_key  = ($issued_at > 0 && ($now - $issued_at) < $recent_window);
    
                                /**
                                 * Allow integrators to override whether we auto-logout on auth error.
                                 *
                                 * @param bool  $should_auto_logout Default decision (true unless key is very recent).
                                 * @param array $context            Context including action and error details.
                                 */
                                $should_auto_logout = apply_filters(
                                    'aialttext_token_should_auto_logout',
                                    !$is_recent_key,
                                    array(
                                        'action'        => $action,
                                        'error_code'    => $err_code,
                                        'error_message' => $err_msg,
                                        'had_access_key'=> $had_access_key,
                                        'has_key'       => $has_key,
                                        'issued_at'     => $issued_at,
                                    )
                                );
    
                                if ($should_auto_logout) {
                                    // Clear stored access key and any cached status that may still contain
                                    // the old “invalid key” response.
                                    delete_option('aialttext_token_access_key');
                                    delete_transient('aialttext_token_status_cache');
                                    set_transient('aialttext_token_auto_logged_out', 1, HOUR_IN_SECONDS);
    
                                    /**
                                     * Mirror the manual logout hook used elsewhere so integrations can tidy up.
                                     */
                                    do_action('aialttext_token_after_logout');
    
                                    if (function_exists('aialttext_token_debug_log')) {
                                        aialttext_token_debug_log(
                                            'Auto-logged out due to auth error from ' . $action .
                                            ' (code=' . $err_code . ', message=' . $err_msg . ')'
                                        );
                                    }
                                } else {
                                    if (function_exists('aialttext_token_debug_log')) {
                                        aialttext_token_debug_log(
                                            'Auth error for recently-issued key; preserving linkage. ' .
                                            'Action=' . $action . ', code=' . $err_code . ', message=' . $err_msg
                                        );
                                    }
                                }
                            }
                    }
        
                    // Return the server's structured JSON (e.g., { success:false, error:{code, message}, ... })
                    return $json;
                }
                $last_error = 'HTTP ' . $code;
            }
            usleep(150000); // brief pause before retry
        }
    
        return array('success' => false, 'error' => array('code' => 'request_failed', 'message' => $last_error ?: 'Unknown error'));
    }
    
    
    public function link_account($email, $password, $domain) {
        return $this->request('aialttext_account_link', 'POST', array(
            'email'    => $email,
            'password' => $password,
            'domain'   => $domain,
        ));
    }

    public function register_account($email, $password) {
        // Explicit register path; if you need to unify, pass create_if_missing=1 to the link endpoint.
        return $this->request('aialttext_account_register', 'POST', array(
            'email'    => $email,
            'password' => $password,
            'domain'   => parse_url(get_site_url(), PHP_URL_HOST),
        ));
    }
    
    public function login_account($email, $password) {
        // Login-only. Never auto-creates an account.
        return $this->request('aialttext_account_login', 'POST', array(
            'email'    => $email,
            'password' => $password,
            'domain'   => parse_url(get_site_url(), PHP_URL_HOST),
        ));
    }

    public function account_exists($email) {
        return $this->request('aialttext_account_exists', 'POST', array('email' => $email));
    }
    
    public function password_forgot($email) {
        // Admin site will send the email and build product-site reset URL
        return $this->request('aialttext_account_password_forgot', 'POST', array('email' => $email));
    }

    public function update_email($new_email) {
        return $this->request('aialttext_account_update_email', 'POST', array(
            'new_email'  => $new_email,
        ));
    }
    
    public function password_change($current_password, $new_password) {
        return $this->request('aialttext_account_password_change', 'POST', array(
            'current_password' => $current_password,
            'new_password'     => $new_password,
        ));
    }
    
    public function status() {
        $cache_key = 'aialttext_token_status_cache';
        $cached = get_transient($cache_key);
        if (is_array($cached)) {
            return $cached;
        }
        $res = $this->request('aialttext_account_status', 'POST');
        if (is_array($res) && array_key_exists('success', $res)) {
            set_transient($cache_key, $res, 60); // cache for 60s
        }
        return $res;
    }
    
    public function pricing() { 
        return $this->request('aialttext_account_pricing', 'GET'); 
    }

    public function delete_account() {
        // explicit confirm flag is required by the server
        return $this->request('aialttext_account_delete', 'POST', array('confirm' => '1'));
    }
    
    public function purchase($package_id, $success_url, $cancel_url, $domain) { 
        return $this->request('aialttext_account_purchase', 'POST', compact('package_id', 'success_url', 'cancel_url', 'domain')); 
    }
    
    public function confirm($session_id) { 
        return $this->request('aialttext_confirm_payment', 'POST', compact('session_id')); 
    }

    public function submit_support_ticket($data){
        return $this->request('aialttext_support_submit', 'POST', $data);
    }
    public function list_support_tickets(){
        return $this->request('aialttext_support_list', 'GET', array());
    }
    public function get_support_ticket($ticket_id){
        return $this->request('aialttext_support_get', 'GET', array('ticket_id' => (int)$ticket_id));
    }
    public function reply_support_ticket($ticket_id, $message){
        return $this->request('aialttext_support_reply', 'POST', array(
            'ticket_id' => (int)$ticket_id,
            'message'   => (string)$message,
        ));
    }
    
    public function generate($image_url, $image_title = '', $context = '', $image_b64 = null, $image_mime = null) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] API Client - generate() called with URL: ' . $image_url);
        }
    
        $payload = array(
            'image_url'       => $image_url,
            'image_title'     => $image_title,
            'context'         => $context,
            'site_name'       => get_bloginfo('name'), // for admin-side logs & usage views
            'output_language' => get_option('aialttext_token_output_language', 'English'),
            'domain'          => (string) wp_parse_url( home_url(), PHP_URL_HOST ),
            // Deterministic idempotency key prevents accidental duplicate charges
            'idempotency_key' => substr(hash('sha256', (string)$image_url . '|' . (string)$image_title . '|' . (string) wp_parse_url(home_url(), PHP_URL_HOST) ), 0, 32),
        );

        if (is_string($image_b64) && $image_b64 !== '' && is_string($image_mime) && $image_mime !== '') {
            $payload['image_b64']  = $image_b64;
            $payload['image_mime'] = $image_mime;
        }
    
        $result = $this->request('aialttext_account_generate', 'POST', $payload);
    
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] API Client - response: ' . print_r($result, true));
        }
        return $result;
    }
    
    /**
     * Fetch usage with window + pagination.
     * Accepts: days, start_date, end_date, per_page, cursor
     */
    public function usage($args = array()) {
        $payload  = array();
        $allowed  = array('days','start_date','end_date','per_page','cursor','domain');
        foreach ($allowed as $k) {
            if (isset($args[$k])) { $payload[$k] = $args[$k]; }
        }
        // Default to this site’s domain if not supplied
        if (empty($payload['domain'])) {
            $host = wp_parse_url( home_url(), PHP_URL_HOST );
            if (is_string($host) && $host !== '') {
                $payload['domain'] = $host;
            }
        }
        return $this->request('aialttext_account_usage', 'POST', $payload);
    }
}

function aialttext_token_cached_pricing(){
    $cached = get_transient('aialttext_token_pricing'); 
    if ($cached) return $cached;
    
    $api = new AiAltText_Token_API_Client(); 
    $result = $api->pricing();
    
    if (!empty($result['success'])) {
        set_transient('aialttext_token_pricing', $result, 15 * MINUTE_IN_SECONDS);
    }
    
    return $result;
}

function aialttext_token_check_balance() {
    // If there's no stored key, short-circuit without calling the remote API.
    $stored_key = trim( (string) get_option('aialttext_token_access_key', '') );
    if ( $stored_key === '' ) {
        return 0;
    }

    $api    = new AiAltText_Token_API_Client();
    $result = $api->status();

    if (!empty($result['success'])) {
        return (int) ($result['token_balance'] ?? 0);
    }

    // Defensive duplication of the interceptor logic in request().
    $err_code = strtolower((string) ($result['error']['code'] ?? ''));
    $err_msg  = strtolower((string) ($result['error']['message'] ?? ''));

    $has_key = ( $stored_key !== '' );

    $is_auth_error =
        in_array($err_code, array('auth', 'unauthorized', 'invalid_key'), true)
        || (
            $err_code === 'not_found'
            && (strpos($err_msg, 'access key not found') !== false
                || strpos($err_msg, 'api key') !== false)
        );

        if ($has_key && $is_auth_error) {
            $issued_at      = (int) get_option('aialttext_token_access_key_issued_at', 0);
            $now            = time();
            $recent_window  = (int) apply_filters('aialttext_token_recent_key_window', 120);
            $is_recent_key  = ($issued_at > 0 && ($now - $issued_at) < $recent_window);
    
            $should_auto_logout = apply_filters(
                'aialttext_token_should_auto_logout',
                !$is_recent_key,
                array(
                    'action'        => 'aialttext_account_status',
                    'error_code'    => $err_code,
                    'error_message' => $err_msg,
                    'had_access_key'=> $has_key,
                    'has_key'       => $has_key,
                    'issued_at'     => $issued_at,
                )
            );
    
            if ($should_auto_logout) {
                delete_option('aialttext_token_access_key');
                set_transient('aialttext_token_auto_logged_out', 1, HOUR_IN_SECONDS);
                do_action('aialttext_token_after_logout');
    
                if (function_exists('aialttext_token_debug_log')) {
                    aialttext_token_debug_log(
                        'Auto-logged out from balance check (code=' . $err_code . ', message=' . $err_msg . ')'
                    );
                }
            } else {
                if (function_exists('aialttext_token_debug_log')) {
                    aialttext_token_debug_log(
                        'Auth error during balance check for recently-issued key; preserving linkage. ' .
                        'Code=' . $err_code . ', message=' . $err_msg
                    );
                }
            }
        }
    
        return 0;
    }

function aialttext_token_has_sufficient_balance($required_tokens = 1) {
    $balance = aialttext_token_check_balance();
    return $balance >= $required_tokens;
}