/**
 * AI Alt Text Generator - Gutenberg Block Editor Integration
 * Adds "Generate AI Alt Text" button to Image block settings panel
 */

// (intentionally blank — remove the global destructuring)

(function(wp, $) {
    'use strict';

    if (!wp || !wp.blocks || !wp.element || !wp.components || !wp.data) {
        console.warn('[AI Alt Text] Required WordPress packages not available');
        return;
    }

    const { createElement: el, Fragment, useState, useEffect } = wp.element;
    const { InspectorControls } = wp.blockEditor || wp.editor;
    const { PanelBody, Button, Notice } = wp.components;
    const { addFilter } = wp.hooks;
    const { select, dispatch } = wp.data;
    const { __, _n, sprintf } = wp.i18n;

    /**
     * Get current post/product ID for context
     */
    function getCurrentPostId() {
        try {
            const editorSelect = select('core/editor');
            if (editorSelect) {
                const postId = editorSelect.getCurrentPostId();
                if (postId) return parseInt(postId, 10);
            }
        } catch (e) {
            console.debug('[AI Alt Text] Could not get post ID from editor:', e);
        }
        
        // Fallback to global
        if (window.aialttext_current_post_id) {
            return parseInt(window.aialttext_current_post_id, 10);
        }
        
        // Fallback to URL parameter
        const urlParams = new URLSearchParams(window.location.search);
        const postParam = urlParams.get('post');
        if (postParam) {
            return parseInt(postParam, 10);
        }
        
        return 0;
    }

    /**
     * Build inline context from current editor state
     */
    function buildInlineContext() {
        try {
            const editorSelect = select('core/editor');
            if (!editorSelect) return '';

            const title = editorSelect.getEditedPostAttribute('title') || '';
            const excerpt = editorSelect.getEditedPostAttribute('excerpt') || '';
            const content = editorSelect.getEditedPostAttribute('content') || '';

            let context = '';
            if (title) {
                context += sprintf( __('This image appears on: "%s". ', 'aialttext-token'), title );
            }
            
            const desc = excerpt || content.substring(0, 500);
            if (desc) {
                // Strip HTML tags for cleaner context
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = desc;
                const cleanDesc = (tempDiv.textContent || tempDiv.innerText || '').trim();
                if (cleanDesc) {
                    context += `Content: ${cleanDesc}. `;
                }
            }

            return context.trim();
        } catch (e) {
            console.debug('[AI Alt Text] Error building context:', e);
            return '';
        }
    }

    /**
     * Generate AI Alt Text Component
     */
    function GenerateAltTextButton({ clientId, imageId, currentAlt }) {
        const [isGenerating, setIsGenerating] = useState(false);
        const [message, setMessage] = useState(null);
        const [messageType, setMessageType] = useState('info');

        const handleGenerate = async () => {
            if (!imageId) {
                setMessage(__('No image selected', 'aialttext-token'));
                setMessageType('error');
                return;
            }

            setIsGenerating(true);
            setMessage(null);

            try {
                const postId = getCurrentPostId();
                const context = buildInlineContext();

                // Build the payload once
                const payload = {
                    attachment_id: imageId,
                    post_id: postId,
                    context: context
                };

                                // Resolve a single, reliable AJAX endpoint so we never accidentally
                // POST back to the editor URL (which causes “not a valid JSON” errors).
                const endpoint =
                    (window.aialttext_token_gutenberg &&
                        window.aialttext_token_gutenberg.ajax_url &&
                        window.aialttext_token_gutenberg.nonce)
                        ? {
                              ajaxUrl: window.aialttext_token_gutenberg.ajax_url,
                              nonce: window.aialttext_token_gutenberg.nonce,
                          }
                        : (window.aialttext_token_ajax &&
                           window.aialttext_token_ajax.ajax_url &&
                           window.aialttext_token_ajax.nonce)
                        ? {
                              ajaxUrl: window.aialttext_token_ajax.ajax_url,
                              nonce: window.aialttext_token_ajax.nonce,
                          }
                        : null;

                if (!endpoint) {
                    console.error('[AI Alt Text] No AJAX configuration found for Gutenberg integration.');
                    setMessage('✗ AI Alt Text could not connect. Please reload the editor.');
                    setMessageType('error');
                    return;
                }

                 let response;

                // Prefer the shared helper if it exists (same behaviour as admin.js / attachment.js)
                if (window.aialttextAjax) {
                    response = await window.aialttextAjax('aialttext_token_generate_single', payload);
                } else {
                    // Fallback to a direct jQuery AJAX call with an explicit endpoint
                    response = await $.ajax({
                        url: endpoint.ajaxUrl,
                        type: 'POST',
                        dataType: 'json',
                        data: {
                            action: 'aialttext_token_generate_single',
                            nonce: endpoint.nonce,
                            attachment_id: imageId,
                            post_id: postId,
                            context: context
                        },
                        // Add error handler for empty responses
                        error: function(jqXHR, textStatus, errorThrown) {
                            console.error('[AI Alt Text] AJAX Error Details:');
                            console.error('Status:', jqXHR.status);
                            console.error('Status Text:', textStatus);
                            console.error('Error Thrown:', errorThrown);
                            console.error('Response Text:', jqXHR.responseText);
                            console.error('Response Headers:', jqXHR.getAllResponseHeaders());
                        }
                    });
                }

                if (response && response.success && response.data) {
                    const altText = response.data.alt_text || response.data.alt || '';

                    if (altText) {
                        // Update the block's alt attribute
                        dispatch('core/block-editor').updateBlockAttributes(clientId, {
                            alt: altText
                        });

                        // Media library alt text is already persisted server-side by
                        // AiAltText_Token_Image_Processor::process_attachment().
                        // No extra AJAX round-trip is required here.

                        setMessage('✓ Alt text generated successfully!');
                        setMessageType('success');

                        console.log('[AI Alt Text] Generated:', altText);


                        // If there are debug logs, show them in the console
                        if (response.data.debug_logs) {
                            console.group('[AI Alt Text] Debug Logs');
                            response.data.debug_logs.forEach(log => {
                                console.log(log.message, log.data || '');
                            });
                            console.groupEnd();
                        }
                    } else {
                        throw new Error('No alt text returned');
                    }
                } else if (response && response.data && response.data.insufficient_tokens) {
                    // Specific insufficient-tokens handling
                    setMessage('⚠ Out of tokens. Please top up.');
                    setMessageType('error');

                    if (window.aialttextShowTopUpModal) {
                        setTimeout(() => {
                            window.aialttextShowTopUpModal(response.data.top_up_url);
                        }, 1000);
                    }
                } else {
                    const errorMsg =
                        response && response.data && response.data.message
                            ? response.data.message
                            : 'Failed to generate alt text';
                    setMessage(`✗ ${errorMsg}`);
                    setMessageType('error');
                }
            } catch (error) {
                console.error('[AI Alt Text] Generation error:', error);
                console.error('[AI Alt Text] Error status:', error.status);
                console.error('[AI Alt Text] Error statusText:', error.statusText);
                console.error('[AI Alt Text] Error responseText:', error.responseText);
                console.error('[AI Alt Text] Full error object:', JSON.stringify(error, null, 2));
                
                // Display detailed error to user
                let errorMsg = '✗ Error generating alt text.';
                
                if (error.responseText) {
                    errorMsg += ' Response: ' + error.responseText.substring(0, 200);
                } else if (error.statusText) {
                    errorMsg += ' Status: ' + error.statusText;
                } else {
                    errorMsg += ' Empty response from server.';
                }
                
                // Check for 402 status (insufficient tokens)
                if (error.status === 402) {
                    setMessage('⚠ Out of tokens. Please top up.');
                    setMessageType('error');
                    
                    if (window.aialttextShowTopUpModal) {
                        setTimeout(() => {
                            window.aialttextShowTopUpModal();
                        }, 1000);
                    }
                } else {
                    setMessage(errorMsg);
                    setMessageType('error');
                }
            } finally {
                setIsGenerating(false);
                
                // Clear message after 5 seconds
                setTimeout(() => {
                    setMessage(null);
                }, 5000);
            }
        };

        return el(Fragment, {},
            el(Button, {
                variant: 'primary',
                onClick: handleGenerate,
                disabled: isGenerating || !imageId,
                style: {
                    background: 'linear-gradient(135deg, #673FC5 0%, #4B2AA3 100%)',
                    border: 'none',
                    color: '#fff',
                    width: '100%',
                    justifyContent: 'center',
                    marginTop: '8px'
                }
            }, isGenerating ? __('Generating...', 'aialttext-token') : __('✨ Generate AI Alt Text', 'aialttext-token')),
            
            message && el(Notice, {
                status: messageType,
                isDismissible: false,
                style: { marginTop: '8px', marginBottom: '0' }
            }, message)
        );
    }

    /**
     * Add our button to the Image block's inspector controls
     */
    const addAltTextGeneratorControl = wp.compose.createHigherOrderComponent((BlockEdit) => {
        return (props) => {
            const { name, attributes, clientId } = props;

            // Only add to core/image blocks
            if (name !== 'core/image') {
                return el(BlockEdit, props);
            }

            const imageId = attributes.id;
            const currentAlt = attributes.alt || '';

            return el(Fragment, {},
                el(BlockEdit, props),
                el(InspectorControls, {},
                    el(PanelBody, {
                        title: __('AI Alt Text Generator', 'aialttext-token'),
                        initialOpen: false
                    },
                        el('div', { 
                            style: { padding: '0' },
                            className: 'aialttext-gutenberg-panel'
                        },
                            el('p', { 
                                style: { 
                                    fontSize: '12px', 
                                    color: '#757575',
                                    marginTop: '0',
                                    marginBottom: '8px'
                                }
                            }, imageId 
                                ? __('Generate AI-powered alt text for this image with SEO and product context. Tip: Please ensure to save the post with SEO focus keyphrase set.', 'aialttext-token')
                                : __('Select an image first to generate alt text.', 'aialttext-token')
                            ),
                            el(GenerateAltTextButton, {
                                clientId: clientId,
                                imageId: imageId,
                                currentAlt: currentAlt
                            })
                        )
                    )
                )
            );
        };
    }, 'addAltTextGeneratorControl');

    // Register the filter
    addFilter(
        'editor.BlockEdit',
        'aialttext-token/add-alt-text-generator',
        addAltTextGeneratorControl
    );

    console.log('[AI Alt Text] Gutenberg integration loaded');

})(window.wp, jQuery);