/* ========================================================================
   AI Alt Text Generator – admin/js/attachment.js
   Safe single-load, Woo product-page compatible, placeholder-alt aware
   ======================================================================== */

// --- One-time include guard (prevents "Identifier ... already declared") ---
if (window.__AIALT_ATTACHMENT_JS_LOADED__) {
    // Already loaded on this page; do nothing.
    // (Intentionally no exception — we just skip the second copy.)
    console.debug('[AI Alt Text] attachment.js already loaded; skipping');
  } else {
    window.__AIALT_ATTACHMENT_JS_LOADED__ = true;
  
    (function ($) {
      'use strict';
  
      // --- One-time init guard (defensive; protects against late re-includes) ---
      if (window.__AIALT_ATTACHMENT_INIT_DONE__) {
        console.debug('[AI Alt Text] init already performed; skipping');
        return;
      }
      window.__AIALT_ATTACHMENT_INIT_DONE__ = true;
  
      console.log('[AI Alt Text] attachment.js initialized');

      function aialttextShowTopUpModal(url) {
        url = url || (window.aialttext_token_ajax && aialttext_token_ajax.ajax_url
          ? aialttext_token_ajax.ajax_url.replace('admin-ajax.php', 'admin.php?page=aialttext-token')
          : 'admin.php?page=aialttext-token');
      
        // If the WP media modal is open, append inside it; otherwise append to <body>.
        var $container = jQuery('.media-modal:visible').last();
        var insideMediaModal = $container.length > 0;
        if (!insideMediaModal) $container = jQuery('body');
      
        // Backdrop style: absolute when inside media modal, fixed on the page otherwise.
        var backdropStyle = insideMediaModal
          ? 'position:absolute;inset:0;background:rgba(0,0,0,.5);z-index:200000;'
          : 'position:fixed;inset:0;background:rgba(0,0,0,.5);z-index:2000000;';
      
        var html =
          '<div class="aialttext-modal-backdrop" style="'+ backdropStyle +'">' +
            '<div class="aialttext-modal" style="max-width:520px;margin:10% auto;background:#fff;border-radius:8px;padding:20px;box-shadow:0 10px 30px rgba(0,0,0,.2);z-index:inherit;">' +
              '<h2 style="margin:0 0 8px;">Out of tokens</h2>' +
              '<p>You have no tokens remaining, so AI alt text could not be generated.</p>' +
              '<div style="margin-top:16px;display:flex;gap:8px;justify-content:flex-end;">' +
                '<a href="#" class="button" id="aialttext-close-modal">Close</a>' +
                '<a href="'+ url +'" class="button button-primary">Top up tokens</a>' +
              '</div>' +
            '</div>' +
          '</div>';
      
        var $el = jQuery(html).appendTo($container);
      
        // Close on outside click or ESC
        $el.on('click', function (e) { if (e.target === this) jQuery(this).remove(); });
        jQuery('#aialttext-close-modal').on('click', function (e) { e.preventDefault(); $el.remove(); });
        jQuery(document).on('keydown.aialttextModal', function (e) {
          if (e.key === 'Escape') { $el.remove(); jQuery(document).off('keydown.aialttextModal'); }
        });
      }
  
      // ---------------------------------------------------------------------
      // Helpers
      // ---------------------------------------------------------------------
  
      // Treat trivial alts (filename, title, generic words) as "empty"
      function aialt_isPlaceholderAlt(data) {
        try {
          const norm = (s) => (s || '')
            .toString()
            .toLowerCase()
            .trim()
            .replace(/\.[a-z0-9]+$/i, '')   // drop extension
            .replace(/[_\-]+/g, ' ')        // underscores/dashes -> space
            .replace(/\s+/g, ' ');
  
          const altN = norm(data.alt);
          if (!altN) return true;
  
          // (1) Equals title?
          const titleN = norm(data.title);
          if (titleN && altN === titleN) return true;
  
          // (2) Equals filename?
          let filename = data.filename || '';
          if (!filename && data.url) {
            try { filename = data.url.split('/').pop(); } catch (e) {}
          }
          const fileN = norm(filename);
          if (fileN && altN === fileN) return true;
  
          // (3) Generic placeholders
          const placeholders = [
            'image', 'photo', 'picture',
            'product image', 'product photo',
            'woocommerce product image', 'wp image', 'default image'
          ].map(norm);
          if (placeholders.includes(altN)) return true;
  
          return false;
        } catch (e) {
          return false;
        }
      }
  
      // Find any visible Alt Text field in the media/details UI
function findAltTextField() {
  return (
    document.getElementById('attachment-details-two-column-alt-text') ||
    document.querySelector('.attachment-details [data-setting="alt"] input, .attachment-details [data-setting="alt"] textarea') ||
    document.querySelector('textarea[aria-describedby="alt-text-description"]') ||
    null
  );
}

/**
 * Resolve the attachment ID that is *currently* shown/selected in UI.
 * Works for:
 *  - Media modal (Backbone) selection
 *  - Two-column "Attachment details" screen (upload.php?item=ID)
 *  - Fallback to selected grid item with data-id
 */
function getActiveAttachmentIdFromUI() {
  // 1) Media modal selection (most reliable when modal is open)
  try {
    if (window.wp && wp.media && wp.media.frame) {
      const state = wp.media.frame.state && wp.media.frame.state();
      const sel   = state && state.get && state.get('selection');
      const att   = sel && sel.first && sel.first();
      const id    = att && (att.get ? att.get('id') : att.id);
      if (id) return parseInt(id, 10);
    }
  } catch (e) {}

  // 2) Two-column "Attachment details" form inputs
  const idFromForm = document.querySelector('input#attachment-id, input[name="attachment_id"], input#post_ID');
  if (idFromForm && idFromForm.value) return parseInt(idFromForm.value, 10);

  // 3) Fallback: any selected item carrying data-id
  const selected = document.querySelector('.media-modal .attachment.selected, .attachments .attachment.selected, [data-id].attachment.selected');
  if (selected && selected.getAttribute('data-id')) {
    return parseInt(selected.getAttribute('data-id'), 10);
  }

  return 0;
}
  
      // Update all likely Alt Text inputs + media model, with UI feedback if present
      function updateAllAltTextFields(altText) {
        if (!altText) return false;
        let updated = false;
  
        const apply = (el) => {
          if (!el) return;
          el.value = altText;
          $(el).val(altText).trigger('input').trigger('change');
          $(el).addClass('field-highlight');
          setTimeout(() => $(el).removeClass('field-highlight'), 1200);
          updated = true;
        };
  
        apply(document.getElementById('attachment-details-two-column-alt-text'));
  
        const ds = document.querySelector('.attachment-details [data-setting="alt"] input, .attachment-details [data-setting="alt"] textarea');
        if (ds) apply(ds);
  
        const aria = document.querySelector('textarea[aria-describedby="alt-text-description"]');
        if (aria) apply(aria);
  
        // Update the in-memory media model if present
        try {
          if (window.wp && wp.media && wp.media.frame) {
            const selection = wp.media.frame.state().get('selection');
            const att = selection && selection.first && selection.first();
            if (att) { att.set('alt', altText); updated = true; }
          }
        } catch (e) {
          console.debug('[AI Alt Text] media frame update failed', e);
        }
  
        return updated;
      }
  
      // Determine the editor context post/product ID (classic + Gutenberg + wc-admin)
      function getContextPostId() {
        let id = 0;
      
        // 0) Localized by PHP (new)
        if (window.aialttext_token_attachment && window.aialttext_token_attachment.current_post_id) {
          id = parseInt(window.aialttext_token_attachment.current_post_id, 10) || 0;
        }
      
        // 1) Global injected by PHP (legacy)
        if (!id && window.aialttext_current_post_id) {
          id = parseInt(window.aialttext_current_post_id, 10) || 0;
        }
      
        // 2) WooCommerce wc-admin (?page=wc-admin&path=/product/123[/edit])
        if (!id && window.location.search.indexOf('page=wc-admin') !== -1) {
          const p = new URLSearchParams(window.location.search).get('path') || '';
          const decoded = decodeURIComponent(p);
          const m = decoded.match(/\/product\/(\d+)/);
          if (m) id = parseInt(m[1], 10) || 0;
        }
      
        // 3) wc-admin hash route (#/product/123[/edit])
        if (!id && window.location.hash) {
          const dm = decodeURIComponent(window.location.hash).match(/\/product\/(\d+)/);
          if (dm) id = parseInt(dm[1], 10) || 0;
        }
      
        // 4) Classic/Gutenberg (?post=123)
        if (!id) {
          const m2 = window.location.search.match(/[?&]post=(\d+)/);
          if (m2) id = parseInt(m2[1], 10) || 0;
        }
      
        // 4b) Gutenberg editor store (new — works for products, too)
        try {
          if (!id && window.wp && wp.data && wp.data.select) {
            const sel = wp.data.select('core/editor');
            if (sel) {
              const maybe = sel.getCurrentPostId && sel.getCurrentPostId();
              if (maybe) id = parseInt(maybe, 10) || 0;
            }
          }
        } catch (e) {}
      
        // 5) Hidden form fallback
        if (!id) {
          const postField = document.querySelector('#post_ID, input[name="post"], input[name="post_id"]');
          if (postField && postField.value) {
            id = parseInt(postField.value, 10) || 0;
          }
        }
      
        return id || 0;
      }
      
      function buildInlineContext(ctxPostId) {
        try {
          const norm = (s) => (s || "").toString().trim().replace(/\s+/g, " ");
          let title = "";
          let shortDesc = "";
          let contentDesc = "";
      
          // Gutenberg editor attributes if present
          try {
            if (window.wp && wp.data && wp.data.select) {
              const sel = wp.data.select('core/editor');
              if (sel) {
                title = norm(sel.getEditedPostAttribute && sel.getEditedPostAttribute('title'));
                shortDesc = norm(sel.getEditedPostAttribute && sel.getEditedPostAttribute('excerpt'));
                contentDesc = norm(sel.getEditedPostAttribute && sel.getEditedPostAttribute('content'));
              }
            }
          } catch (e) {}
      
          // DOM fallbacks (classic/product screens)
          if (!title) {
            const t = document.querySelector('#title, input#title, input[name="post_title"]');
            if (t && t.value) title = norm(t.value);
          }
      
          const shortCandidates = [
            'textarea#excerpt',
            'textarea[name="_short_description"]',
            'textarea[name="product_short_description"]',
            'div#wp-excerpt-wrap textarea.wp-editor-area'
          ];
          for (const sel of shortCandidates) {
            if (shortDesc) break;
            const el = document.querySelector(sel);
            if (el && el.value) shortDesc = norm(el.value);
          }
      
          if (!shortDesc) {
            const c = document.querySelector('textarea#content, div#wp-content-wrap textarea.wp-editor-area');
            if (c && c.value) contentDesc = norm(c.value);
          }
      
          const desc = norm(shortDesc || contentDesc).slice(0, 500);
          const safeTitle = title || (ctxPostId ? `Post #${ctxPostId}` : '');
      
          let ctx = '';
          if (safeTitle) {
            ctx += `This image appears on: "${safeTitle}". `;
            ctx += `PRODUCT_TITLE: ${safeTitle}. `;
          }
          if (desc) {
            ctx += `PRODUCT_DESC: ${desc}. `;
          }
      
          // Optional: bring in visible Yoast fields if present (no keyword stuffing)
          const yTitle = document.querySelector('#yoast_wpseo_title');
          const yMeta  = document.querySelector('#yoast_wpseo_metadesc');
          if (yTitle && yTitle.value) ctx += `SEO_TITLE: ${norm(yTitle.value)}. `;
          if (yMeta && yMeta.value)   ctx += `SEO_META: ${norm(yMeta.value)}. `;
          const yKey = document.querySelector('#yoast_wpseo_focuskw');
          if (!yKey || !yKey.value) ctx += 'INSTRUCTION: No SEO_KEYPHRASE; use SEO_TITLE and SEO_META as context only—avoid keyword stuffing.';
      
          return ctx.trim();
        } catch (e) {
          return '';
        }
      }
      
  
      /**
 * Core generator
 * @param {jQuery|null} $button  Button that was clicked, or null for silent auto-gen
 * @param {number}      imageId  Attachment ID
 * @param {number}      contextPostId  Post/Product context (optional; resolved if falsy)
 * @param {{silent?:boolean}=}  opts
 */
function handleGenerateAltText($button, imageId, contextPostId, opts) {
    opts = opts || {};
    const silent = !!opts.silent;

    // Resolve context if not provided
    if (!contextPostId) {
        contextPostId = getContextPostId();
    }

    // === DEBUG: trace context resolution and Woo/Product detection ===
try {
  console.group('[AI Alt Text] Generate request');
  console.info('[AI Alt Text] imageId:', imageId);
  console.info('[AI Alt Text] contextPostId (pre):', contextPostId);

  // Where are we?
  const isWcAdmin = window.location.search.indexOf('page=wc-admin') !== -1;
  const hashHasProduct = !!(window.location.hash && decodeURIComponent(window.location.hash).match(/\/product\/(\d+)/));
  console.info('[AI Alt Text] URL hints:', { isWcAdmin, hashHasProduct, href: window.location.href });

} catch (e) { /* no-op */ }


    if (!imageId || isNaN(imageId)) {
        if (!silent) alert('Could not determine the image ID.');
        console.warn('[AI Alt Text] Missing/invalid imageId');
        return;
    }

    // UI handles (only when a button triggered it)
    let $spinner = jQuery(), $result = jQuery();
    if ($button && $button.length) {
        $spinner = $button.closest('.field, .misc-pub-section, .wrap').find('.spinner, .aialttext_token_spinner').first();
        $result  = $button.closest('.field, .misc-pub-section, .wrap').find('.aialttext_token_result_container').first();
        $spinner.addClass('is-active');
        $button.prop('disabled', true).text(
            (window.aialttext_token_attachment && window.aialttext_token_attachment.generating) || 'Generating...'
        );
        if ($result.length) $result.hide();
    }

    const ajaxUrl =
        (window.aialttext_token_attachment && window.aialttext_token_attachment.ajax_url) ||
        window.ajaxurl || '/wp-admin/admin-ajax.php';
    const nonce = window.aialttext_token_attachment ? window.aialttext_token_attachment.nonce : '';

// Build inline context from editor DOM and WP data stores
let inlineContext = buildInlineContext(contextPostId || getContextPostId());

// Optional: pull Woo/product context from server (brand, attributes, taxonomies) and merge when missing
// This is especially important in wc-admin where DOM fields aren't always accessible.
function needsWooAugment(ctx) {
  if (!ctx) return true;
  const s = ctx.toLowerCase();
  const hasTitle = s.indexOf('product_title:') !== -1;
  const hasDesc  = s.indexOf('product_desc:')  !== -1;
  const hasBrand = s.indexOf('product_brand:') !== -1;
  // If any of the canonical product markers are missing, we’ll augment
  return !(hasTitle && hasDesc && hasBrand);
}

const postIdForContext = contextPostId || getContextPostId();
if (postIdForContext && needsWooAugment(inlineContext)) {
  if (window.console) console.log('[AI Alt Text] Requesting server Woo context for post', postIdForContext);
  try {
    const ajaxUrl = (window.aialttext_token_ajax && aialttext_token_ajax.ajax_url) || window.ajaxurl;
    const nonce   = (window.aialttext_token_ajax && aialttext_token_ajax.nonce)  || null;

    // Synchronous flow via promise chaining: fetch Woo context first, then generate.
    // We'll do the fetch here and stash/merge the result before the generate call below.
    window.__AIALT_WOO_CONTEXT_PROMISE__ = jQuery.ajax({
      url: ajaxUrl,
      type: 'POST',
      dataType: 'json',
      data: {
        action: 'aialttext_token_get_product_context',
        nonce:  nonce,
        post_id: postIdForContext
      }
    }).then(function(res) {
      if (res && res.success && res.data && res.data.context) {
        console.info('[AI Alt Text] Server Woo context received (chars):', res.data.context.length);
        // Merge only missing parts; preserve any user/inline data already present
        inlineContext = mergeContextMarkers(inlineContext, res.data.context);
      } else {
        console.warn('[AI Alt Text] No Woo context returned or missing payload', res);
      }
    }).catch(function(err){
      console.warn('[AI Alt Text] Woo context fetch failed', err);
    });
  } catch (e) {
    console.warn('[AI Alt Text] Exception requesting Woo context', e);
  }
} else {
  // Nothing to fetch; normalize the promise to keep the flow unified below.
  window.__AIALT_WOO_CONTEXT_PROMISE__ = Promise.resolve();
}

// Merge helper: only graft PRODUCT_* markers that are missing in the left-hand context
// Also tolerates Woo contexts that use looser labels like "Brand: Acme".
function mergeContextMarkers(leftCtx, rightCtx) {
  try {
    const L = (leftCtx  || '').toString();
    const R = (rightCtx || '').toString();

    // Helpful debug: show a sample of the raw Woo context
    if (R && window.console) {
      console.groupCollapsed('[AI Alt Text] mergeContextMarkers – Woo ctx (first 300 chars)');
      console.log(R.slice(0, 300));
      console.groupEnd();
    }

    const Llow = L.toLowerCase();
    const needTitle = Llow.indexOf('product_title:') === -1;
    const needDesc  = Llow.indexOf('product_desc:')  === -1;
    const needBrand = Llow.indexOf('product_brand:') === -1;

    // If we already have all canonical markers and nothing extra from Woo, just return L
    if (!R || !(needTitle || needDesc || needBrand)) {
      return L || R;
    }

    let grafts = [];

    // Exact-label helper (PRODUCT_TITLE / PRODUCT_DESC / PRODUCT_BRAND)
    const takeExact = (label) => {
      const rx = new RegExp('(^|\\n)'+label+'\\s*:\\s*([^\\n]+)', 'i');
      const m = R.match(rx);
      if (m && m[2]) {
        grafts.push(label + ': ' + m[2].trim());
      }
    };

    if (needTitle) takeExact('PRODUCT_TITLE');
    if (needDesc)  takeExact('PRODUCT_DESC');

    if (needBrand) {
      // 1) Try strict PRODUCT_BRAND first
      takeExact('PRODUCT_BRAND');

      // 2) If not found, look for a generic "Brand: Something" style line
      const alreadyHasBrand = grafts.some(function (g) {
        return g.indexOf('PRODUCT_BRAND:') === 0;
      });
      if (!alreadyHasBrand) {
        const mBrand = R.match(/(^|\n)[^\n]*brand[^\n:]*:\s*([^\n]+)/i);
        if (mBrand && mBrand[2]) {
          grafts.push('PRODUCT_BRAND: ' + mBrand[2].trim());
        }
      }
    }

    // 3) As a final fallback, if we still didn't graft anything but Woo gave us context,
    //    inject a compact generic PRODUCT_CONTEXT so the model still sees Woo details.
    if (!grafts.length && R) {
      const snippet = R.replace(/\s+/g, ' ').trim().slice(0, 400);
      grafts.push('PRODUCT_CONTEXT: ' + snippet);
    }

    const merged = (L + '\n' + grafts.join('\n')).trim();
    if (window.console) {
      console.debug('[AI Alt Text] Context merged. Before chars:', L.length, ' After chars:', merged.length);
    }
    return merged;
  } catch (e) {
    console.warn('[AI Alt Text] mergeContextMarkers error', e);
    return leftCtx || rightCtx || '';
  }
}



// Wait for possible Woo context augmentation first, then send the generate request
var wooPromise = window.__AIALT_WOO_CONTEXT_PROMISE__ || Promise.resolve();

wooPromise.then(function () {
  console.groupCollapsed('[AI Alt Text] Final generate payload');
  console.log('post_id:', (contextPostId || 0));
  console.log('inlineContext (chars):', (inlineContext || '').length);
  console.groupEnd();

  jQuery.ajax({
    url: ajaxUrl,
    type: 'POST',
    dataType: 'json',
    data: {
      action: 'aialttext_token_generate_single',
      nonce:  nonce,
      attachment_id: imageId,
      post_id: contextPostId || 0,
      context: inlineContext || ''
    }
  })
  .done(function (response) {

    // Log debug information if present
    if (response && response.data && response.data.debug_logs) {
        console.group('%c[AI Alt Text] Debug Logs', 'color: #0066cc; font-weight: bold');
        response.data.debug_logs.forEach(function(log) {
            console.log('%c' + log.time + '%c ' + log.message, 
                'color: #666', 
                'color: #000',
                log.data || '');
            if (log.data) {
                console.log(log.data);
            }
        });
        console.groupEnd();
    }

    // Also surface which model/provider actually served the request
if (response && response.data && response.data.model) {
  console.info('[AI Alt Text] Used model:', response.data.model);
}

if (!response || !response.success) {
  // Token-exhausted UX stays the same
  if (response && response.data && response.data.insufficient_tokens) {
    aialttextShowTopUpModal(response.data.top_up_url);
    return;
  }

  var help = (window.aialttext_token_attachment && window.aialttext_token_attachment.help_url)
    || (ajaxUrl || '').replace('admin-ajax.php', 'admin.php?page=aialttext-token');

  // Default technical message used for generic errors
  var defaultMsg = 'There was a technical issue generating alt text. Plea...nt and try again or contact support@arcticfoxdevelopments.com.';
  var msg;

  // Special-case unsupported image file types
  if (response && response.data && response.data.unsupported_type) {
    msg = 'There was a technical issue generating alt text. Img Alt Gen Pro supports: PNG, JPG, webP, and GIF.';
  } else {
    msg = defaultMsg;
    if (response && response.data && response.data.message) {
      msg += ' (' + response.data.message + ')';
    }
  }

  if (!silent) {
    var html = '<div class="notice notice-error inline" style="padding:8px;margin:5px 0;">' +
               '<p><span style="color:#d63638;">⚠</span> ' + msg + '</p>' +
               '<p><a class="button" href="'+ help +'" target="_blank" rel="noopener">Open plugin help</a></p>' +
               '</div>';
    if ($result.length) { $result.html(html).show(); } else { alert(msg); }
  }

  if (window.console && console.error) console.error('[AI Alt Text] Technical issue (single):', response);
  return;
}

        const altText = response.data.alt_text || response.data.alt || '';

        // 2) Persist alt to DB
        jQuery.ajax({
            url: ajaxUrl,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'aialttext_token_force_update_and_get_alt',
                nonce: nonce,
                image_id: imageId,
                alt_text: altText
            }
        }).done(function (dbRes) {
            if (!dbRes || !dbRes.success) {
                const msg = 'Error saving alt text to the database.';
                if (!silent) {
                    const html = '<div class="notice notice-error inline" style="padding:8px;margin:5px 0;">' +
                                 '<p><span style="color:#d63638;">⚠</span> ' + msg + '</p></div>';
                    if ($result.length) { $result.html(html).show(); } else { alert(msg); }
                }
                return;
            }

            // 3) Update visible fields / media model, but only if the same attachment is still active
const activeIdNow = getActiveAttachmentIdFromUI();
if (activeIdNow && parseInt(activeIdNow, 10) !== parseInt(imageId, 10)) {
  // We navigated to a different image while the request was in-flight.
  // The alt text has been saved to the correct attachment in the DB; skip UI mutation here.
  console.debug('[AI Alt Text] Skipping UI update; request was for #' + imageId + ' but UI shows #' + activeIdNow + '. Value saved to DB.');
} else {
  const updated = updateAllAltTextFields(altText);
  // Some screens re-render immediately after updates; retry once if the first write didn’t “stick”.
  setTimeout(function(){ if (!updated) updateAllAltTextFields(altText); }, 400);
}

            if (!silent) {
                let html = '<div class="notice notice-success inline" style="padding:8px;margin:5px 0;">' +
                           '<p><span style="color:#46b450;font-weight:700;">✓</span> Alt text generated and saved!</p>' +
                           '<p style="margin:4px 0 0;font-size:12px;">Value: "' + altText.replace(/"/g,'&quot;') + '"</p>';
                if (contextPostId) {
                    html += '<p style="margin:2px 0 0;font-size:11px;color:#0073aa;">🎯 Context: Post/Product #' + contextPostId + '</p>';
                }
                html += '</div>';
                if ($result.length) $result.html(html).show();
            }
        }).fail(function (xhr) {
            if (!silent) {
              if (xhr && xhr.status === 402) {
                aialttextShowTopUpModal();
                return;
            }
            const msg = (window.aialttext_token_attachment && window.aialttext_token_attachment.error) || 'Error generating alt text.';
            const html = '<div class="notice notice-error inline" style="padding:8px;margin:5px 0;">' +
                         '<p><span style="color:#d63638;">⚠</span> ' + msg + '</p></div>';
            if ($result.length) { $result.html(html).show(); } else { alert(msg); }
            }
        });
    }).fail(function (xhr) {
      if (xhr && xhr.status === 402) {
        aialttextShowTopUpModal();
        return;
      }
      if (!silent) {
        var help = (window.aialttext_token_attachment && window.aialttext_token_attachment.help_url)
          || (ajaxUrl || '').replace('admin-ajax.php', 'admin.php?page=aialttext-token');
        var msg = 'There was a technical issue. Please wait a moment and try again or contact support@arcticfoxdevelopments.com.';
        var html = '<div class="notice notice-error inline" style="padding:8px;margin:5px 0;">' +
                   '<p><span style="color:#d63638;">⚠</span> ' + msg + '</p>' +
                   '<p><a class="button" href="'+ help +'" target="_blank" rel="noopener">Open plugin help</a></p>' +
                   '</div>';
        if ($result.length) { $result.html(html).show(); } else { alert(msg); }
      }
      if (window.console && console.error) console.error('[AI Alt Text] Technical error (XHR):', xhr);
    }) .always(function () {
        if ($button && $button.length) {
            $button.prop('disabled', false).text('Generate AI Alt Text');
            if ($spinner.length) $spinner.removeClass('is-active');
        }
    });
  });  // Close wooPromise.then()
}  // Close handleGenerateAltText function

  
      // ---------------------------------------------------------------------
      // UI binding (explicit “Generate” buttons)
      // ---------------------------------------------------------------------
      function bindSingleGenerateButtons(root) {
        const $scope = root ? $(root) : $(document);
        const $buttons = $scope
          .find('.aialttext-token-generate, #aialttext-token-generate-single, #aialttext_generate_single, .aialttext_token_generate_single, .aialttext_generate_single')
          .filter(':not([data-aialttext-bound="1"])');
  
        $buttons.each(function () {
          const $btn = $(this);
          $btn.attr('data-aialttext-bound', '1');
  
          const handler = function (e) {
            e.preventDefault();
            e.stopPropagation();
            if (e.stopImmediatePropagation) e.stopImmediatePropagation();
  
            let imageId = parseInt($btn.data('attachment-id') || $btn.data('image-id'), 10);
  
            // Fallback: current media selection
            if (!imageId && window.wp && wp.media && wp.media.frame) {
              try {
                const sel = wp.media.frame.state().get('selection');
                const att = sel && sel.first();
                if (att) imageId = parseInt(att.get('id'), 10);
              } catch (err) {
                console.debug('[AI Alt Text] selection lookup failed', err);
              }
            }
            console.log('[AI Alt Text] Click → Generate for imageId', imageId, 'post context', getContextPostId());
  
            handleGenerateAltText($btn, imageId, getContextPostId(), { silent: false });
          };
  
          // Bubble-phase jQuery handler
          $btn.on('click.aialttext', handler);
          // Capture-phase native handler (survives other plugins stopping propagation)
          if ($btn[0] && $btn[0].addEventListener) {
            $btn[0].addEventListener('click', handler, true);
          }
        });
      }
  
      // ---------------------------------------------------------------------
      // Auto-generate when selecting images in the media modal (product image/gallery)
      // ---------------------------------------------------------------------
      const processed = new Set();
  
      function autoGenerateForAttachmentModel(attachment) {
        if (!attachment) return;
        const data = attachment.toJSON ? attachment.toJSON() : attachment;
        if (!data || data.type !== 'image') return;

        // Respect plugin setting: do not auto-generate if disabled
  try {
    if (!window.aialttext_token_attachment || !Number(window.aialttext_token_attachment.auto_generate)) {
      return;
    }
  } catch (e) { return; }
  
        const id = parseInt(data.id, 10) || 0;
        if (!id || processed.has(id)) return;
  
        // Skip only if the alt looks meaningful; otherwise auto-generate
        const hasUsableAlt = data.alt && !aialt_isPlaceholderAlt(data);
        if (hasUsableAlt) return;
  
        const ctxPostId = getContextPostId();
if (!ctxPostId) return;

processed.add(id);
try {
  handleGenerateAltText(null, id, ctxPostId, { silent: true });
} catch (e) {
  processed.delete(id); // allow retry on error
  console.error('[AI Alt Text] Auto-generate error:', e);
}

      }
  
      function hookFrame(frame) {
        if (!frame || frame._aialt_auto_hooked) return;
        frame._aialt_auto_hooked = true;
  
        // When user selects images in the media modal (e.g., product image/gallery)
        frame.on('select', function () {
          const selection = frame.state().get('selection');
          if (!selection || !selection.each) return;
          selection.each(autoGenerateForAttachmentModel);
        });
  
        // Extra safety if selection is modified one-by-one
        frame.on('library:selection:add', autoGenerateForAttachmentModel);
      }
  
      // Periodically hook media frames (covers classic, Gutenberg, and wc-admin)
      const frameInterval = setInterval(function () {
        if (!window.wp || !wp.media) return;
        if (wp.media.frame) hookFrame(wp.media.frame);
        if (wp.media.frames) {
          try {
            Object.keys(wp.media.frames).forEach(function (k) {
              const f = wp.media.frames[k];
              if (f && typeof f.on === 'function') hookFrame(f);
            });
          } catch (e) {}
        }
      }, 800);

      // Also re-bind buttons when the media modal opens/closes, because Woo admin
// and some themes recreate DOM trees on the fly.
(function observeMediaModal() {
  const root = document.body;
  const mo = new MutationObserver(function(muts){
    for (const m of muts) {
      if (!m.addedNodes || !m.addedNodes.length) continue;
      // If a media modal or attachment toolbar is inserted, (re)bind our buttons
      if ([].some.call(m.addedNodes, n =>
        (n.classList && (n.classList.contains('media-modal') || n.classList.contains('attachments-browser')))
      )) {
        try { bindSingleGenerateButtons(document); } catch(e){}
      }
    }
  });
  mo.observe(root, { childList: true, subtree: true });
})();

  
      // ---------------------------------------------------------------------
      // Bind buttons now and on DOM mutations (media modal is dynamic)
      // ---------------------------------------------------------------------
      bindSingleGenerateButtons(document);
  
      if (window.MutationObserver) {
        const binderObserver = new MutationObserver(function (mutations) {
          for (const m of mutations) {
            if (m.addedNodes) {
              for (const node of m.addedNodes) {
                if (node && node.nodeType === 1) bindSingleGenerateButtons(node);
              }
            }
          }
        });
        binderObserver.observe(document.body, { childList: true, subtree: true });
      }
  
      // Gutenberg sidebar legacy button (if present)
      $(document).on('click', '.aialttext-gutenberg-generate', function (e) {
        e.preventDefault();
        bindSingleGenerateButtons(); // ensure late-inserted buttons are bound
        const imageId = parseInt($(this).data('image-id'), 10);
        handleGenerateAltText($(this), imageId, getContextPostId(), { silent: false });
      });
  
    })(jQuery);
  }
  