/* global jQuery, aialttext_token_ajax, wp */
(function (wp, $) {
  'use strict';
  const { __, _n, sprintf } = wp.i18n;

  jQuery(function ($) {
    /* -------------------------------
     * Small utilities
     * ------------------------------- */

    // Make the helper available to later script blocks (without polluting globals with a generic name)
window.aialttextAjax = ajax;

    function ajax(action, data) {
      data = data || {};
      data.action = action;
      data.nonce  = aialttext_token_ajax.nonce;
    
      return $.ajax({
        url: aialttext_token_ajax.ajax_url,
        type: "POST",
        data: data,
        dataType: "json",          // force JSON; non-JSON will go to .fail()
        timeout: 30000             // 30s cap so buttons can’t sit forever
      });
    }

    /* -------------------------------
 * Delete account (modal + AJAX)
 * ------------------------------- */
var $delBtn  = $('#aialttext_token_delete_account_btn');
var $modal   = $('#aialttext-delete-modal');
var $ack     = $('#aialttext_delete_ack');
var $cancel  = $('#aialttext_delete_cancel');
var $confirm = $('#aialttext_delete_confirm');

if ($delBtn.length) {
  $delBtn.on('click', function (e) {
    e.preventDefault();
    $modal.show();
  });

  $cancel.on('click', function (e) {
    e.preventDefault();
    $modal.hide();
  });

  $ack.on('change', function () {
    $confirm.prop('disabled', !this.checked);
  });

  $confirm.on('click', function (e) {
    e.preventDefault();
    if ($confirm.is(':disabled')) return;

    $confirm.prop('disabled', true);

    // Reuse the localized nonce + ajax_url via the existing helper
    ajax('aialttext_token_delete_account', {})
      .done(function (resp) {
        if (resp && resp.success) {
          // Show the standard success notice the page already supports
          window.location = aialttext_token_ajax.help_url + '&loggedout=1';
        } else {
          alert((resp && resp.data && resp.data.message) ? resp.data.message : 'Deletion failed.');
          $confirm.prop('disabled', false);
          $modal.hide();
        }
      })
      .fail(function (xhr) {
        var msg = (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message)
          ? xhr.responseJSON.data.message
          : 'Deletion failed.';
        alert(msg);
        $confirm.prop('disabled', false);
        $modal.hide();
      });
  });
}

    /* Keep role="switch" aria-checked in sync for accessibility */
$('.aialttext-switch input[type="checkbox"]').each(function () {
  $(this)
    .attr('role', 'switch')
    .attr('aria-checked', this.checked ? 'true' : 'false')
    .on('change', function () {
      $(this).attr('aria-checked', this.checked ? 'true' : 'false');
    });
});

    // --- NEW: limit helpers + dynamic button label ---
function aialttext_getUserLimit() {
  var v = parseInt(jQuery("#aialttext_token_limit").val(), 10);
  return (isNaN(v) || v <= 0) ? 0 : v;
}

function aialttext_processButtonLabel() {
  var n = aialttext_getUserLimit();
  return n > 0
  ? sprintf( __('Process %d images', 'aialttext-token'), n )
  : __('Process all available images', 'aialttext-token');
}

// NEW: live updates when the user types or changes the field
jQuery(document).on("input change", "#aialttext_token_limit", function () {
  aialttext_applyLimitFromInput();
});

// NEW: ensure the latest input is honored right before we start processing
jQuery(document).on("click", "#aialttext_token_process_button", function () {
  aialttext_applyLimitFromInput();
});

// When the "Only images on published posts/pages" slider is toggled,
// immediately re-run the scan so the X-count updates without extra clicks.
jQuery(document).on("change", "#aialttext_only_published", (function () {
  let debounceTimer = null;
  return function () {
    // Never change filters mid-run
    if (window.aialttext_processingActive === true) {
      // Optionally flip back visually if you prefer:
      // const checked = jQuery(this).is(":checked");
      // jQuery(this).prop("checked", !checked);
      return;
    }

    jQuery("#aialttext_token_process_button").prop("disabled", true);
    // Adjust batch size dynamically with the mode
    batchSize = 1; // Force sequential updates for clearer UX when filtering to published content

    // Visual hint that we’re rescanning (if label exists).
    var $label = jQuery("#aialttext_token_progress_label, .aialttext-progress-label").first();
    if ($label.length) {
      $label.text(wp.i18n.__("Rescanning…", "aialttext-token"));
    }

    // Reset the progress bar UI only (state will be re-initialized by scan).
    jQuery(".aialttext-progress").css("width", "0%");

    // Debounce to avoid multiple back-to-back rescans when users flick the toggle
    clearTimeout(debounceTimer);
    debounceTimer = setTimeout(function () {
      if (typeof scanForImages === "function" && window.aialttext_processingActive !== true) {
        scanForImages();
      }
    }, 250);
  };
})());

/**
 * Apply the current input value to maxImages and adjust totals.
 * - Keeps totals stable while processing
 * - Clamps to the number of available images
 */
function aialttext_applyLimitFromInput() {
  if (typeof maxImages === "undefined") return;

  // Only adjust cap when not actively processing to avoid flicker
  if (!processing) {
    var userLimit = aialttext_getUserLimit();

    // Update run cap
    maxImages = userLimit; // 0 means "no cap" (process all)

    // If we've scanned, we can safely clamp the display totals
    if (scannedOnce) {
      var available = initialTotalImages || totalImages || 0;
      if (userLimit > 0) {
        totalImages = Math.min(available, userLimit);
      } else {
        totalImages = available; // show all available when no cap
      }
      updateStatsDisplay && updateStatsDisplay();
    }
  }

  // Always refresh the button label
  jQuery("#aialttext_token_process_button").text(aialttext_processButtonLabel());
}

    // Add: formatting helpers
function withCommas(n) {
  try { return Number(n).toLocaleString('en-US'); } catch(e) { return n; }
}
function perImageText(pkg) {
  var per = Number(pkg.price) / Number(pkg.tokens);        // dollars per image
  var dollars = per.toFixed(3).replace(/\.?0+$/, '');      // e.g., 0.019 → "0.019", 0.018 → "0.018"
  var cents   = (per * 100).toFixed(1).replace(/\.0$/, ''); // e.g., 1.9
  return '$' + dollars + ' per image (' + cents + '¢)';
}

    // First-run onboarding: highlight the Link/Relink button and focus the email
(function () {
  var cfg = window.aialttext_token_ajax || {};
  // force numeric interpretation; avoid "0" (string) being truthy
  var firstRun = parseInt(cfg.first_run, 10) === 1;

  var $email = jQuery('#aialttext_token_account_email');
  var $btn   = jQuery('#aialttext_token_link_account_btn');

  if ($btn.length) {
    if (firstRun) {
      try { console.log('[IMG Alt Gen Pro] First run detected. Guiding user to link account…'); } catch (e) {}
      if ($email.length) { $email.trigger('focus'); }
      jQuery('html, body').animate({ scrollTop: $btn.offset().top - 120 }, 300);
      $btn.addClass('aialttext-onboard-pulse');
    } else {
      // ensure any leftover pulse is removed if already linked
      $btn.removeClass('aialttext-onboard-pulse')
          .removeAttr('aria-describedby')
          .attr('data-linked', '1');
      try { console.log('[IMG Alt Gen Pro] Account appears linked; disabling onboarding pulse.'); } catch (e) {}
    }
  }
})();

// Optional: log success linking to DevTools (useful while debugging)
$(document).on('click', '#aialttext_token_link_account_btn', function () {
  try { console.log('[IMG Alt Gen Pro] Attempting to link account…'); } catch (e) {}
});

    // Intercept Media Library bulk action "Generate AI Alt Text (Tokens)"
jQuery(document).on('click', '#doaction, #doaction2', function (e) {
  var $form = jQuery(this).closest('form');
  if (!$form.length) return;

  var action = $form.find('select[name="action"]').val() || $form.find('select[name="action2"]').val();
  if (action !== 'generate_alt_text_token') return;

  e.preventDefault();
  ajax('aialttext_token_check_balance_ajax', {})  // nonce is added by ajax()
    .done(function (res) {
      if (res && res.success && res.data && parseInt(res.data.balance, 10) > 0) {
        // Proceed with the original bulk action
        $form.get(0).submit();
      } else {
        aialttextShowTopUpModal(res && res.data ? res.data.top_up_url : null);
      }
    })
    .fail(function (xhr) {
      if (xhr && xhr.status === 402) {
        aialttextShowTopUpModal();
        return;
      }
      var help = (aialttext_token_ajax && aialttext_token_ajax.help_url)
        ? aialttext_token_ajax.help_url
        : (aialttext_token_ajax.ajax_url || '').replace('admin-ajax.php','admin.php?page=aialttext-token');
    
      $("#aialttext_token_process_status").append(
        '<div class="notice notice-error">' +
          '<p><strong>We hit a technical issue while generating alt text.</strong> Please wait a moment and try again. If it keeps happening, email ' +
          '<a href="mailto:support@arcticfoxdevelopments.com">support@arcticfoxdevelopments.com</a>.</p>' +
          '<p><a class="button" href="'+ help +'" target="_blank" rel="noopener">Open plugin help</a></p>' +
        '</div>'
      );
    
      if (window.console && console.error) console.error('[AI Alt Text] Technical error during bulk processing:', xhr);
    });
});

    // ---- New: local buffer + renderer for processed thumbnails ----
    var aialttextImageBuffer = [];      // newest first
    var aialttextVisibleLimit = 20;     // default window (4 rows × 5)
    var aialttextMoreClicks = 0;        // first "Show more" = +20, subsequent = +15

    function aialttextShowTopUpModal(url) {
      url = url || (window.aialttext_token_ajax && aialttext_token_ajax.ajax_url
        ? aialttext_token_ajax.ajax_url.replace('admin-ajax.php', 'admin.php?page=aialttext-token')
        : 'admin.php?page=aialttext-token');
    
      // If the WP media modal is open, append inside it; otherwise append to <body>.
      var $container = jQuery('.media-modal:visible').last();
      var insideMediaModal = $container.length > 0;
      if (!insideMediaModal) $container = jQuery('body');
    
      // Backdrop style: absolute when inside media modal, fixed on the page otherwise.
      var backdropStyle = insideMediaModal
        ? 'position:absolute;inset:0;background:rgba(0,0,0,.5);z-index:200000;'
        : 'position:fixed;inset:0;background:rgba(0,0,0,.5);z-index:2000000;';
    
      var html =
        '<div class="aialttext-modal-backdrop" style="'+ backdropStyle +'">' +
          '<div class="aialttext-modal" style="max-width:520px;margin:10% auto;background:#fff;border-radius:8px;padding:20px;box-shadow:0 10px 30px rgba(0,0,0,.2);z-index:inherit;">' +
          '<h2 style="margin:0 0 8px;">' + __('Out of tokens', 'aialttext-token') + '</h2>' +
          '<p>' + __('You have no tokens remaining, so AI alt text could not be generated.', 'aialttext-token') + '</p>' +
            '<div style="margin-top:16px;display:flex;gap:8px;justify-content:flex-end;">' +
            '<a href="#" class="button" id="aialttext-close-modal">' + __('Close', 'aialttext-token') + '</a>' +
            '<a href="'+ url +'" class="button button-primary">' + __('Top up tokens', 'aialttext-token') + '</a>' +
            '</div>' +
          '</div>' +
        '</div>';
    
      var $el = jQuery(html).appendTo($container);
    
      // Close on outside click or ESC
      $el.on('click', function (e) { if (e.target === this) jQuery(this).remove(); });
      jQuery('#aialttext-close-modal').on('click', function (e) { e.preventDefault(); $el.remove(); });
      jQuery(document).on('keydown.aialttextModal', function (e) {
        if (e.key === 'Escape') { $el.remove(); jQuery(document).off('keydown.aialttextModal'); }
      });
    }

function aialttextEditHref(id) {
  if (aialttext_token_ajax && aialttext_token_ajax.upload_item_url) {
    return aialttext_token_ajax.upload_item_url + id;                 // upload.php?item=ID
  }
  if (aialttext_token_ajax && aialttext_token_ajax.post_edit_base) {
    return aialttext_token_ajax.post_edit_base + '?post=' + id + '&action=edit';
  }
  return '#';
}

function renderProcessedGrid() {
  var $panel = $("#aialttext_token_latest_images");
  var $grid  = $("#aialttext_images_grid");
  if (!$grid.length) return;

  $panel.show();
  $grid.empty();

  var visible = aialttextImageBuffer.slice(0, aialttextVisibleLimit);
  visible.forEach(function (image) {
    var filename   = image.url ? image.url.split('/').pop() : 'Unknown';
    var statusText = image.status === 'success' ? 'Generated'
                    : image.status === 'error'   ? 'Failed'
                    : __('Processing…', 'aialttext-token');

    var html = '' +
      '<div class="aialttext-image-item ' + image.status + '" data-id="' + image.id + '">' +
        '<a href="' + aialttextEditHref(image.id) + '" target="_blank" rel="noopener noreferrer">' +
          '<img src="' + (image.url || '') + '" alt="' + (image.title || '') + '">' +
        '</a>' +
        '<div class="aialttext-image-filename">' + filename + '</div>' +
        '<div class="aialttext-image-status ' + image.status + '">' + statusText + '</div>' +
        '<div class="aialttext-image-alt">' + (image.alt || __('Generating…', 'aialttext-token')) + '</div>' +
      '</div>';

    $grid.append(html);
  });

  var total = aialttextImageBuffer.length;
  var hasHidden = total > aialttextVisibleLimit;

  $("#aialttext_images_controls").toggle(total > 20);
  $("#aialttext_show_more").toggle(hasHidden);
  $("#aialttext_show_all").toggle(hasHidden);
}

// Button handlers
$(document).on('click', '#aialttext_show_more', function (e) {
  e.preventDefault();
  // Always reveal the next 20 thumbnails.
  var inc = 20;

  // Do not clamp to the current buffer length — this lets the grid
  // auto-grow up to the newly requested limit as more images finish.
  aialttextVisibleLimit = aialttextVisibleLimit + inc;

  aialttextMoreClicks++;
  renderProcessedGrid();
});

$(document).on('click', '#aialttext_show_all', function (e) {
  e.preventDefault();
  // Use Infinity so the grid continues to show every new image as it completes.
  aialttextVisibleLimit = Infinity;
  aialttextMoreClicks = 0; // reset for next run
  renderProcessedGrid();
});
  
    function param(name) {
      var match = new RegExp("[?&]" + name + "=([^&#]*)").exec(location.search);
      return match ? decodeURIComponent(match[1].replace(/\+/g, " ")) : null;
    }
  
    function banner(kind, text) {
      var cls =
        kind === "success"
          ? "notice-success"
          : kind === "warn"
          ? "notice-warning"
          : "notice-error";
      $('<div class="notice ' + cls + ' is-dismissible"><p>' + text + "</p></div>").insertBefore(".wrap h1");
    }

    // Make the helper available to later script blocks
window.aialttextBanner = banner;


    // --- Lightweight purchase modal (popup) ---
(function(){
    if (!document.getElementById("aialttext-purchase-modal-style")) {
      var css = `
        .aialttext-modal-backdrop{position:fixed;inset:0;background:rgba(0,0,0,.35);display:flex;align-items:center;justify-content:center;z-index:100000;}
        .aialttext-modal{background:#fff;min-width:420px;max-width:520px;border-radius:12px;box-shadow:0 12px 30px rgba(0,0,0,.2);overflow:hidden}
        .aialttext-modal header{padding:16px 20px;border-bottom:1px solid #eee;font-weight:600;font-size:16px}
        .aialttext-modal .body{padding:18px 20px;font-size:14px;line-height:1.5}
        .aialttext-modal .muted{color:#555;margin-top:4px}
        .aialttext-modal footer{padding:12px 20px;border-top:1px solid #eee;text-align:right}
        .aialttext-modal .btn{background:#6740C5;color:#fff;border:0;border-radius:6px;padding:8px 14px;cursor:pointer}
        .aialttext-spinner{display:inline-block;width:16px;height:16px;border-radius:50%;border:2px solid #ccc;border-top-color:#6740C5;animation:aialttext-spin 0.9s linear infinite;margin-right:8px;vertical-align:-2px}
        @keyframes aialttext-spin{to{transform:rotate(360deg)}}
      `;
      var style = document.createElement("style");
      style.id = "aialttext-purchase-modal-style";
      style.appendChild(document.createTextNode(css));
      document.head.appendChild(style);
    }
  })();
  
  function showPurchaseModal() {
    var html = '' +
      '<div class="aialttext-modal-backdrop" id="aialttext-purchase-backdrop">' +
        '<div class="aialttext-modal" role="dialog" aria-modal="true" aria-labelledby="aialttext-purchase-title">' +
          '<header id="aialttext-purchase-title">Thank you for your purchase!</header>' +
          '<div class="body">' +
             '<div class="muted"><span class="aialttext-spinner"></span><span class="aialttext-modal-message">We’re confirming your payment and updating your balance…</span></div>' +
          '</div>' +
          '<footer><button class="btn" id="aialttext-purchase-close" style="display:none">Close</button></footer>' +
        '</div>' +
      '</div>';
    jQuery("body").append(html);
  }
  
  function updatePurchaseModal(text, done) {
    jQuery(".aialttext-modal-message").text(text);
    if (done) {
      jQuery(".aialttext-spinner").remove();
      jQuery("#aialttext-purchase-close").show().off("click").on("click", function(){ closePurchaseModal(); });
    }
  }
  
  function closePurchaseModal() {
    jQuery("#aialttext-purchase-backdrop").remove();
  }
  
  
    /* -------------------------------
     * Pricing + account balance
     * ------------------------------- */
    function renderPricing(packages) {
      var $wrap = $("#aialttext_token_pricing").empty();
      if (window.console && console.log) console.log("[AI Alt Text] Pricing packages:", packages);
    
      $.each(packages, function (key, pkg) {
        // Localize known package names by key, falling back to server label
var nameMap = {
  starter:      __('Starter', 'aialttext-token'),
  basic:        __('Basic', 'aialttext-token'),
  standard:     __('Standard', 'aialttext-token'),
  professional: __('Professional', 'aialttext-token'),
  business:     __('Business S', 'aialttext-token'),
  enterprise:   __('Business M', 'aialttext-token'),
  ultimate:     __('Business L', 'aialttext-token'),
  business_xxl: __('Business XL', 'aialttext-token'),
  ultimate_25k: __('Ultimate', 'aialttext-token')
};
if (nameMap[key]) { pkg.name = nameMap[key]; }
        var badge = pkg.popular ? '<div class="badge">' + __('Most Popular', 'aialttext-token') + '</div>' : "";
        var html  = '<div class="aialttext-token-plan">';
        html     += '<div class="head"><h3>' + pkg.name + "</h3>" + badge + "</div>";
    
        if (pkg.contact_url) {
          // Non-Stripe: show a contact CTA instead of a purchase button/price
          html += '<div class="price">' + __('Contact us for enterprise pricing', 'aialttext-token') + '</div>';
html += '<a class="button button-secondary" href="' + pkg.contact_url + '" target="_blank" rel="noopener">' + __('Contact us', 'aialttext-token') + '</a>';
        } else {
          // Count + price
          html += '<div class="tokens">' + sprintf(
            _n('%s token', '%s tokens', pkg.tokens, 'aialttext-token'),
            withCommas(pkg.tokens)
          ) + '</div>';
          html += '<div class="price">$' + Number(pkg.price).toFixed(2) + '</div>';
        
          // Optional "Save X%" badge (unchanged)
          if (pkg.discount) html += '<div class="save">Save ' + pkg.discount + '%</div>';
        
          // Group per-image + button together so they sit at the bottom and the line is
          // guaranteed to appear immediately above the button.
          html += '<div class="cta">';
          html +=   '<div class="per-image">→ ' + perImageText(pkg) + '</div>';
          html +=   '<button class="button button-primary buy" data-id="' + key + '">Purchase</button>';
          html += '</div>';
        }
        $wrap.append(html);
      });
    }
  
    function fetchBalance() {
      ajax("aialttext_token_refresh_balance").done(function (res) {
        if (res && res.success && res.data) {
          $("#aialttext-token-balance").text(res.data.balance);
          // Update bulk processing token display
          var $rem = $("#aialttext_tokens_remaining");
          if ($rem.length) {
            $rem.text(res.data.balance);
            $rem.data("initial-balance", res.data.balance);
        
            // Apply threshold color on first paint
            $rem.removeClass("aialttext-warn aialttext-danger");
            if (res.data.balance < 10) {
              $rem.addClass("aialttext-danger");
            } else if (res.data.balance < 20) {
              $rem.addClass("aialttext-warn");
            }
          }
        }
      });
    }
  
    if ($("#aialttext_token_pricing").length) {
      ajax("aialttext_token_get_pricing").done(function (res) {
        if (res && res.success && res.data && res.data.packages) {
          renderPricing(res.data.packages);
        }
      });
    }
  
    $(document).on("click", "#aialttext_token_refresh_balance", function (e) {
      e.preventDefault();
      fetchBalance();
    });
  
    if ($("#aialttext-token-balance").length) {
      fetchBalance(); // load on page ready
    }
  
    $(document).on("click", "#aialttext_token_link_account_btn", function (e) {
      e.preventDefault();
    
      var $btn      = $("#aialttext_token_link_account_btn");
      var email     = $("#aialttext_token_account_email").val();
      var password  = $("#aialttext_token_account_password").val() || "";
      var consent   = $("#aialttext_token_consent").is(":checked");
    
      if (!email) {
        banner("error", __('Email required', 'aialttext-token'));
        return;
      }
      if (!password || password.length < 8) {
        banner("error", __('Password (min. 8 chars) required', 'aialttext-token'));
        return;
      }
      if (!consent) {
        banner("error", __('Please agree to the Terms and Privacy Policy to continue.', 'aialttext-token'));
        return;
      }
    
      $btn.prop("disabled", true).text(__('Checking…', 'aialttext-token'));
    
      ajax("aialttext_token_link_account", { email: email, password: password, consent: (consent ? 1 : 0) })
        .done(function (res) {
          if (res && res.success) {
            try { console.log('[IMG Alt Gen Pro] Account linked via AJAX.'); } catch (e) {}
            banner("success", __('Account linked successfully.', 'aialttext-token'));
            jQuery("#aialttext-token-balance").text((res.data && res.data.balance) || 0);
    
            jQuery('#aialttext_token_link_account_btn')
              .removeClass('aialttext-onboard-pulse')
              .removeAttr('aria-describedby')
              .attr('data-linked', '1');
    
            setTimeout(function(){ location.reload(); }, 900);
          } else {
            banner("error", (res && res.data && res.data.message) || __('Failed to link account', 'aialttext-token'));
          }
        })
        .fail(function () {
          banner("error", __('Network or server error. Please try again.', 'aialttext-token'));
        })
        .always(function(){
          $btn.prop("disabled", false).text(__('Link / Log in', 'aialttext-token'));
        });
    });
  
    $(document).on("click", ".buy", function (e) {
      e.preventDefault();
      var $btn = $(this);
      var packageId = $btn.data("id");
      if (window.console && console.log) console.log("[AI Alt Text] Purchase clicked", { packageId: packageId });
    
      $btn.prop("disabled", true).text("Redirecting…");
      ajax("aialttext_token_purchase_tokens", { package_id: packageId })
        .done(function (res) {
          if (res && res.success && res.data && res.data.checkout_url) {
            if (window.console && console.log) console.log("[AI Alt Text] Redirecting to checkout:", res.data.checkout_url);
            location.href = res.data.checkout_url;
          } else {
            $btn.prop("disabled", false).text("Purchase");
            banner("error", (res && res.data && res.data.message) || "Unable to start checkout");
          }
        })
        .fail(function (xhr) {
          $btn.prop("disabled", false).text("Purchase");
          banner("error", "Connection error");
          if (window.console && console.error) console.error("[AI Alt Text] Checkout request failed:", xhr);
        });
    });
  
    function confirmAndRefresh(sessionId, attempts) {
        attempts = attempts || 0;
        ajax("aialttext_token_confirm_payment", { session_id: sessionId })
          .done(function (res) {
            if (res && res.success) {
              var added = Number(res.data.tokens_added || 0);
              var pkg   = Number(res.data.package_tokens || added || 0);
      
              // Update balance (optimistic if webhook lagging)
              if (typeof res.data.balance !== "undefined") {
                jQuery("#aialttext-token-balance").text(res.data.balance);
              } else if (pkg) {
                var cur = parseInt(jQuery("#aialttext-token-balance").text(), 10);
                if (!isNaN(cur)) jQuery("#aialttext-token-balance").text(cur + pkg);
              }
      
              // Update the modal and tidy up any legacy banners
              updatePurchaseModal("Payment confirmed. " + pkg + " tokens added to your account.", true);
              jQuery(".notice-warning:contains('We’re confirming your payment')").remove();
      
              // Drop the querystring so refreshes won't re-enter this flow
              if (history.replaceState) history.replaceState({}, document.title, location.pathname);
            } else {
              if (attempts < 10) {
                setTimeout(function(){ confirmAndRefresh(sessionId, attempts + 1); }, 1000 * (attempts + 1));
              } else {
                updatePurchaseModal("We couldn’t confirm the payment. Please refresh this page or contact support.", true);
              }
            }
          })
          .fail(function () {
            if (attempts < 10) {
              setTimeout(function(){ confirmAndRefresh(sessionId, attempts + 1); }, 1000 * (attempts + 1));
            } else {
              updatePurchaseModal("Network error while confirming payment. Please try refreshing this page.", true);
            }
          });
      }      
  
      var payment = param("payment");
      var sessionId = param("session_id");
      if (payment === "success" && sessionId) {
        showPurchaseModal();
        confirmAndRefresh(sessionId, 0);
      }
  
      (function ($, wp) {
        const { __, _n, sprintf } = wp.i18n;
      
        // state for paged loading
        let usageState = {
          days: 30,
          perPage: 100,
          nextCursor: null,
          items: []
        };
      
        function setButtonsLoading(isLoading) {
          $("#aialttext_token_load_usage").prop("disabled", isLoading).text(isLoading ? __("Loading…", "aialttext-token") : __("Load Usage", "aialttext-token"));
          $("#aialttext_usage_load_more").prop("disabled", isLoading);
        }
      
        function fetchUsageBatch(initial) {
          setButtonsLoading(true);
      
          const data = {
            days: usageState.days,
            per_page: usageState.perPage
          };
          if (usageState.nextCursor) data.cursor = usageState.nextCursor;
      
          return ajax("aialttext_token_load_usage", data).done(function (res) {
            const $list = $("#aialttext_token_usage_list");
      
            if (!res || !res.success || !res.data) {
              $list.html("<p><em>" + __("No usage history found.", "aialttext-token") + "</em></p>");
              setButtonsLoading(false);
              return;
            }
      
            const batch = Array.isArray(res.data.usage) ? res.data.usage : [];
            if (initial) usageState.items = [];
            usageState.items = usageState.items.concat(batch);
      
            // keep pagination state
            usageState.nextCursor = res.data.has_more ? res.data.next_cursor : null;
      
            // Render (group by local date)
            renderUsage(usageState.items);
      
            // Show/hide more/expand/collapse
            if (usageState.nextCursor) {
              $("#aialttext_usage_load_more").show();
            } else {
              $("#aialttext_usage_load_more").hide();
            }
            $("#aialttext_usage_expand_all, #aialttext_usage_collapse_all").show();
      
            setButtonsLoading(false);
          }).fail(function () {
            banner("error", __("Connection error while loading usage history", "aialttext-token"));
            setButtonsLoading(false);
          });
        }
      
        function renderUsage(items) {
          // Remember which groups the user has open *before* we replace the DOM
          const previouslyOpen = new Set(
            Array.prototype.map.call(
              document.querySelectorAll('.aialttext-usage-toggle[aria-expanded="true"]'),
              function(btn){ return btn.getAttribute('data-date'); }
            )
          );
        
          // Map -> { 'YYYY-MM-DD': [items] }
          const byDate = {};

          items.forEach(function (it) {
            const dt = new Date(it.created_at.replace(' ', 'T'));
            const key = dt.getFullYear() + "-" + ('0' + (dt.getMonth()+1)).slice(-2) + "-" + ('0' + dt.getDate()).slice(-2);
            if (!byDate[key]) byDate[key] = [];
            byDate[key].push(it);
          });
        
          const dates = Object.keys(byDate).sort().reverse();
          let html = "";
        
          dates.forEach(function (dateKey) {
            const dayItems = byDate[dateKey];
            const tokenSum = dayItems.reduce(function (s, it) { return s + (parseInt(it.tokens_used, 10) || 0); }, 0);
            // Stable id for the date group (no random suffix)
const gid = "aialttext-usage-" + dateKey.replace(/[^\d]/g, "");
html += '<div class="aialttext-usage-group">';
html +=   '<button class="aialttext-usage-toggle" data-date="' + dateKey + '" aria-expanded="false" aria-controls="' + gid + '">';
            html +=     '<span class="date">' + dateKey + '</span>';
            html +=     '<span class="meta">('
              + sprintf(_n("%s image", "%s images", dayItems.length, "aialttext-token"), dayItems.length)
              + " • "
              + sprintf(_n("%s token", "%s tokens", tokenSum, "aialttext-token"), tokenSum)
              + ')</span>';
            html +=   '</button>';
            html +=   '<div id="' + gid + '" class="aialttext-usage-items" hidden>';
        
            dayItems.forEach(function (it) {
              const ttl = it.image_title ? it.image_title : "";
              const url = it.image_url ? it.image_url : "";
              const alt = it.alt_text ? it.alt_text : "";
        
              const dt = it.created_at ? new Date(it.created_at.replace(' ', 'T')) : null;
              const timeStr = dt ? dt.toLocaleTimeString([], {hour: '2-digit', minute: '2-digit', second: '2-digit'}) : "";
        
              html += '<div class="aialttext-usage-item">';
        
              // 1) Thumbnail (viewable size)
              if (url) {
                html += '<div class="thumb"><img src="' + url.replace(/"/g, "&quot;") + '" loading="lazy" /></div>';
              } else {
                html += '<div class="thumb"></div>';
              }
        
              // 2) “View Image” button – link is resolved to Media Library edit page via a follow-up AJAX map
              html += '<div class="col-view">';
              if (url) {
                html += '<a class="button aialttext-view-image" data-url="' + url.replace(/"/g, "&quot;") + '" href="' + url.replace(/"/g, "&quot;") + '" target="_blank" rel="noopener">' + __("View Image", "aialttext-token") + '</a>';
              }
              html += '</div>';
        
              // 3) Alt text + title
              html += '<div class="meta">';
              html +=   '<div class="title">' + "File Name: " + (ttl ? ttl.replace(/</g,"&lt;").replace(/>/g,"&gt;") : __("Untitled image", "aialttext-token")) + '</div>';
              html +=   '<div class="alt"><code>' + "Alt: " + (alt ? alt.replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;") : "") + '</code></div>';
              html += '</div>';
        
              // 4) Exact time added (site locale)
              html += '<div class="col-time">' + "Time: " + (timeStr || "") + '</div>';
        
              // 5) Token usage
              var tokens = parseInt(it.tokens_used, 10) || 0;
html += '<div class="col-tokens">' +
  sprintf(_n("%s token used", "%s tokens used", tokens, "aialttext-token"), tokens) +
  '</div>';
        
              html += '</div>'; // .aialttext-usage-item
            });
        
            html +=   '</div></div>';
          });
        
          $("#aialttext_token_usage_list").html(html);

// Restore expanded groups so interactions remain usable while more data arrives
$('.aialttext-usage-toggle').each(function(){
  var $t = jQuery(this), date = $t.data('date');
  if (previouslyOpen.has(date)) {
    $t.attr('aria-expanded', true);
    jQuery('#' + $t.attr('aria-controls')).attr('hidden', false);
  }
});
        
          // Resolve Media Library edit links for known URLs
          const urls = [];
          $("#aialttext_token_usage_list .aialttext-view-image").each(function(){ 
            const u = $(this).attr("data-url");
            if (u && urls.indexOf(u) === -1) urls.push(u);
          });
          if (urls.length) {
            ajax("aialttext_token_map_urls_to_media", { urls: urls }).done(function(res){
              if (!res || !res.success || !res.data || !res.data.map) return;
              const map = res.data.map || {};
              $("#aialttext_token_usage_list .aialttext-view-image").each(function(){
                const u = $(this).attr("data-url");
                if (u && map[u] && map[u].edit_link) {
                  $(this).attr("href", map[u].edit_link).attr("target", "_blank").text(__("View Image", "aialttext-token"));
                }
              });
            });
          }
        }
      
        // Initial load click
        $(document).on("click", "#aialttext_token_load_usage", function (e) {
          e.preventDefault();
          const daysSel = parseInt($("#aialttext_usage_range").val(), 10) || 30;
          usageState.days = Math.max(1, Math.min(365, daysSel));
          usageState.perPage = 100;
          usageState.nextCursor = null;
          usageState.items = [];
          $("#aialttext_token_usage_list").empty();
          fetchUsageBatch(true).then(function () {
            // Auto-load more batches until the window is exhausted or ~5 pages, whichever comes first
            let autoPages = 1, MAX_AUTO = 5;
            const loop = function () {
              if (usageState.nextCursor && autoPages < MAX_AUTO) {
                autoPages++;
                fetchUsageBatch(false).then(loop);
              }
            };
            loop();
          });
          // === AUTO-LOAD: first 30 days on page open ================================
(function autoLoadUsage(){
  // Only run on the Usage page (guards keep other admin pages untouched)
  var $list = jQuery('#aialttext_token_usage_list');
  var $btn  = jQuery('#aialttext_token_load_usage');

  if (!$list.length || !$btn.length) return; // Not on the Usage screen

  // Prevent double-trigger if the script is enqueued twice or navigated back to
  if (window.aialttextUsageAutoLoaded) return;
  window.aialttextUsageAutoLoaded = true;

  // Default the range to 30 only when nothing is set yet
  var $range = jQuery('#aialttext_usage_range');
  if ($range.length && !$range.val()) {
    $range.val('30'); // your control already understands "30 days"
  }

  // Fire the existing logic (uses your current click handler’s flow)
  $btn.trigger('click');
})();
        });


      
        // Manual "Load more"
        $(document).on("click", "#aialttext_usage_load_more", function (e) {
          e.preventDefault();
          if (usageState.nextCursor) fetchUsageBatch(false);
        });
      
        // Keep your existing expand/collapse handlers unchanged
      })(jQuery, wp);
      
      /* Toggle handlers + bulk expand/collapse */
      $(document).on("click", ".aialttext-usage-toggle", function () {
        var expanded = $(this).attr("aria-expanded") === "true";
        var target   = $("#" + $(this).attr("aria-controls"));
        $(this).attr("aria-expanded", !expanded);
        target.attr("hidden", expanded);
      });
      
      $(document).on("click", "#aialttext_usage_expand_all", function () {
        $(".aialttext-usage-toggle").each(function () {
          var $t = $(this), tgt = $("#" + $t.attr("aria-controls"));
          $t.attr("aria-expanded", true); tgt.attr("hidden", false);
        });
      });
      
      $(document).on("click", "#aialttext_usage_collapse_all", function () {
        $(".aialttext-usage-toggle").each(function () {
          var $t = $(this), tgt = $("#" + $t.attr("aria-controls"));
          $t.attr("aria-expanded", false); tgt.attr("hidden", true);
        });
      });
  
    /* -------------------------------
 * Process Images (bulk)
 * ------------------------------- */
var processing = false;
// Expose a read-only mirror for other scripts/handlers to consult.
window.aialttext_processingActive = false;

var offset = 0;
var totalProcessed = 0;
var totalErrors = 0;
// Start slightly higher; server still enforces a cap and processes sequentially.
// Always respect server-provided batch size (default 1)
var batchSize = parseInt(aialttext_token_ajax.batch_size, 10) || 1;
    var totalImages = 0;
    var timeRemaining = "Unknown";
    var maxImages = parseInt(aialttext_token_ajax.max_images, 10) || 0;
  
    // run-scoped tracking (prevents retries within a run)
    var processedIds = new Set();
    var failedIds = new Set();
    var errorById = new Map();
    var initialTotalImages = 0;
    var lastDone = 0;
    var stalledTicks = 0;
    var scannedOnce = false;

  
    function updateProgressBar(percent) {
      percent = Math.min(100, Math.max(0, percent || 0));
      $("#aialttext_token_progress").css("width", percent + "%").text(percent + "%");
    }
  
    function updateStatsDisplay() {
      // keep the counters updated (even though we no longer render them directly)
      $("#aialttext_token_total_images").text(totalImages);
      $("#aialttext_token_processed_count").text(totalProcessed);
      $("#aialttext_token_error_count").text(totalErrors);
      $("#aialttext_token_time_remaining").text(
        timeRemaining !== "Unknown" ? timeRemaining + " min" : "Unknown"
      );
    
      // LEFT number: tokens used (1 per successful image)
      $("#aialttext_tokens_used").text(totalProcessed);
    
      // RIGHT number: tokens remaining = initial balance − used
      var $rem = $("#aialttext_tokens_remaining");
      if ($rem.length) {
        var initialBalance = parseInt($rem.data("initial-balance"), 10);
        if (isNaN(initialBalance)) {
          // First run: derive + store a stable starting balance
          initialBalance = parseInt($rem.text(), 10) + (parseInt($("#aialttext_tokens_used").text(), 10) || 0);
          $rem.data("initial-balance", initialBalance);
        }
        var tokensRemaining = Math.max(0, initialBalance - totalProcessed);
        $rem.text(tokensRemaining);
    
        // Threshold coloring
        $rem.removeClass("aialttext-warn aialttext-danger");
        if (tokensRemaining < 10) {
          $rem.addClass("aialttext-danger");
        } else if (tokensRemaining < 20) {
          $rem.addClass("aialttext-warn");
        }
      }
    
      // ----- New: human-friendly progress line -----
      var $label = $("#aialttext_progress_label");
      if ($label.length) {
        var completed = (totalImages > 0) && (totalProcessed >= totalImages);
        if (processing) {
          $label.text("Processing " + totalProcessed + " of " + totalImages + " images");
        } else if (completed) {
          $label.text(totalProcessed + " images processed");
        } else if (totalImages > 0) {
          var remaining = Math.max(0, totalImages - totalProcessed);
          $label.text(remaining + " images left to process");
        } else if (scannedOnce && totalImages === 0) {
          $label.text("🎉 All of your images have alt text and there are none left to process at this time.");
        } else {
          $label.text("Scan to find images without alt text");
        }
      }
    
      // ----- New: banner + indicator logic -----
      var $indicator = $("#aialttext_progress_indicator");
      var $banner    = $("#aialttext_processing_banner");
      var isComplete = (!processing && totalImages > 0 && totalProcessed >= totalImages);
    
      $indicator.toggleClass("active", processing);
    
      if (processing) {
        $banner
          .removeClass("is-complete")
          .html('<strong>Working…</strong> Please keep this page open while images are being processed.')
          .show();
      } else if (isComplete) {
        $banner
          .addClass("is-complete")
          .html('🎉 <strong>Hooray, all images processed now have alternative text!</strong>')
          .show();
      } else {
        $banner.removeClass("is-complete").hide();
      }
    }
  
    function updateStatusMessage(percent) {
      var message;
      if (initialTotalImages > 0) {
        var p = Math.min(100, Math.round(percent));
        message = "Progress: " + p + "% complete";
        if (timeRemaining !== "Unknown") {
          message += " • Estimated time remaining: " + timeRemaining + " minutes";
        }
      } else {
        message = "Scanning for images…";
      }
      $("#aialttext_token_process_status").html(message);
    }
  
    function renderErrorsPanel() {
      var $wrap = $("#aialttext_token_error_details").empty();
      if (!errorById.size) return;
      var html = '<details open><summary>Show error details</summary><ul>';
      errorById.forEach(function (msg, id) {
        html += "<li><strong>Image " + id + ":</strong> " + msg + "</li>";
      });
      html += "</ul></details>";
      $wrap.html(html);
    }
  
    function scanForImages() {
      if (window.aialttext_processingActive === true) return;
      $("#aialttext_token_process_button").prop("disabled", true).text("Scanning…");
      $("#aialttext_token_process_status").html("Scanning your media library for images without alt text…");
      ajax("aialttext_token_scan_images_ajax", {
        only_published: jQuery("#aialttext_only_published").is(":checked") ? 1 : 0
      })
        .done(function (res) {
          if (res && res.success) {
            totalImages = res.total_images || 0;
            initialTotalImages = totalImages; // freeze denominator
            scannedOnce = true;

            // NEW: make the input reflect the available max and clamp value if needed
jQuery("#aialttext_token_limit").attr("max", initialTotalImages);
aialttext_applyLimitFromInput(); // reads current input, clamps totals, sets label
          
            if (maxImages > 0 && totalImages > maxImages) {
              totalImages = maxImages;
            }
          
            if (totalImages === 0) {
              // UX for the "all done" state
              var doneMsg = "🎉 All of your images have alt text and there are none left to process at this time.";
              $("#aialttext_token_process_button")
                .prop("disabled", true)
                .addClass("disabled")
                .text("No images to process");
              $("#aialttext_token_stop_button").hide();
              $("#aialttext_progress_label").text(doneMsg);
              $("#aialttext_token_process_status").html(doneMsg);
              // keep the bar at 0% and counters consistent
              updateStatsDisplay();
            } else {
              $("#aialttext_token_process_button")
                .prop("disabled", false)
                .removeClass("disabled")
                .text(aialttext_processButtonLabel());
              $("#aialttext_token_process_status").html("Found " + totalImages + " images without alt text");
              updateStatsDisplay();
            
              // Ensure "Stop processing" is hidden after a scan; only show it during an active run
              $("#aialttext_token_stop_button").hide();
            }
          } else {
            // Reset UI to idle on scan error (prevents stuck "Stop processing" + "Retry Scan")
            processing = false;
            window.aialttext_processingActive = false;
            jQuery("#aialttext_only_published").prop("disabled", false);
            $("#aialttext_token_stop_button").hide();
          
            $("#aialttext_token_process_status").html("Error: " + ((res && res.message) || "Scan failed"));
            $("#aialttext_token_process_button").prop("disabled", false).text("Retry Scan");
          }
        })
        .fail(function (jqXHR, textStatus, errorThrown) {
          // Reset UI to idle on scan transport errors too
          processing = false;
          window.aialttext_processingActive = false;
          jQuery("#aialttext_only_published").prop("disabled", false);
          $("#aialttext_token_stop_button").hide();
        
          $("#aialttext_token_process_status").html(
            "An error occurred during scan. Please try again. Details: " + textStatus + " " + errorThrown
          );
          $("#aialttext_token_process_button").prop("disabled", false).text("Retry Scan");
        });
    }
  
    function processImages() {
      if (!processing) return;
  
      // stop if user limited max images
      if (maxImages > 0 && totalProcessed >= maxImages) {
        $("#aialttext_token_process_status").prepend(
          "<strong>Processing completed!</strong> Reached maximum image limit of " + maxImages + ".<br>"
        );
        $("#aialttext_token_process_button").prop("disabled", false).text( __('Process More Images', 'aialttext-token') );
$("#aialttext_token_stop_button").hide();
processing = false;
window.aialttext_processingActive = false;
jQuery("#aialttext_only_published").prop("disabled", false);
return;
      }
  
      var currentBatchSize = Math.min(25, batchSize);
      if (maxImages > 0 && totalProcessed + currentBatchSize > maxImages) {
        currentBatchSize = maxImages - totalProcessed;
      }
  
      ajax("aialttext_token_process_images_ajax", {
        offset: offset,
        total_processed: totalProcessed,
        batch_size: currentBatchSize,
        max_images: maxImages,
        only_published: jQuery("#aialttext_only_published").is(":checked") ? 1 : 0,
        exclude_ids: Array.from(new Set([].concat(Array.from(processedIds), Array.from(failedIds)))),
      })
      .done(function (res) {
        // Accept both flat ({…}) and WP-wrapped ({success:true, data:{…}}) shapes
        if (res && res.data) { res = Object.assign({ success: !!res.success }, res.data); }      
          if (res && res.success) {
            // mark successes
            (res.processed_ids || []).forEach(function (id) {
              processedIds.add(parseInt(id, 10));
            });
            // mark failures once
            (res.failed || []).forEach(function (f) {
              var fid = parseInt(f.id, 10);
              failedIds.add(fid);
              if (!errorById.has(fid)) errorById.set(fid, f.message || "Failed");
            });
  
            totalProcessed = processedIds.size;
            totalErrors = failedIds.size;
  
            var prevOffset = offset;
offset = (typeof res.offset === "number") ? res.offset : (offset + currentBatchSize);
timeRemaining = res.time_remaining || "Unknown";

// recompute denominator-limited percent
var denom = Math.max(1, initialTotalImages);
var done = processedIds.size + failedIds.size;
var percent = Math.min(100, Math.round((done / denom) * 100));

// Consider progress if counts changed OR the server moved us
if (done !== lastDone || offset !== prevOffset) {
  stalledTicks = 0;
  lastDone = done;
} else {
  stalledTicks++;
}

if (stalledTicks >= 4) {
  processing = false;
  $("#aialttext_token_process_status").append(
    '<p class="notice notice-success">No further progress (remaining images may be unsupported or already attempted). Stopping.</p>'
  );
  $("#aialttext_token_stop_button").hide();
  $("#aialttext_token_process_button").prop("disabled", false).text(aialttext_processButtonLabel());

  // When we stop due to lack of further progress, also re-sync token counts from the Account service
  if (typeof fetchBalance === "function" && jQuery("#aialttext_tokens_remaining").length) {
    fetchBalance();
  }

  return;
}  
            updateProgressBar(percent);
            updateStatsDisplay();
            updateStatusMessage(percent);
            renderErrorsPanel();
  
            // live preview (optional)
if (res.latest_image) {
  $("#aialttext_token_preview_image").attr("src", res.latest_image.url);
  $("#aialttext_token_preview_title").text(res.latest_image.title);
  $("#aialttext_token_preview_alt").text(res.latest_image.alt_text);
  $("#aialttext_token_latest_images").show();
  
  // Add to modern images grid
  addImageToGrid(res.latest_image, 'success');
}

// Update modern interface (prefer detailed payload)
if (Array.isArray(res.processed_details) && res.processed_details.length) {
  // Always insert items so the grid grows one-by-one in real time
  res.processed_details.forEach(function(d) {
    addImageToGrid({
      id:    d.id,
      url:   d.url || '',
      title: d.title || '',
      alt:   d.alt || d.alt_text || ''
    }, 'success');
  });
} else if (Array.isArray(res.processed_ids) && res.processed_ids.length) {
  // Fallback: if we only have IDs, create minimal cards so progress is visible
  res.processed_ids.forEach(function(id) {
    var isLatest = (res.latest_image && String(res.latest_image.id) === String(id));
    var alt = isLatest ? (res.latest_image.alt_text || res.latest_image.alt || '') : '';
    var known = aialttextImageBuffer.some(function (x) { return String(x.id) === String(id); });

    if (!known) {
      addImageToGrid({
        id:    id,
        url:   isLatest && res.latest_image.url ? res.latest_image.url : '',
        title: isLatest && res.latest_image.title ? res.latest_image.title : '',
        alt:   alt
      }, 'success');
    } else {
      updateImageInGrid(id, 'success', alt);
    }
  });
}

if (Array.isArray(res.failed) && res.failed.length) {
  res.failed.forEach(function(f) {
    updateImageInGrid(f.id, 'error');
  });
}
  
            // NEW: stop immediately if tokens ran out
            if (
                res.insufficient_tokens ||
                (Array.isArray(res.failed) && res.failed.some(function(f){ return /insufficient.*token/i.test(f.message || '') || /\b402\b/.test(f.message || ''); }))
              ) {
    processing = false;
  
    // Prefer server-provided URL; otherwise derive the Account page from the AJAX URL
    var topUp = (res.top_up_url || (aialttext_token_ajax.ajax_url || '')
      .replace('admin-ajax.php', 'admin.php?page=aialttext-token'));
  
    $("#aialttext_token_process_status").append(
      '<div class="notice notice-error">' +
        '<p><strong>Insufficient tokens.</strong> Bulk processing has been paused.</p>' +
        '<p><a class="button button-primary" href="' + topUp + '">Top up tokens</a></p>' +
      '</div>'
    );
  
    $("#aialttext_token_stop_button").hide();
    $("#aialttext_token_process_button").prop("disabled", false).text("Resume After Top-Up");

    // After stopping due to insufficient tokens, re-sync the token panel with the account balance
    if (typeof fetchBalance === "function" && jQuery("#aialttext_tokens_remaining").length) {
      fetchBalance();
    }
  } else if (res.completed || done >= denom) {
    processing = false;
window.aialttext_processingActive = false;
jQuery("#aialttext_only_published").prop("disabled", false);

    // Flip the UI into the finished state; the banner now communicates completion.
    updateStatsDisplay();

    // On successful completion, refresh the token usage panel from the server so the final balance is accurate
    if (typeof fetchBalance === "function" && jQuery("#aialttext_tokens_remaining").length) {
      fetchBalance();
    }
$("#aialttext_token_stop_button").hide();
$("#aialttext_token_process_button").prop("disabled", false).text(aialttext_processButtonLabel());
$("#aialttext_token_process_status").empty();
  // Refresh counts for the current filter so the "Hooray" state is explicit.
  // This will set total_images to 0 (for the chosen scope) and switch the
  // banner/button into the "all done" UX without the user clicking anything.
  if (typeof scanForImages === "function") {
    setTimeout(function(){ scanForImages(); }, 100);
  }
  } else {
    if (processing) {
      setTimeout(processImages, 500);
    }
  }
  
          } else {
            // Recoverable server error; advance and continue
            $("#aialttext_token_process_status").append(
              '<p class="error">Error occurred but attempting to continue: ' +
              ((res && (res.message || (res.error && res.error.message))) || "Unknown error") +
                "</p>"
            );
            offset += currentBatchSize;
            if (processing) setTimeout(processImages, 1000);
          }
        })
        .fail(function (jqXHR, textStatus, errorThrown) {
          $("#aialttext_token_process_status").append(
            '<p class="error">AJAX error occurred: ' + textStatus + " " + errorThrown + " - attempting to continue</p>"
          );
          offset += currentBatchSize;
          if (processing) setTimeout(processImages, 3000);
        });
    }
  
    $(document).on("click", "#aialttext_token_process_button", function () {
      if (totalImages === 0) {
        scanForImages();
        return;
      }
  
      if (!processing) {
        // fresh run (not resume)
        if ($(this).text().indexOf("Resume") === -1) {
          processedIds = new Set();
          failedIds = new Set();
          errorById = new Map();
          initialTotalImages = Math.max(1, totalImages);
          totalProcessed = 0;
          totalErrors = 0;
          offset = 0;
        }
        processing = true;
window.aialttext_processingActive = true;
// Don’t allow the filter to change mid-run
jQuery("#aialttext_only_published").prop("disabled", true);

$(this).prop("disabled", true).text("Processing…");
$("#aialttext_token_stop_button").show();
$("#aialttext_token_progress_bar").show();
        updateProgressBar(0);
        $("#aialttext_token_process_status").empty();
        $("#aialttext_token_error_details").empty();
  
        processImages();
      }
    });
  
    $(document).on("click", "#aialttext_token_stop_button", function () {
      processing = false;
      window.aialttext_processingActive = false;
      jQuery("#aialttext_only_published").prop("disabled", false);
    
      $(this).hide();
      $("#aialttext_token_process_button").prop("disabled", false).text( __('Resume Processing', 'aialttext-token') );
    });
  
    $(document).on("click", ".toggle-errors", function (e) {
      e.preventDefault();
      $(this).next(".error-details").toggle();
      var txt = $(this).next(".error-details").is(":visible") ? "Hide error details" : "Show error details";
      $(this).text(txt);
    });
  
    if ($("#aialttext_token_process_button").length) {
      // auto-scan on page load
      scanForImages();
    }
  
    /* -------------------------------
     * Single-image generation
     * ------------------------------- */
  
    // Attachment screen button with id and data-id
    $(document).on("click", "#aialttext-token-generate-single", function () {
      var id = parseInt($(this).data("id"), 10);
      if (!id) return;
      var $status = $("#aialttext-token-single-status");
      $status.text("Generating…");
      
      // Enhanced context detection (safe name)
let aialtCtxPostId = 0;

// Prefer localized ID (matches attachment.js)
if (window.aialttext_token_attachment && window.aialttext_token_attachment.current_post_id) {
  aialtCtxPostId = parseInt(window.aialttext_token_attachment.current_post_id, 10) || 0;
} else if (window.aialttext_current_post_id) {
  aialtCtxPostId = parseInt(window.aialttext_current_post_id, 10) || 0;
} else {
  // From URL parameters
  const urlParams = new URLSearchParams(window.location.search);
  if (urlParams.has('post')) {
    aialtCtxPostId = parseInt(urlParams.get('post'), 10) || 0;
  }

  // From hidden form fields
  const postField = document.querySelector('#post_ID, input[name="post"], input[name="post_id"]');
  if (!aialtCtxPostId && postField && postField.value) {
    aialtCtxPostId = parseInt(postField.value, 10) || 0;
  }
}

if (aialtCtxPostId && window.console) {
  console.log('🎯 Context detected for single generation:', aialtCtxPostId);
}

  
ajax("aialttext_token_generate_single", { 
  attachment_id: id,
  post_id: aialtCtxPostId || 0
})
.done(function (res) {
  const altText = res && res.data ? (res.data.alt_text || res.data.alt || '') : '';

  if (res && res.success && altText) {
    var successMsg = "Alt text generated and saved.";
    if (contextPostId) {
        successMsg += " (Context: Post #" + contextPostId + ")";
    }
    $status.html('<span style="color:#46b450;">✓</span> ' + successMsg);
    // try to reflect in field if visible
    $('input[name="attachments[' + id + '][image_alt]"], textarea#attachment-details-two-column-alt-text').val(
      altText
    ).addClass('field-highlight');
    setTimeout(function() {
        $('input[name="attachments[' + id + '][image_alt]"], textarea#attachment-details-two-column-alt-text').removeClass('field-highlight');
    }, 2000);
  } else {
    $status.html('<span style="color:#d63638;">⚠</span> Failed: ' + ((res && res.data && res.data.message) || "Error"));
  }
})

        .fail(function () {
          $status.html('<span style="color:#d63638;">⚠</span> Network error');
        });
    });
  
    // Media Library / modal buttons: class with data-attachment-id
    $(document).on("click", ".aialttext-token-generate", function (e) {
      e.preventDefault();
      var $btn = $(this);
      var id = parseInt($btn.data("attachment-id"), 10);
      if (!id) return;
      
      // Enhanced context detection
      var contextPostId = 0;
      
      // Try to get context from various sources
      if (window.aialttext_current_post_id) {
          contextPostId = window.aialttext_current_post_id;
      } else {
          // From URL parameters
          var urlParams = new URLSearchParams(window.location.search);
          if (urlParams.has('post')) {
              contextPostId = parseInt(urlParams.get('post'), 10);
          }
          
          // From hidden form fields
          var postField = document.querySelector('#post_ID, input[name="post"], input[name="post_id"]');
          if (postField && postField.value) {
              contextPostId = parseInt(postField.value, 10);
          }
      }
      
      if (contextPostId && window.console) {
          console.log('🎯 Context detected for media library generation:', contextPostId);
      }
  
      $btn.prop("disabled", true).text("Generating…");
  
      ajax("aialttext_token_generate_single", { 
          attachment_id: id,
          post_id: contextPostId || 0
      })
        .done(function (res) {
          if (res && res.success && res.data && res.data.alt) {
            $btn.text( __('Generated!', 'aialttext-token') ).css("background-color", "#46b450");
            var $alt =
              $('input[name="attachments[' + id + '][image_alt]"]') ||
              $("#attachment-details-two-column-alt-text");
            if ($alt && $alt.length) {
              $alt.val(res.data.alt).trigger("change").css("background-color", "#e7f7e3");
              setTimeout(function () {
                $alt.css("background-color", "");
              }, 2000);
            }
            
            // Show context info if available
            if (contextPostId) {
                $btn.after('<div class="aialttext-context-detected">Context: Post/Page #' + contextPostId + '</div>');
                setTimeout(function() {
                    $('.aialttext-context-detected').fadeOut();
                }, 3000);
            }
          } else {
            $btn.prop("disabled", false).text("Generate");
            alert("Error: " + ((res && res.data && res.data.message) || "Failed to generate alt text"));
          }
        })
        .fail(function () {
          $btn.prop("disabled", false).text("Generate");
          alert("Network error occurred");
        })
        .always(function () {
          setTimeout(function () {
            $btn.prop("disabled", false).text("Generate").css("background-color", "");
          }, 2000);
        });
    });
  
    // If a single-generate button is present and we can read balance, disable if zero
    if ($("#aialttext-token-generate-single").length) {
      ajax("aialttext_token_refresh_balance").done(function (res) {
        if (res && res.success) {
          var bal = parseInt(res.data.balance || 0, 10);
          if (bal <= 0) $("#aialttext-token-generate-single").prop("disabled", true).after(" <em>(no tokens)</em>");
        }
      });
    }

    // Paging controls
$(document).on('click', '#aialttext_show_more', function(e) {
  e.preventDefault();
  if (!aialttextFirstMoreClick) {
    aialttextDisplayLimit += 20; // first click: +20 (so 40 total)
    aialttextFirstMoreClick = true;
  } else {
    aialttextDisplayLimit += 15; // subsequent clicks: +15
  }
  renderAialttextGrid();
});

$(document).on('click', '#aialttext_show_all', function(e) {
  e.preventDefault();
  aialttextDisplayLimit = aialttextGridItems.length;
  renderAialttextGrid();
});

   /* -------------------------------
     * Media Library Table View Generate Button
     * ------------------------------- */
    
    // Use event delegation for dynamically added buttons - improved selector
    $(document).on('click', '.aialttext-generate-table', function(e) {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation(); // Prevent any other handlers
      
      var $btn = $(this);
      
      // Skip if already processing
      if ($btn.prop('disabled') || $btn.hasClass('processing')) {
          console.log('AI Alt Text: Button already processing, skipping');
          return false;
      }
      
      // Mark as processing
      $btn.addClass('processing');
      
      var imageId = parseInt($btn.data('image-id') || $btn.attr('data-image-id'), 10);
      
      if (!imageId || isNaN(imageId)) {
          console.error('AI Alt Text: No valid image ID found on button', $btn);
          return false;
      }
      
      console.log('AI Alt Text: Generating for image ID:', imageId);
      
      // Store original text
      var originalText = $btn.text();
      
      // Disable button and show processing state
      $btn.prop('disabled', true)
          .text('Generating...')
          .css('opacity', '0.6');
      
      // Make the AJAX call
      $.ajax({
          url: aialttext_token_ajax.ajax_url,
          type: 'POST',
          dataType: 'json',
          data: {
              action: 'aialttext_token_generate_single',
              nonce: aialttext_token_ajax.nonce,
              attachment_id: imageId,
              post_id: 0  // No context in media library list view
          },
          success: function(res) {
              console.log('AI Alt Text: Response received', res);
              
              if (res && res.success && res.data && (res.data.alt || res.data.alt_text)) {
                  var altText = res.data.alt || res.data.alt_text;
                  
                  // Update the entire cell content to show success
                  var $container = $btn.closest('.aialttext-alt-indicator');
                  if ($container.length) {
                      $container.replaceWith('<div class="aialttext-alt-indicator" style="color:#46b450;font-weight:bold;">✓ Yes</div>');
                  } else {
                      // Fallback: replace the parent div
                      $btn.parent().html('<div style="color:#46b450;font-weight:bold;">✓ Yes</div>');
                  }
                  
                  // Show a brief success message
                  var $row = $btn.closest('tr');
                  if ($row.length) {
                      var $msg = $('<span style="color:#46b450;margin-left:10px;font-weight:normal;">Generated!</span>');
                      $row.find('.aialttext-alt-indicator').append($msg);
                      setTimeout(function() {
                          $msg.fadeOut();
                      }, 3000);
                  }
              } else {
                  // Error - restore button
                  $btn.prop('disabled', false)
                      .text(originalText)
                      .css('opacity', '1');
                  
                  var errorMsg = (res && res.data && res.data.message) ? res.data.message : 'Failed to generate alt text';
                  alert('Error: ' + errorMsg);
              }
            },
            error: function(xhr, status, error) {
                console.error('AI Alt Text: AJAX error', status, error, xhr.responseText);
                
                // Remove processing class
                $btn.removeClass('processing');
                
                // Restore button on error
                $btn.prop('disabled', false)
                    .text(originalText)
                    .css('opacity', '1');
                
                // More detailed error message
                var errorMsg = 'Network error occurred';
                if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMsg = xhr.responseJSON.data.message;
                  } else if (xhr.status === 403) {
                    errorMsg = __('Security check failed. Please refresh the page and try again.', 'aialttext-token');
                }
                
                alert(__('Error:', 'aialttext-token') + ' ' + errorMsg);
            },
            complete: function() {
                // Always remove processing class
                $btn.removeClass('processing');
            }
        });
        
        return false;
    });
    // Modern interface functions
    function addImageToGrid(imageData, status) {
      status = status || 'processing';
    
      var item = {
        id:    imageData.id || '',
        url:   imageData.url || '',
        title: imageData.title || '',
        alt:   imageData.alt || imageData.alt_text || 'Generating…',
        status: status
      };
    
      // De-dup and move to top (newest-first buffer)
      var idx = aialttextImageBuffer.findIndex(function (x) { return String(x.id) === String(item.id); });
      if (idx > -1) aialttextImageBuffer.splice(idx, 1);
      aialttextImageBuffer.unshift(item);
    
      renderProcessedGrid();
    }

    function updateImageInGrid(imageId, status, altText, fallbackData) {
      var idx = aialttextImageBuffer.findIndex(function (x) { return String(x.id) === String(imageId); });
    
      if (idx > -1) {
        if (status) aialttextImageBuffer[idx].status = status;
        if (typeof altText === 'string' && altText.length) {
          aialttextImageBuffer[idx].alt = altText;
        }
        renderProcessedGrid();
        return;
      }
    
      // If we get an update for something we don't yet track,
      // create a minimal card so progress is visible immediately.
      addImageToGrid(Object.assign({
        id: imageId,
        url: '',
        title: '',
        alt: (typeof altText === 'string' ? altText : '')
      }, (fallbackData || {})), status || 'success');
      // addImageToGrid() re-renders for us
    }

// Initialize token balance on page load
if ($("#aialttext_tokens_remaining").length) {
  fetchBalance();
}

// Check if token panel should be hidden based on CSS (for browser compatibility)
$(document).ready(function() {
  var $tokenPanel = $('#aialttext_token_usage_panel');
  if ($tokenPanel.length && $tokenPanel.is(':hidden')) {
    $('.aialttext-modern-interface').addClass('aialttext-hide-tokens');
  }
});
    });

    // Debug log display helper
function displayDebugLogs(response) {
  if (response && response.debug_logs && Array.isArray(response.debug_logs)) {
      console.group('🔍 AI Alt Text Debug Logs');
      response.debug_logs.forEach(function(log) {
          const style = 'color: #2563eb; font-weight: bold;';
          console.log('%c' + log.time + ' - ' + log.message, style);
          if (log.data) {
              console.log('   Data:', log.data);
          }
      });
      console.groupEnd();
  }
  // Also surface which model/provider actually served the request
if (response && response.data && response.data.model) {
  console.info('[AI Alt Text] Used model:', response.data.model);
}
}

// Wrap existing AJAX success handlers to show debug logs
if (typeof jQuery !== 'undefined') {
  jQuery(document).ajaxSuccess(function(event, xhr, settings) {
      try {
          const response = JSON.parse(xhr.responseText);
          if (response && response.debug_logs) {
              displayDebugLogs(response);
          }
      } catch (e) {
          // Not JSON or doesn't have debug_logs, ignore
      }
  });
}

// FAQ accordion on Help page
jQuery(document).on('click', '.aialttext-token-faq h4', function () {
  var $h = jQuery(this);
  $h.toggleClass('active');
  $h.next('.aialttext-token-faq-answer').stop(true, true).slideToggle(150);
});

/* ================================
 * IMG Alt Gen – UX enhancements
 * ================================ */
jQuery(function($){

  // Inject a "Forgot password?" link below the existing password help text (once)
  (function addForgotLink(){
    var $pwd = $('#aialttext_token_account_password');
    if (!$pwd.length || $('#aialttext_token_forgot').length) return;
    var $desc = $pwd.closest('td').find('p.description').last();
    var $p = $('<p class="description" style="margin-top:6px;"></p>');
    var $a = $('<a id="aialttext_token_forgot" href="#">'+ __('Forgot your password?', 'aialttext-token') +'</a>');
    $p.append($a);
    if ($desc.length) { $p.insertAfter($desc); } else { $pwd.closest('td').append($p); }
  })();

  // Override the existing "Link / Log in" click handler with a preflight
  $(document).off('click', '#aialttext_token_link_account_btn');
  $(document).on('click', '#aialttext_token_link_account_btn', function(e){
    e.preventDefault();

    // Consent (Terms + Privacy) is required
var consent = $("#aialttext_token_consent").is(":checked");
if (!consent) {
  window.aialttextBanner("error", __('Please agree to the Terms and Privacy Policy to continue.', 'aialttext-token'));
  return;
}

    var $btn     = $("#aialttext_token_link_account_btn");
    var email    = $("#aialttext_token_account_email").val();
    var password = $("#aialttext_token_account_password").val() || "";

    if (!email) {
      window.aialttextBanner("error", __('Email required', 'aialttext-token'));
      return;
    }
    if (!password || password.length < 8) {
      window.aialttextBanner("error", __('Password (min. 8 chars) required', 'aialttext-token'));
      return;
    }

    $btn.prop('disabled', true).text(__('Checking…', 'aialttext-token'));

    // 1) Preflight: does this email exist?
    window.aialttextAjax('aialttext_token_account_exists', { email: email })
      .done(function(res){
        var exists = !!(res && (res.exists || (res.data && res.data.exists)));
        $btn.text( exists ? __('Logging In…', 'aialttext-token') : __('Signing Up…', 'aialttext-token') );

        // 2) Proceed with the existing link endpoint (create-or-link)
        window.aialttextAjax('aialttext_token_link_account', {
          email: email,
          password: password,
          consent: ( $("#aialttext_token_consent").is(":checked") ? 1 : 0 )
        })
          .done(function (linkRes) {
            if (linkRes && linkRes.success) {
              try { console.log('[IMG Alt Gen Pro] Account linked via AJAX.'); } catch (e) {}
              window.aialttextBanner("success", __('Account linked successfully.', 'aialttext-token'));
              jQuery("#aialttext-token-balance").text(linkRes.data && linkRes.data.balance ? linkRes.data.balance : 0);

              // Stop onboarding pulse defensively
              jQuery('#aialttext_token_link_account_btn')
                .removeClass('aialttext-onboard-pulse')
                .removeAttr('aria-describedby')
                .attr('data-linked', '1');

              setTimeout(function(){ location.reload(); }, 900);
            } else {
              window.aialttextBanner("error", (linkRes && linkRes.data && linkRes.data.message) || __('Failed to link account', 'aialttext-token'));
            }
          })
          .fail(function(){
            window.aialttextBanner("error", __('Network or server error. Please try again.', 'aialttext-token'));
          })
          .always(function(){
            $btn.prop('disabled', false).text(__('Link / Log in', 'aialttext-token'));
          });
      })
      .fail(function(){
        // If the existence check fails, fall back to generic label and still try linking
        $btn.text(__('Linking…', 'aialttext-token'));
        window.aialttextAjax('aialttext_token_link_account', {
          email: email,
          password: password,
          consent: ( $("#aialttext_token_consent").is(":checked") ? 1 : 0 )
        })
          .done(function (linkRes) {
            if (linkRes && linkRes.success) {
              window.aialttextBanner("success", __('Account linked successfully.', 'aialttext-token'));
              setTimeout(function(){ location.reload(); }, 900);
            } else {
              window.aialttextBanner("error", (linkRes && linkRes.data && linkRes.data.message) || __('Failed to link account', 'aialttext-token'));
            }
          })
          .fail(function(){
            window.aialttextBanner("error", __('Network or server error. Please try again.', 'aialttext-token'));
          })
          .always(function(){
            $btn.prop('disabled', false).text(__('Link / Log in', 'aialttext-token'));
          });
      });
  });

  // Handle "Forgot password?" click → fire admin reset email
  $(document).on('click', '#aialttext_token_forgot', function(e){
    e.preventDefault();
    var email = $("#aialttext_token_account_email").val();
    if (!email) { window.aialttextBanner('error', __('Enter your email above first.', 'aialttext-token')); return; }

    var $btn = $("#aialttext_token_link_account_btn");
    var original = $btn.text();
    $btn.prop('disabled', true).text(__('Sending reset…', 'aialttext-token'));

    window.aialttextAjax('aialttext_token_password_forgot', { email: email })
      .done(function(res){
        window.aialttextBanner('success', __('If an account exists, a reset link has been emailed.', 'aialttext-token'));
      })
      .fail(function(){
        window.aialttextBanner('error', __('Could not send reset email. Please try again.', 'aialttext-token'));
      })
      .always(function(){
        $btn.prop('disabled', false).text(original);
      });
  });

});

// ADD: simple password meter on account password field
(function(){
  var pw = document.getElementById('aialttext_token_account_password');
  if (!pw) return;
  var reqs = {
    len: s => s.length >= 8,
    upper: s => /[A-Z]/.test(s),
    num: s => /\d/.test(s),
    special: s => /[^A-Za-z0-9]/.test(s),
  };
  function paint(){
    var v = pw.value || '';
    ['len','upper','num','special'].forEach(function(k){
      var li = document.querySelector('#aialttext-password-meter li[data-req="'+k+'"]');
      if (li) li.style.color = reqs[k](v) ? 'inherit' : '#d63638';
    });
  }
  pw.addEventListener('input', paint);
  paint();
})();

})(window.wp, jQuery);
