<?php
if (!defined('ABSPATH')) exit;

/**
 * Treat checkbox-style options ('1', 1, true, 'true', 'on', 'yes') as enabled.
 */
if (!function_exists('aialttext_token_is_enabled')) {
    function aialttext_token_is_enabled(string $option_name, $default = 0): bool {
        $v = get_option($option_name, $default);
        return in_array($v, array(1, '1', true, 'true', 'on', 'yes'), true);
    }
}

// Build contextual hints (WooCommerce, Yoast, RankMath)
function aialttext_token_build_context_for_attachment( $attachment_id ) {
    $parts  = array();
    $parent = wp_get_post_parent_id( $attachment_id );

    if ( $parent && 'product' === get_post_type( $parent ) && function_exists( 'wc_get_product' ) ) {
        $product = wc_get_product( $parent );
        if ( $product ) {
            $p_title = wp_strip_all_tags( $product->get_name() );
            if ( $p_title !== '' ) {
                $parts[] = 'PRODUCT_TITLE: ' . $p_title . '.';
            }
            $desc_raw = $product->get_short_description();
            if ( ! $desc_raw ) { $desc_raw = $product->get_description(); }
            $desc = wp_strip_all_tags( $desc_raw );
            if ( $desc !== '' ) {
                $parts[] = 'PRODUCT_DESC: ' . wp_html_excerpt( $desc, 300, '' ) . '.';
            }
        }
    } elseif ( $parent && defined( 'WPSEO_VERSION' ) ) {
        // SEO keyphrase when available (normalized)
        $kw = (string) get_post_meta( $parent, '_yoast_wpseo_focuskw', true );
        if ( $kw !== '' ) {
            $parts[] = 'SEO_KEYPHRASE: ' . rtrim( wp_strip_all_tags( $kw ), " .\t\n\r\0\x0B" );
        }
    }

    return implode( "\n", array_filter( $parts ) );
}


/**
 * Detect the current parent post ID when an image is uploaded from edit screens.
 * Works in Classic Editor, Gutenberg (REST), and WooCommerce product editor.
 *
 * @param int $attachment_id Optional. Attachment ID as a fallback to read post_parent.
 * @return int Post ID or 0 when unknown.
 */
function aialttext_token_detect_current_post_id( $attachment_id = 0 ) {
    $pid = 0;

    // 1) Standard upload params (media modal / REST)
    if ( isset( $_REQUEST['post_id'] ) ) {
        $pid = absint( $_REQUEST['post_id'] );
    }
    if ( ! $pid && isset( $_REQUEST['post'] ) ) {
        $pid = absint( $_REQUEST['post'] );
    }

    // 2) Classic editor screen
    if ( ! $pid && is_admin() && function_exists( 'get_current_screen' ) ) {
        $screen = get_current_screen();
        if ( $screen && ! empty( $_GET['post'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $pid = absint( $_GET['post'] );
        }
    }

    // 3) Referer patterns (covers Classic/Gutenberg + Woo block product editor + media modal)
if ( ! $pid && ! empty( $_SERVER['HTTP_REFERER'] ) ) {
    $ref = (string) $_SERVER['HTTP_REFERER'];

    // Classic/Gutenberg edit (?post=123) and REST parent hint
    if ( preg_match( '/[?&]post=([0-9]+)/', $ref, $m ) ) {
        $pid = (int) $m[1];
    } elseif ( preg_match( '#/wp-json/wp/v2/media.*[?&]parent=([0-9]+)#', $ref, $m ) ) {
        $pid = (int) $m[1];
    } elseif ( preg_match( '#wp-admin/(?:post|post-new|post\.php|post-new\.php).*?[?&]post=([0-9]+)#', $ref, $m ) ) {
        $pid = (int) $m[1];
    // WooCommerce block product editor (wc-admin) – multiple URL shapes
} elseif ( preg_match( '#[?&]page=wc-admin[^#]*[?&]path=(?:%2F|/)?product(?:%2F|/)([0-9]+)(?:%2F|/)?(?:edit)?#', $ref, $m ) ) {
    $pid = (int) $m[1];
} elseif ( preg_match( '#[?&]path=(?:%2F|/)?product(?:%2F|/)([0-9]+)(?:%2F|/)?(?:edit)?#', $ref, $m ) ) {
    $pid = (int) $m[1];
} elseif ( preg_match( '#/product/([0-9]+)(?:/edit)?#', $ref, $m ) ) {
    $pid = (int) $m[1];
}
}
    // 4) Session/transient fallback (populated by admin_init tracking in this file)
    if ( ! $pid ) {
        if ( ! session_id() ) {
            @session_start();
        }
        if ( ! empty( $_SESSION['aialttext_last_edited_post'] ) && ! empty( $_SESSION['aialttext_edit_timestamp'] ) ) {
            if ( ( time() - (int) $_SESSION['aialttext_edit_timestamp'] ) <= HOUR_IN_SECONDS ) {
                $pid = absint( $_SESSION['aialttext_last_edited_post'] );
            }
        }
        if ( ! $pid && function_exists( 'get_current_user_id' ) ) {
            $t = get_transient( 'aialttext_current_editing_post_' . get_current_user_id() );
            if ( $t ) {
                $pid = absint( $t );
            }
        }
    }

    // 5) Final fallback: attachment parent
    if ( ! $pid && $attachment_id ) {
        $pid = (int) wp_get_post_parent_id( $attachment_id );
    }

    if ( defined( 'AIALTTEXT_TOKEN_DEBUG' ) && AIALTTEXT_TOKEN_DEBUG ) {
        error_log( '[AI Alt Text] detect_current_post_id => ' . ( $pid ?: 0 ) );
    }

    return $pid > 0 ? $pid : 0;
}



/**
 * Build high-quality context for the AI prompt, pulling from SEO plugins and WooCommerce
 * depending on where the image was uploaded.
 */
function aialttext_token_get_image_context( $image_id, $post_id = 0 ) {
    // Detect parent post if not provided (Classic, Gutenberg, wc-admin)
    if ( ! $post_id ) {
        $post_id = aialttext_token_detect_current_post_id( $image_id );
    }

    $context      = '';
    $is_product   = ( $post_id && get_post_type( $post_id ) === 'product' );
    $woo_enabled  = intval( get_option( 'aialttext_token_woocommerce_integration', 1 ) ) === 1;
    $seo_enabled  = intval( get_option( 'aialttext_token_seo_integration', 1 ) ) === 1;

    // PRODUCT FIRST: when editing a product and Woo is enabled, use ONLY Woo context.
    // PRODUCT: prefer Woo context, then append SEO context if enabled
if ( $is_product && $woo_enabled && class_exists( 'WooCommerce' ) ) {
    $woo = aialttext_token_get_woocommerce_context( $post_id );
    if ( ! empty( $woo ) ) {
        $context = trim( $context . ' ' . $woo );
    }
}

// Append SEO context (Yoast/RankMath) when enabled — works for products and posts/pages
if ( $seo_enabled && $post_id ) {
    $seo = aialttext_token_get_seo_context( $post_id );
    if ( ! empty( $seo ) ) {
        $context = trim( $context . ' ' . $seo );
    }
}

    return trim( $context );
}


/**
 * Extract SEO intent for the current post from major SEO plugins.
 * Rule: If a Yoast/Rank Math/AIOSEO/SEOPress keyphrase exists, include it.
 * Otherwise, include the SEO title + meta description as context.
 */
function aialttext_token_get_seo_context( $post_id ) {
    if ( ! $post_id ) {
        return '';
    }

    $parts = array();

    // Yoast SEO
    // Initialize variables
$focus = '';
$seo_title = '';
$seo_desc = '';

// Yoast SEO
if ( defined( 'WPSEO_VERSION' ) ) {
    if ( class_exists( 'WPSEO_Meta' ) && is_callable( array( 'WPSEO_Meta', 'get_value' ) ) ) {
        $focus = (string) WPSEO_Meta::get_value( 'focuskw', $post_id );
    }
    if ( $focus === '' ) {
        $focus = (string) get_post_meta( $post_id, '_yoast_wpseo_focuskw', true );
    }
    $seo_title = (string) get_post_meta( $post_id, '_yoast_wpseo_title', true );
    $seo_desc  = (string) get_post_meta( $post_id, '_yoast_wpseo_metadesc', true );
}

// RankMath SEO
elseif ( class_exists( 'RankMath' ) || defined( 'RANK_MATH_VERSION' ) ) {
    $focus = (string) get_post_meta( $post_id, 'rank_math_focus_keyword', true );
    $seo_title = (string) get_post_meta( $post_id, 'rank_math_title', true );
    $seo_desc = (string) get_post_meta( $post_id, 'rank_math_description', true );
}

// All in One SEO (AIOSEO)
elseif ( function_exists( 'aioseo' ) && aioseo()->core->db->start() ) {
    // AIOSEO stores data differently - in a custom table
    global $wpdb;
    $aioseo_data = $wpdb->get_row( $wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}aioseo_posts WHERE post_id = %d",
        $post_id
    ) );
    
    if ( $aioseo_data ) {
        // AIOSEO stores keyphrases as JSON
        if ( !empty( $aioseo_data->keyphrases ) ) {
            $keyphrases = json_decode( $aioseo_data->keyphrases, true );
            if ( is_array( $keyphrases ) && !empty( $keyphrases['focus']['keyphrase'] ) ) {
                $focus = (string) $keyphrases['focus']['keyphrase'];
            }
        }
        $seo_title = !empty( $aioseo_data->title ) ? (string) $aioseo_data->title : '';
        $seo_desc = !empty( $aioseo_data->description ) ? (string) $aioseo_data->description : '';
    }
}

// SEOPress
elseif ( function_exists( 'seopress_get_service' ) ) {
    $focus = (string) get_post_meta( $post_id, '_seopress_analysis_target_kw', true );
    $seo_title = (string) get_post_meta( $post_id, '_seopress_titles_title', true );
    $seo_desc = (string) get_post_meta( $post_id, '_seopress_titles_desc', true );
}

// Add the SEO data to parts
if ( $focus !== '' ) {
    $parts[] = 'SEO_KEYPHRASE: ' . wp_strip_all_tags( $focus ) . '.';
} else {
    if ( $seo_title !== '' ) {
        $parts[] = 'SEO_TITLE: ' . wp_strip_all_tags( $seo_title ) . '.';
    }
    if ( $seo_desc !== '' ) {
        $parts[] = 'SEO_META: ' . wp_strip_all_tags( $seo_desc ) . '.';
    }

        // Add improved instructions for natural SEO integration
$has_focus = false;
foreach ( $parts as $p ) {
    if (strpos($p, 'SEO_KEYPHRASE:') === 0) { $has_focus = true; break; }
}
if ( $has_focus ) {
    $parts[] = 'INSTRUCTION: Naturally incorporate the SEO_KEYPHRASE into the alt text when it accurately describes the image. Focus on creating descriptive, natural-sounding alt text that includes the keyphrase organically rather than forcing it in verbatim.';
} else {
    $parts[] = 'INSTRUCTION: No SEO_KEYPHRASE found; use SEO_TITLE and SEO_META as contextual guidance for creating natural, descriptive alt text.';
}


    }

    // Fallback to core post data if no SEO meta found
    if ( empty( $parts ) ) {
        $post = get_post( $post_id );
        if ( $post ) {
            $parts[] = 'SEO_TITLE: ' . wp_strip_all_tags( get_the_title( $post_id ) ) . '.';
            $excerpt = $post->post_excerpt ? $post->post_excerpt : wp_trim_words( $post->post_content, 30, '' );
            if ( $excerpt ) {
                $parts[] = 'SEO_META: ' . wp_strip_all_tags( $excerpt ) . '.';
            }
        }
    }



    return trim( implode( ' ', $parts ) );
}


function aialttext_token_get_woocommerce_context( $post_id ) {
    if ( ! $post_id || ! class_exists( 'WooCommerce' ) ) {
        return '';
    }

    $product = wc_get_product( $post_id );
    if ( ! $product ) {
        return '';
    }

    $parts = array();

    // Title (keep as PRODUCT_TITLE which the server already parses)
    // Title
$p_title = wp_strip_all_tags( $product->get_name() );
if ( $p_title !== '' ) {
    $parts[] = 'PRODUCT_TITLE: ' . $p_title . '.';
}

// Brand (explicit) — supports common brand setups
// Enhanced brand detection - check multiple possible sources
$brand = '';

// 1. Check common brand taxonomies
$brand_taxes = array( 'product_brand', 'pwb-brand', 'brand', 'pa_brand' ); // Added pa_brand for WooCommerce attributes
foreach ( $brand_taxes as $tax ) {
    if ( taxonomy_exists( $tax ) ) {
        $terms = wp_get_post_terms( $product->get_id(), $tax, array( 'fields' => 'names' ) );
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) { 
            $brand = $terms[0]; 
            break; 
        }
    }
}

// 2. Check product attributes if no brand taxonomy found
if ( $brand === '' ) {
    // Check for any attribute with "brand" in the name
    foreach ( $product->get_attributes() as $attribute ) {
        $name = $attribute->get_name();
        $label = wc_attribute_label( $name );
        
        // Check both name and label for "brand"
        if ( stripos( $name, 'brand' ) !== false || stripos( $label, 'brand' ) !== false ) {
            if ( $attribute->is_taxonomy() ) {
                $terms = wp_get_post_terms( $product->get_id(), $name, array( 'fields' => 'names' ) );
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) { 
                    $brand = $terms[0]; 
                    break; 
                }
            } else {
                $options = (array) $attribute->get_options();
                if ( ! empty( $options ) ) { 
                    $brand = reset( $options ); 
                    break; 
                }
            }
        }
    }
}

// 3. Final fallback: check if brand is in the product title itself (common pattern)
if ( $brand === '' && !empty($p_title) ) {
    // Common brand patterns in titles (e.g., "Nike Running Shoes" or "Adidas - Sport Jacket")
    if ( preg_match('/^([A-Z][a-zA-Z]+(?:\s+[A-Z][a-zA-Z]+)?)\s+[-–]/u', $p_title, $m ) ) {
        $brand = trim( $m[1] );
    } elseif ( preg_match('/^([A-Z][a-zA-Z]+(?:\s+[A-Z][a-zA-Z]+)?)\s+[A-Z]/u', $p_title, $m ) ) {
        // Check if first word(s) look like a brand (capitalized)
        $potential_brand = trim( $m[1] );
        // Only use if it's a reasonable length for a brand name
        if ( strlen( $potential_brand ) <= 20 && strlen( $potential_brand ) >= 3 ) {
            $brand = $potential_brand;
        }
    }
}

foreach ( $brand_taxes as $tax ) {
    if ( taxonomy_exists( $tax ) ) {
        $terms = wp_get_post_terms( $product->get_id(), $tax, array( 'fields' => 'names' ) );
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) { $brand = $terms[0]; break; }
    }
}
if ( $brand === '' ) {
    // Visible product attribute labelled like "Brand"
    foreach ( $product->get_attributes() as $attribute ) {
        $label = wc_attribute_label( $attribute->get_name() );
        if ( stripos( $label, 'brand' ) !== false ) {
            if ( $attribute->is_taxonomy() ) {
                $terms = wp_get_post_terms( $product->get_id(), $attribute->get_name(), array( 'fields' => 'names' ) );
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) { $brand = $terms[0]; break; }
            } else {
                $options = (array) $attribute->get_options();
                if ( ! empty( $options ) ) { $brand = reset( $options ); break; }
            }
        }
    }
}
if ( $brand !== '' ) {
    $parts[] = 'PRODUCT_BRAND: ' . wp_strip_all_tags( $brand ) . '.';
}


    // Single normalized description field (PRODUCT_DESC) – trimmed for signal/noise
    $desc_raw = $product->get_short_description();
    if ( ! $desc_raw ) {
        $desc_raw = $product->get_description();
    }
    $desc = wp_strip_all_tags( $desc_raw );
    if ( $desc !== '' ) {
        // Keep it short; models do better with concise hints
        $parts[] = 'PRODUCT_DESC: ' . wp_html_excerpt( $desc, 300, '' ) . '.';
    }

    // Attributes: limit to a few, and a few values each → PRODUCT_ATTRS
    $attrs_out = array();
    foreach ( $product->get_attributes() as $attribute ) {
        if ( ! $attribute->get_visible() ) {
            continue;
        }
        $label = wc_attribute_label( $attribute->get_name() );
        if ( $attribute->is_taxonomy() ) {
            $terms = wp_get_post_terms( $product->get_id(), $attribute->get_name(), array( 'fields' => 'names' ) );
            if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                $attrs_out[] = $label . ': ' . implode( ', ', array_slice( $terms, 0, 3 ) );
            }
        } else {
            $options = (array) $attribute->get_options();
            if ( ! empty( $options ) ) {
                $attrs_out[] = $label . ': ' . implode( ', ', array_slice( $options, 0, 3 ) );
            }
        }
        if ( count( $attrs_out ) >= 4 ) { // cap total attributes
            break;
        }
    }
    if ( ! empty( $attrs_out ) ) {
        $parts[] = 'PRODUCT_ATTRS: ' . implode( '; ', $attrs_out ) . '.';
    }

    return trim( implode( ' ', $parts ) );
}



if (!function_exists('aialttext_token_build_context_for_post')) {
    /**
     * Build rich context for a given post/page/product to guide AI alt text generation.
     * - On WooCommerce products: prefer product title + descriptions (short/long).
     * - On posts/pages with Yoast: prefer focus keyphrase; fallback to Yoast SEO title/description;
     *   final fallback to post title/excerpt/content.
     * - Respects plugin settings toggles.
     */
    function aialttext_token_build_context_for_post(int $post_id): string {
        $post = get_post($post_id);
        if (!$post || is_wp_error($post)) {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] aialttext_token_build_context_for_post: Invalid post ID ' . $post_id);
            }
            return '';
        }
    
        $parts = array();
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] Building context for post ID: ' . $post_id . ', post type: ' . $post->post_type);
        }
    
        // Always include a simple anchor for the AI
        $title = get_the_title($post);
        if (!empty($title)) {
            $parts[] = 'This image appears on: "' . wp_strip_all_tags($title) . '".';
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Added post title to context: ' . $title);
            }
        }
    
        // WooCommerce product context (preferred on product pages)
        $is_product = ($post->post_type === 'product');
        $woo_enabled = (int) get_option('aialttext_token_woocommerce_integration', 1) === 1;
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] WooCommerce check - is_product: ' . ($is_product ? 'yes' : 'no') . ', woo_enabled: ' . ($woo_enabled ? 'yes' : 'no'));
        }
    
        if ($is_product && class_exists('WooCommerce')) {
            // If the helper exists (already in this file), use it for full detail.
            if (function_exists('aialttext_token_get_woocommerce_context')) {
                $woo_ctx = aialttext_token_get_woocommerce_context($post_id);
                if (!empty($woo_ctx)) {
                    $parts[] = $woo_ctx;
                    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                        error_log('[AI Alt Text] Added WooCommerce context: ' . substr($woo_ctx, 0, 200));
                    }
                }
            } else {
                // Minimal fallback (title + descriptions)
                if (function_exists('wc_get_product')) {
                    $product = wc_get_product($post_id);
                    if ($product) {
                        $desc = $product->get_short_description();
                        if (empty($desc)) $desc = $product->get_description();
                        if (!empty($desc)) {
                            $parts[] = 'Product description: ' . wp_strip_all_tags($desc);
                            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                                error_log('[AI Alt Text] Added basic product description');
                            }
                        }
                    }
                }
            }
        }
    
        // SEO context for any post type (posts/pages/products), when enabled
        $seo_enabled = (int) get_option('aialttext_token_seo_integration', 1) === 1;
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] SEO integration enabled: ' . ($seo_enabled ? 'yes' : 'no'));
        }
    
        if ( $seo_enabled && ( ! $is_product || ! $woo_enabled ) ) {
            $focus = '';
            $seo_title = '';
            $seo_desc  = '';
    
            // Multi-SEO plugin support with detailed logging
$seo_plugin_detected = '';

// Yoast SEO
if (defined('WPSEO_VERSION')) {
    $seo_plugin_detected = 'Yoast';
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Yoast SEO detected, checking focus keyphrase');
    }
    
    // Focus keyphrase
    $focus = (string) get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
    
    // Fallbacks for Yoast variants
    if ($focus === '') {
        $maybe = (string) get_post_meta($post_id, 'yoast_focuskeyphrase', true);
        if ($maybe !== '') $focus = $maybe;
        
        if ($focus === '') {
            $multi = get_post_meta($post_id, '_yoast_wpseo_focuskeywords', true);
            if (!empty($multi) && is_string($multi) && strpos($multi, '{') === 0) {
                $decoded = json_decode($multi, true);
                if (is_array($decoded) && !empty($decoded[0]['keyword'])) {
                    $focus = (string) $decoded[0]['keyword'];
                }
            }
        }
    }
    
    $seo_title = (string) get_post_meta($post_id, '_yoast_wpseo_title', true);
    $seo_desc  = (string) get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
}
// RankMath SEO
elseif (class_exists('RankMath') || defined('RANK_MATH_VERSION')) {
    $seo_plugin_detected = 'RankMath';
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] RankMath SEO detected, checking focus keyphrase');
    }
    
    $focus = (string) get_post_meta($post_id, 'rank_math_focus_keyword', true);
    $seo_title = (string) get_post_meta($post_id, 'rank_math_title', true);
    $seo_desc = (string) get_post_meta($post_id, 'rank_math_description', true);
}
// All in One SEO (AIOSEO)
elseif (function_exists('aioseo') && aioseo()->core->db->start()) {
    $seo_plugin_detected = 'AIOSEO';
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] AIOSEO detected, checking focus keyphrase');
    }
    
    global $wpdb;
    $aioseo_data = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}aioseo_posts WHERE post_id = %d",
        $post_id
    ));
    
    if ($aioseo_data) {
        if (!empty($aioseo_data->keyphrases)) {
            $keyphrases = json_decode($aioseo_data->keyphrases, true);
            if (is_array($keyphrases) && !empty($keyphrases['focus']['keyphrase'])) {
                $focus = (string) $keyphrases['focus']['keyphrase'];
            }
        }
        $seo_title = !empty($aioseo_data->title) ? (string) $aioseo_data->title : '';
        $seo_desc = !empty($aioseo_data->description) ? (string) $aioseo_data->description : '';
    }
}
// SEOPress
elseif (function_exists('seopress_get_service')) {
    $seo_plugin_detected = 'SEOPress';
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] SEOPress detected, checking focus keyphrase');
    }
    
    $focus = (string) get_post_meta($post_id, '_seopress_analysis_target_kw', true);
    $seo_title = (string) get_post_meta($post_id, '_seopress_titles_title', true);
    $seo_desc = (string) get_post_meta($post_id, '_seopress_titles_desc', true);
}

if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG && $seo_plugin_detected) {
    error_log('[AI Alt Text] SEO plugin: ' . $seo_plugin_detected . ', focus keyphrase found: ' . ($focus ? $focus : 'none'));
}
    
            if ($focus !== '') {
                // Use the exact marker the server expects
                $parts[] = 'SEO_KEYPHRASE: ' . rtrim( wp_strip_all_tags($focus), " .\t\n\r\0\x0B" );
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] Added SEO_KEYPHRASE to context: ' . $focus);
                }
            }
            
            if ($seo_title !== '') {
                // Normalize to server-expected key
                $parts[] = 'SEO_TITLE: ' . wp_strip_all_tags($seo_title) . '.';
            }
            
            if ($seo_desc !== '') {
                // Normalize to server-expected key
                $parts[] = 'SEO_META: ' . wp_strip_all_tags($seo_desc) . '.';
            }
            
            // Give the model improved instructions for natural SEO integration
$has_focus = ($focus !== '');
if ($has_focus) {
    $parts[] = 'INSTRUCTION: Naturally incorporate the SEO_KEYPHRASE into the alt text when it accurately describes the image. Focus on creating descriptive, natural-sounding alt text that includes the keyphrase organically rather than forcing it in verbatim.';
} else {
    $parts[] = 'INSTRUCTION: No SEO_KEYPHRASE found; use SEO_TITLE and SEO_META as contextual guidance for creating natural, descriptive alt text.';
}
    
            // Fallbacks if Yoast fields are empty
            if ($focus === '' && $seo_title === '' && $seo_desc === '') {
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] No SEO data found, using fallback content');
                }
                
                // fallback to post excerpt or the first ~160 chars of content
                $fallback = '';
                if (!empty($post->post_excerpt)) {
                    $fallback = wp_strip_all_tags($post->post_excerpt);
                } elseif (!empty($post->post_content)) {
                    $fallback = wp_strip_all_tags(wp_trim_words($post->post_content, 30, ''));
                }
                if ($fallback !== '') {
                    $parts[] = 'Page summary: ' . $fallback . '.';
                }
            }
        }
        
        $final_context = trim(implode("\n", array_filter($parts)));
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] Final context for post ' . $post_id . ': ' . $final_context);
        }
    
        return $final_context;
    }    
}

// Media Library column indicator - REMOVED TO PREVENT DUPLICATES
// The main plugin file handles this functionality
// Metabox removed - auto-generation on upload now handles SEO and WooCommerce context automatically

/* ---------------------------------------
 * Enhanced tracking for better context detection  
 * --------------------------------------- */
add_action('admin_init', function() {
    // Classic/Gutenberg post editor
    if (is_admin() && isset($_GET['action'], $_GET['post']) && $_GET['action'] === 'edit') {
        $post_id = (int) $_GET['post'];
        if ($post_id > 0) {
            if (!session_id()) { @session_start(); }
            $_SESSION['aialttext_last_edited_post'] = $post_id;
            $_SESSION['aialttext_edit_timestamp'] = time();
            set_transient('aialttext_current_editing_post_' . get_current_user_id(), $post_id, HOUR_IN_SECONDS);
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] TRACKING (classic/gutenberg): post '.$post_id);
            }
        }
    }

    // Keep transient fresh when a screen is present
    if (is_admin() && function_exists('get_current_screen')) {
        $screen = get_current_screen();
        if ($screen && isset($_GET['post'])) {
            $post_id = (int) $_GET['post'];
            if ($post_id > 0) {
                set_transient('aialttext_current_editing_post_' . get_current_user_id(), $post_id, HOUR_IN_SECONDS);
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] TRACKING (screen): post '.$post_id.' on '.$screen->id);
                }
            }
        }
    }

    // WooCommerce block product editor (wc-admin) – parse product ID from 'path' (/product/123)
    if (is_admin() && isset($_GET['page'], $_GET['path']) && $_GET['page'] === 'wc-admin') {
        $decoded = rawurldecode((string) $_GET['path']); // e.g. /product/360
        if (preg_match('#/product/([0-9]+)#', $decoded, $m)) {
            $post_id = (int) $m[1];
            if ($post_id > 0) {
                if (!session_id()) { @session_start(); }
                $_SESSION['aialttext_last_edited_post'] = $post_id;
                $_SESSION['aialttext_edit_timestamp'] = time();
                set_transient('aialttext_current_editing_post_' . get_current_user_id(), $post_id, HOUR_IN_SECONDS);
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] TRACKING (wc-admin): product '.$post_id.' from path='.$decoded);
                }
            }
        }
    }
});


/* ---------------------------------------
 * Enhanced JavaScript for admin pages to pass context
 * --------------------------------------- */
add_action('admin_enqueue_scripts', function($hook_suffix) {
    $post_id = 0;

    // Classic/Gutenberg
    if (in_array($hook_suffix, array('post.php','post-new.php'), true) && isset($_GET['post'])) {
        $post_id = (int) $_GET['post'];
    }
    // WooCommerce block product editor (wc-admin)
    elseif (
        ( $hook_suffix === 'woocommerce_page_wc-admin' || $hook_suffix === 'toplevel_page_wc-admin' )
        && isset($_GET['path'])
    ) {
        // wc-admin product editor, e.g. /wp-admin/admin.php?page=wc-admin&path=/product/360 or /product/360/edit
        $decoded = rawurldecode((string) $_GET['path']);
        if (preg_match('#/product/([0-9]+)#', $decoded, $m)) {
            $post_id = (int) $m[1];
        }
    }
    if ($post_id > 0) {
        if (function_exists('wp_enqueue_media')) {
            wp_enqueue_media();
        }

        // Primary injection via media-views when available
        wp_add_inline_script('media-views', "
            window.aialttext_current_post_id = {$post_id};
            (function(){
                if (typeof wp !== 'undefined' && wp) {
                    if (wp.media && wp.media.events) {
                        wp.media.events.on('open', function(){ window.aialttext_current_post_id = {$post_id}; });
                    }
                    if (wp.hooks && typeof wp.hooks.addFilter === 'function') {
                        wp.hooks.addFilter('media.upload.params','aialttext-token/add-post-context',function(params){
                            params = params || {};
                            params.post = {$post_id};
                            params.context_post_id = {$post_id};
                            return params;
                        });
                    }
                }
            })();
        ");

        // Footer fallback for wc-admin when media-views isn’t present
        add_action('admin_print_footer_scripts', function() use ($post_id) {
            echo '<script>
                window.aialttext_current_post_id = '.intval($post_id).';
                (function(){
                    if (typeof wp !== "undefined" && wp && wp.hooks && typeof wp.hooks.addFilter === "function") {
                        wp.hooks.addFilter("media.upload.params","aialttext-token/add-post-context-fallback", function(params){
                            params = params || {};
                            params.post = '.intval($post_id).';
                            params.context_post_id = '.intval($post_id).';
                            return params;
                        });
                    }
                })();
            </script>';
        }, 99);
    }
});



/* ---------------------------------------
 * Additional Gutenberg/Block Editor Support
 * --------------------------------------- */
add_action('enqueue_block_editor_assets', function() {
    // Only if auto-generation is enabled
    if (!get_option('aialttext_token_auto_generate', 0)) {
        return;
    }
    
    global $post;
    if ($post && $post->ID) {
        wp_add_inline_script('wp-editor', "
            // Store current post ID for Gutenberg media uploads
            window.aialttext_current_post_id = {$post->ID};
            
            // Hook into media upload for Gutenberg
            if (typeof wp !== 'undefined' && wp.media) {
                wp.hooks.addFilter('media.upload.params', 'aialttext/add-post-context', function(params) {
                    params.post = {$post->ID};
                    return params;
                });
            }
        ");
    }
});

/* ---------------------------------------
 * Clean up expired session data
 * --------------------------------------- */
add_action('wp_loaded', function() {
    if (!session_id()) {
        session_start();
    }
    
    // Clean up old edit session data (older than 1 hour)
    if (isset($_SESSION['aialttext_edit_timestamp']) && 
        (time() - $_SESSION['aialttext_edit_timestamp']) > 3600) {
        unset($_SESSION['aialttext_last_edited_post']);
        unset($_SESSION['aialttext_edit_timestamp']);
    }
});
