<?php
if (!defined('ABSPATH')) exit;

/**
 * Build lightweight context for an attachment to help the model.
 * Safe on any site: no Woo/SEO hard dependencies.
 */
if (!function_exists('aialttext_token_build_context_for_image')) {
    function aialttext_token_build_context_for_image(int $attachment_id): string {
        $bits = [];

        // Parent post context
        $parent_id = (int) wp_get_post_parent_id($attachment_id);
        if ($parent_id) {
            $parent = get_post($parent_id);
            if ($parent && !is_wp_error($parent)) {
                $title = wp_strip_all_tags($parent->post_title);
                if ($title !== '') {
                    $bits[] = 'Parent title: ' . $title;
                }

                // Common taxonomies if present
                $taxes = array('category','post_tag','product_cat','product_tag');
                foreach ($taxes as $tx) {
                    $terms = get_the_terms($parent_id, $tx);
                    if (!empty($terms) && !is_wp_error($terms)) {
                        $names = array_filter(array_map('wp_strip_all_tags', wp_list_pluck($terms, 'name')));
                        if ($names) {
                            $bits[] = ucfirst(str_replace('_',' ',$tx)) . ': ' . implode(', ', $names);
                        }
                    }
                }
            }
        }

        // Attachment title & filename
        $title = get_the_title($attachment_id);
        if (!empty($title)) {
            $bits[] = 'Attachment title: ' . wp_strip_all_tags($title);
        }
        $url = wp_get_attachment_url($attachment_id);
        if ($url) {
            $path = parse_url($url, PHP_URL_PATH);
            if ($path) $bits[] = 'Filename: ' . basename($path);
        }

        // Site hint
        $sitename = get_bloginfo('name');
        if ($sitename) $bits[] = 'Site: ' . wp_strip_all_tags($sitename);

        return implode(' | ', array_filter($bits));
    }
}

/**
 * Enhanced context detection function for image uploads
 * Detects the post/page/product context from various sources
 */
function aialttext_token_detect_upload_context($attachment_id) {
    $context_post_id = 0;

    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Starting context detection for attachment ' . $attachment_id);
    }

    // 1) Standard params (we also accept our own 'context_post_id')
    $param_candidates = array('post_id','post_ID','post','parent','uploadedTo','uploaded_to','context_post_id');
    foreach ($param_candidates as $k) {
        if (isset($_REQUEST[$k])) {
            $maybe = (int) $_REQUEST[$k];
            if ($maybe > 0) {
                $context_post_id = $maybe;
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] Found context from $_REQUEST['.$k.']: '.$context_post_id);
                }
                break;
            }
        }
    }

    // 2) Attachment parent if it’s already set
    if ($context_post_id <= 0) {
        $parent = (int) wp_get_post_parent_id($attachment_id);
        if ($parent > 0) {
            $context_post_id = $parent;
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from attachment parent: ' . $context_post_id);
            }
        }
    }

    // 3) Referer patterns — Classic, Gutenberg, and WooCommerce wc-admin
    if ($context_post_id <= 0 && !empty($_SERVER['HTTP_REFERER'])) {
        $referer = (string) $_SERVER['HTTP_REFERER'];

        // Classic/Gutenberg edit (?post=123)
        if (preg_match('/[?&]post=([0-9]+)/', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from referer (?post=): '.$context_post_id);
            }
        }
        // Explicit post.php?post=123
        elseif (preg_match('#post\.php.*[?&]post=([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from referer (post.php): '.$context_post_id);
            }
        }
        // Gutenberg edit for other types
        elseif (preg_match('#wp-admin/(post|page|product)\.php.*[?&]post=([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[2];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from referer (gutenberg): '.$context_post_id);
            }
        }
        // WooCommerce block product editor (wc-admin), encoded path
        elseif (preg_match('#[?&]page=wc-admin[^#]*[?&]path=(?:%2F|/)product%2F([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from wc-admin (encoded path): '.$context_post_id);
            }
        }
        // WooCommerce block product editor (wc-admin), unencoded path
        elseif (preg_match('#[?&]path=(?:%2F|/)product/([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from wc-admin (path): '.$context_post_id);
            }
        }
        // wc-admin hash/other shapes
        elseif (preg_match('#wc-admin.+(?:%2F|/)?product(?:%2F|/)([0-9]+)#', $referer, $m)) {
            $context_post_id = (int) $m[1];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from wc-admin (hash): '.$context_post_id);
            }
        }
    }

    // 4) Current screen (admin)
    if ($context_post_id <= 0 && is_admin()) {
        global $post;
        if ($post && is_object($post) && $post->ID) {
            $context_post_id = (int) $post->ID;
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from global $post: ' . $context_post_id);
            }
        }
    }

    // 5) Session fallback
    if ($context_post_id <= 0) {
        if (!session_id()) { @session_start(); }
        if (isset($_SESSION['aialttext_last_edited_post'])) {
            $context_post_id = (int) $_SESSION['aialttext_last_edited_post'];
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from session: ' . $context_post_id);
            }
        }
    }

    // 6) Transient per user
    if ($context_post_id <= 0 && function_exists('get_current_user_id')) {
        $t = get_transient('aialttext_current_editing_post_' . get_current_user_id());
        if ($t) {
            $context_post_id = (int) $t;
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Found context from transient: ' . $context_post_id);
            }
        }
    }

    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Final context detection result: ' . ($context_post_id ?: 'none'));
    }
    return $context_post_id;
}


/**
 * Core image processor using the token-based API client.
 */
class AiAltText_Token_Image_Processor {
    /**
     * @param int   $attachment_id
     * @param array $args ['context' => string, 'mime' => string, 'post_id' => int]
     * @return array|WP_Error array('alt' => '...') on success
     */
    public static function process_attachment($attachment_id, $args = array()) {
        $attachment_id = (int) $attachment_id;

        // Validate attachment
        $post = get_post($attachment_id);
        if (!$post || $post->post_type !== 'attachment') {
            return new WP_Error('invalid_attachment', 'Invalid attachment.');
        }
        $mime = !empty($args['mime']) ? (string) $args['mime'] : get_post_mime_type($attachment_id);
        if (strpos((string) $mime, 'image/') !== 0) {
            return new WP_Error('not_image', 'Attachment is not an image.');
        }

        // URL & title
        $url = wp_get_attachment_url($attachment_id);
        if (!$url) return new WP_Error('no_url', 'Missing attachment URL');

        $title = get_the_title($attachment_id);
        if (!$title) {
            $path = parse_url($url, PHP_URL_PATH);
            $title = $path ? basename($path) : ('attachment-' . $attachment_id);
        }

        // SVG: try to rasterize locally for richer AI results; otherwise continue.
        // The account service can also handle SVG (Imagick/data-URL or text-only).
        if ($mime === 'image/svg+xml') {
            $maybe_png = aialttext_token_rasterize_svg_to_png_url($url);
            if ($maybe_png) {
                $url  = $maybe_png;          // hand AI a real bitmap
                $mime = 'image/png';
            }
            // If rasterization fails, keep $url as-is; the account service will still produce AI alt text.
        }

        // Enhanced context building with better SEO/WooCommerce integration
        $context = '';
        
        // Start with basic context if provided
        if (isset($args['context']) && is_string($args['context'])) {
            $context = $args['context'];
        } else {
            $context = aialttext_token_build_context_for_image($attachment_id);
        }
        
        // Get the parent post ID with enhanced detection
        $parent_id = !empty($args['post_id']) ? (int) $args['post_id'] : (int) wp_get_post_parent_id($attachment_id);
        
        // Try the more robust detector (covers Classic, Gutenberg, media modal, and product editor)
        if ($parent_id <= 0 && function_exists('aialttext_token_detect_current_post_id')) {
            $parent_id = (int) aialttext_token_detect_current_post_id($attachment_id);
        }
        // Fallback to the upload-context heuristics
        if ($parent_id <= 0) {
            $parent_id = aialttext_token_detect_upload_context($attachment_id);
        }
        
        // Build rich context with SEO and WooCommerce data
        if ($parent_id > 0 && function_exists('aialttext_token_build_context_for_post')) {
            $rich_context = aialttext_token_build_context_for_post($parent_id);
            if (!empty($rich_context)) {
                $context = trim($context . ' ' . $rich_context);
            }
        }
        
        // Debug logging
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log(sprintf(
                '[AI Alt Text] Processing attachment %d, parent %d, final context length: %d chars',
                $attachment_id,
                $parent_id,
                strlen($context)
            ));
            error_log('[AI Alt Text] Final context being sent to API: ' . $context);
        }

        // Call token API
        if (!class_exists('AiAltText_Token_API_Client')) {
            return new WP_Error('missing_client', 'API client not available.');
        }
        
        $api = new AiAltText_Token_API_Client();
        
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] About to call API with: URL=' . $url . ', Title=' . $title . ', Context=' . substr($context, 0, 200) . (strlen($context) > 200 ? '...' : ''));
        }

// Prefer parent post/product title as the "page title" for the AI
$page_title_for_api = $title;
if ($parent_id > 0) {
    $parent_title = get_the_title($parent_id);
    if (!empty($parent_title)) {
        $page_title_for_api = $parent_title;
    }
}

// CRITICAL: Prefer sending bytes so Gemini/Claude can work even if the admin server can't fetch client media
// This is essential for the tier system to work correctly - Gemini REQUIRES inline bytes
$image_b64 = null;
$image_mime = null;

$max_bytes = (int) apply_filters('aialttext_token_max_inline_bytes', 15 * 1024 * 1024); // Increased from 10MB to 15MB

$file_path = get_attached_file($attachment_id);
$bytes = false;

if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
    error_log('[AI Alt Text] Starting to fetch image bytes for attachment ' . $attachment_id);
    error_log('[AI Alt Text] File path: ' . ($file_path ?: 'none'));
    error_log('[AI Alt Text] Max bytes: ' . $max_bytes);
}

// 1) Try local file first
if (is_string($file_path) && file_exists($file_path) && is_readable($file_path)) {
    $size = @filesize($file_path);
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Local file size: ' . ($size !== false ? $size : 'unknown'));
    }
    
    if ($size !== false && $size <= $max_bytes) {
        $bytes = @file_get_contents($file_path);
        if ($bytes !== false) {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Successfully read local file: ' . strlen($bytes) . ' bytes');
            }
        }
    } elseif ($size !== false && $size > $max_bytes && function_exists('wp_get_image_editor')) {
        // File too large - resize it
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] File too large (' . $size . ' bytes), attempting to resize');
        }
        
        $tmp = wp_tempnam($file_path);
        if ($tmp) {
            @copy($file_path, $tmp);
            $ed = wp_get_image_editor($tmp);
            if (!is_wp_error($ed)) {
                $ed->resize(1600, 1600, false);
                $saved = $ed->save($tmp);
                if (!is_wp_error($saved) && !empty($saved['path'])) {
                    $bytes = @file_get_contents($saved['path']);
                    // Prefer saved mime if supplied
                    if (!empty($saved['mime-type'])) {
                        $image_mime = $saved['mime-type'];
                    }
                    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                        error_log('[AI Alt Text] Successfully resized large file to: ' . strlen($bytes) . ' bytes');
                    }
                }
            }
            @unlink($tmp);
        }
    }
}

// 2) CRITICAL fallback: If local file failed, fetch from public URL
// This is essential for Gemini tiers to work
if ($bytes === false || $bytes === null || $bytes === '') {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Local file read failed, attempting to fetch from URL: ' . $url);
    }
    
    $remote = wp_remote_get($url, array(
        'timeout'     => 20, // Increased timeout
        'redirection' => 3,
        'sslverify'   => true,
        'headers'     => array('Accept' => 'image/*'),
    ));
    
    if (!is_wp_error($remote)) {
        $ctype = wp_remote_retrieve_header($remote, 'content-type');
        $body  = wp_remote_retrieve_body($remote);

        if (is_string($body) && $body !== '') {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] Successfully fetched from URL: ' . strlen($body) . ' bytes');
            }
            
            // Keep under cap; if too big, try re-encode/resize
            if (strlen($body) > $max_bytes && function_exists('wp_get_image_editor')) {
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] Downloaded image too large, attempting to resize');
                }
                
                $tmp = wp_tempnam($url);
                if ($tmp) {
                    @file_put_contents($tmp, $body);
                    $ed = wp_get_image_editor($tmp);
                    if (!is_wp_error($ed)) {
                        $ed->resize(1600, 1600, false);
                        $saved = $ed->save($tmp);
                        if (!is_wp_error($saved) && !empty($saved['path'])) {
                            $body = @file_get_contents($saved['path']);
                            if (!empty($saved['mime-type'])) {
                                $ctype = $saved['mime-type'];
                            }
                            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                                error_log('[AI Alt Text] Successfully resized downloaded image to: ' . strlen($body) . ' bytes');
                            }
                        }
                    }
                    @unlink($tmp);
                }
            }
            
            if (is_string($body) && strlen($body) <= $max_bytes) {
                $bytes = $body;
                if (!$image_mime) {
                    $image_mime = $ctype ?: get_post_mime_type($attachment_id);
                    if (!$image_mime) {
                        $ft = wp_check_filetype(parse_url($url, PHP_URL_PATH));
                        if (!empty($ft['type'])) $image_mime = $ft['type'];
                    }
                    if (!$image_mime) $image_mime = 'image/jpeg';
                }
            }
        }
    } else {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] Failed to fetch from URL: ' . $remote->get_error_message());
        }
    }
}

// 3) Finalize base64 if we have bytes
if ($bytes !== false && $bytes !== null && $bytes !== '') {
    $image_b64 = base64_encode($bytes);
    if (!$image_mime) {
        $image_mime = get_post_mime_type($attachment_id);
        if (!$image_mime) {
            $fp = $file_path ?: parse_url($url, PHP_URL_PATH);
            $ft = $fp ? wp_check_filetype($fp) : array();
            if (!empty($ft['type'])) $image_mime = $ft['type'];
        }
        if (!$image_mime) $image_mime = 'image/jpeg';
    }
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] Successfully prepared image bytes for API: ' . strlen($image_b64) . ' base64 chars, mime: ' . $image_mime);
    }
} else {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] WARNING: Could not get image bytes for attachment ' . $attachment_id . ' - Gemini tiers may be skipped on the server!');
    }
}

// Send bytes when available; fall back to URL-only (will skip Gemini tiers if no bytes)
if ($image_b64 && $image_mime) {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] ✓ Sending API request WITH image bytes');
        error_log('[AI Alt Text]   - Base64 length: ' . strlen($image_b64) . ' chars');
        error_log('[AI Alt Text]   - MIME type: ' . $image_mime);
        error_log('[AI Alt Text]   - URL: ' . $url);
    }
    $result = $api->generate($url, $page_title_for_api, $context, $image_b64, $image_mime);
    
    // Log the API response
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        if (!empty($result['debug_logs'])) {
            error_log('[AI Alt Text] Debug logs from server:');
            foreach ($result['debug_logs'] as $log) {
                error_log('[AI Alt Text]   - ' . $log['message']);
                if (!empty($log['data'])) {
                    error_log('[AI Alt Text]     Data: ' . print_r($log['data'], true));
                }
            }
        }
    }
} else {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] ⚠ WARNING: Sending API request WITHOUT image bytes - Gemini will be skipped!');
        error_log('[AI Alt Text]   - has_b64: ' . (is_string($image_b64) && $image_b64 !== '' ? 'yes' : 'NO'));
        error_log('[AI Alt Text]   - has_mime: ' . (is_string($image_mime) && $image_mime !== '' ? 'yes' : 'NO'));
    }
    $result = $api->generate($url, $page_title_for_api, $context);
}


if (empty($result) || empty($result['success'])) {
    $msg = isset($result['error']['message']) ? (string) $result['error']['message'] : 'API error';
    $http_code = isset($result['error']['code']) ? (int) $result['error']['code'] : 0;

    // Normalize token exhaustion so batch processing can reliably detect it
    if ($http_code === 402 || stripos($msg, 'insufficient') !== false) {
        return new WP_Error('insufficient_tokens', ($msg !== '' ? $msg : 'Insufficient tokens'));
    }
    return new WP_Error('api_error', $msg !== '' ? $msg : 'API error');
}

        $alt = isset($result['alt_text']) ? trim((string) $result['alt_text']) : '';
        if ($alt === '') {
            return new WP_Error('empty_alt', 'No alt text returned.');
        }

        // Persist
        update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);

        return array('alt' => $alt);
    }
}

/**
 * If possible, rasterize an SVG URL to a temporary PNG in uploads and return its public URL.
 * Returns null on failure (no Imagick, download error, or no delegate).
 */
function aialttext_token_rasterize_svg_to_png_url(string $svg_url): ?string {
    if (!class_exists('Imagick')) return null;

    $res = wp_remote_get($svg_url);
    if (is_wp_error($res)) return null;
    $svg = wp_remote_retrieve_body($res);
    if (!$svg) return null;

    try {
        $im = new Imagick();
        $im->setBackgroundColor(new ImagickPixel('transparent'));
        $im->readImageBlob($svg);           // needs librsvg/inkscape delegate on server
        $im->setImageFormat('png32');
        $blob = $im->getImageBlob();

        $upload = wp_upload_dir();
        $dir    = trailingslashit($upload['basedir']) . 'aialttext-svg-cache/';
        if (!file_exists($dir)) wp_mkdir_p($dir);

        $fname = 'svg-' . md5($svg_url) . '.png';
        $path  = $dir . $fname;
        file_put_contents($path, $blob);

        return trailingslashit($upload['baseurl']) . 'aialttext-svg-cache/' . $fname;
    } catch (Throwable $e) {
        return null;
    }
}

/**
 * Scan next slice of the media library for images lacking alt text.
 * Supports excluding IDs we already attempted in this run.
 */
function aialttext_token_scan_media_library(int $batch_size = 10, int $offset = 0, array $exclude_ids = array()) {
    // IMPORTANT: here $offset is treated as a CURSOR (the last processed attachment ID),
    // not an SQL OFFSET. This avoids skipping items when earlier ones drop out after being processed.
    global $wpdb;

    $exclude_ids = array_filter(array_map('intval', (array) $exclude_ids));
    $exclude_sql = $exclude_ids ? ' AND p.ID NOT IN (' . implode(',', $exclude_ids) . ')' : '';

    $sql = $wpdb->prepare(
        "SELECT p.ID, p.post_title, p.guid, p.post_mime_type
         FROM $wpdb->posts p
         WHERE p.post_type = 'attachment'
           AND p.post_mime_type IN ('image/jpeg','image/png','image/gif','image/webp','image/svg+xml')
           $exclude_sql
           AND p.ID > %d
           AND p.ID NOT IN (
               SELECT post_id FROM $wpdb->postmeta
               WHERE meta_key = '_wp_attachment_image_alt' AND meta_value != ''
           )
         ORDER BY p.ID ASC
         LIMIT %d",
        $offset, $batch_size
    );

    return $wpdb->get_results($sql);
}


/** Simple helper to update alt text directly. */
function aialttext_token_update_alt_text($image_id, $alt_text) {
    return update_post_meta((int) $image_id, '_wp_attachment_image_alt', (string) $alt_text);
}

// Treat timeouts and temporary upstream issues as transient (worth a retry)
if (!function_exists('aialttext_token_is_transient_error')) {
    function aialttext_token_is_transient_error($message): bool {
        $m = strtolower((string)$message);
        $needles = array(
            'timeout',                 // “Timeout while downloading …”, “request timed out”
            'timed out',
            'operation timed out',
            'connection timed out',
            'cURL error 28',           // WP HTTP API cURL timeout
            '429',                     // rate limits
            '502', '503', '504',       // gateway / service interruptions
            'temporarily unavailable',
            'rate limit'
        );
        foreach ($needles as $n) {
            if (strpos($m, $n) !== false) return true;
        }
        return false;
    }
}


/**
 * Legacy processor (kept for compatibility). Uses the newer processor internally.
 * NOTE: The current UI calls aialttext_token_process_images_with_params().
 */
function aialttext_token_process_images($offset, $batch_size) {
    $res = aialttext_token_process_images_with_params((int) $offset, (int) $batch_size, 0, array());
if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
    error_log(sprintf('[AI Alt Text] run summary: processed=%d last_id=%s next_offset=%d insufficient_tokens=%s',
        (int) (isset($res['processed']) ? $res['processed'] : 0),
        !empty($res['processed_ids']) ? max(array_map('intval', (array) $res['processed_ids'])) : 'none',
        (int) (isset($res['offset']) ? $res['offset'] : 0),
        !empty($res['insufficient_tokens']) ? 'yes' : 'no'
    ));
}

    
    // Map to legacy response shape
    return array(
        'success'        => $res['success'],
        'processed'      => $res['processed'],
        'errors'         => $res['errors'],
        'error_messages' => $res['error_messages'],
        'offset'         => $res['offset'],
        'total_images'   => $res['total_images'],
        'progress'       => $res['progress'],
        'completed'      => $res['completed'],
    );
}

/** Count images missing alt text. */
function aialttext_token_get_total_images_without_alt() {
    global $wpdb;
    return (int) $wpdb->get_var(
        "SELECT COUNT(p.ID)
         FROM $wpdb->posts p
         LEFT JOIN $wpdb->postmeta pm
           ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
         WHERE p.post_type = 'attachment'
           AND p.post_mime_type IN ('image/jpeg','image/png','image/gif','image/webp','image/svg+xml')
           AND (pm.meta_value IS NULL OR pm.meta_value = '')"
    );
}

/**
 * Batch processor with exclude list and rich return payload.
 * Returns:
 *  - processed (int), processed_ids (int[]), errors (int), failed ([['id'=>int,'message'=>string], ...])
 *  - error_messages (string[]), offset (int), total_images (int), completed (bool), latest_image (array|null)
 */
function aialttext_token_process_images_with_params(int $offset, int $batch_size, int $max_images = 0, array $exclude_ids = array()) {
    $offset      = max(0, $offset);
    $batch_size  = max(1, $batch_size);
    $max_images  = max(0, $max_images);
    $exclude_ids = array_filter(array_map('intval', (array) $exclude_ids));

    // Fetch next candidates (front of the queue)
    $images = aialttext_token_scan_media_library($batch_size, $offset, $exclude_ids);

    $processed      = 0;
    $processed_ids  = array();
    $processed_details = array(); // NEW: id+alt+url+title for UI
    $errors         = 0;
    $failed         = array();   // each: ['id'=>int, 'message'=>string]
    $error_messages = array();
    $latest_image   = null;
    $out_of_tokens  = false; // stop the batch immediately when balance hits zero

// Process up to the requested batch size exactly.
// We intentionally avoid time-slicing so the UI’s batch size is honored.
// Hosts with strict timeouts will still be protected by the outer AJAX loop.

foreach ($images as $img) {
        $id   = (int) $img->ID;
        $mime = (string) $img->post_mime_type;

        try {
            $context      = aialttext_token_build_context_for_image($id);
            $max_attempts = 3;
            $attempts     = 0;
            $result       = null;
            $last_error   = '';
        
            while ($attempts < $max_attempts) {
                $attempts++;
        
                $result = AiAltText_Token_Image_Processor::process_attachment($id, array(
                    'context' => $context,
                    'mime'    => $mime,
                ));
        
                // Success
                if (!is_wp_error($result)) {
                    break;
                }
        
                // Error path
                $last_error = $result->get_error_message();
        
                // If we hit a token exhaustion condition, mark and stop immediately (not transient)
// Prefer structured signal (our normalized WP_Error code), then fall back to message/HTTP hints.
$err_code = ($result instanceof WP_Error) ? $result->get_error_code() : '';
if (
    $err_code === 'insufficient_tokens'
    || preg_match('/\b402\b/', (string) $last_error)            // e.g., "HTTP 402"
    || (stripos($last_error, 'insufficient') !== false && stripos($last_error, 'token') !== false)
) {
    $out_of_tokens = true;
    break;
}

        
                // Retry only on transient network-ish issues
                if (!aialttext_token_is_transient_error($last_error)) {
                    break;
                }
        
                // simple back-off: ~250ms, ~500ms
                usleep(250000 * $attempts);
            }
        
            if (!is_wp_error($result)) {
                $processed++;
                $processed_ids[] = $id;
            
                // NEW: collect details per processed image (safe, additive)
                $edit_link = get_edit_post_link($id, 'raw');
if (!$edit_link) {
    $edit_link = admin_url('post.php?post=' . $id . '&action=edit');
}

$processed_details[] = array(
    'id'        => $id,
    'title'     => get_the_title($id),
    'alt'       => isset($result['alt']) ? $result['alt'] : '',
    'url'       => wp_get_attachment_url($id),
    'edit_link' => $edit_link,
);
            
                if (!$latest_image && !empty($result['alt'])) {
                    $latest_image = array(
                        'id'        => $id,
                        'title'     => get_the_title($id),
                        'alt_text'  => $result['alt'],
                        'url'       => wp_get_attachment_url($id),
                        'edit_link' => get_edit_post_link($id, 'raw') ?: admin_url('post.php?post=' . $id . '&action=edit'),
                    );
                }
            } else {
                $msg = $last_error !== '' ? $last_error : $result->get_error_message();
        
                // Record the failure once, annotating attempts so the UI error panel is informative
                $failed[] = array(
                    'id'      => $id,
                    'message' => ($attempts > 1 ? "[{$attempts} attempts] " : "") . $msg,
                );
                $error_messages[] = "Failed to generate alt text for image {$id}: {$msg}";
        
                if ($out_of_tokens) {
                    break; // whole batch pauses for top-up
                }
            }
        } catch (\Throwable $e) {
            $errors++;
            $failed[] = array('id' => $id, 'message' => $e->getMessage());
            $error_messages[] = "Error processing image {$id}: " . $e->getMessage();
        }
        
    }

    $total_images = aialttext_token_get_total_images_without_alt();

    $empty_slice = empty($images);

    // Defaults: assume not completed and keep current cursor
    $completed   = false;
    $next_offset = $offset;

    // If we processed anything, advance cursor to the last processed ID
    if (!empty($processed_ids)) {
        $next_offset = (int) max(array_map('intval', (array) $processed_ids));
    }

    // If we ran out of tokens, pause exactly where we are (overrides advance)
    if (!empty($out_of_tokens)) {
        $next_offset = $offset;
    }

    // If this slice had no candidates, decide whether to wrap or finish
    if ($empty_slice) {
        if ((int) $total_images > 0) {
            // Work remains elsewhere (e.g., after wrapping or excluding attempted IDs)
            if (count($exclude_ids) >= (int) $total_images) {
                // Everything remaining is excluded for this run — stop politely
                $completed   = true;
                $next_offset = 0;
            } else {
                // Wrap to the front to continue scanning
                $next_offset = 0;
                // completed stays false
            }
        } else {
            // Truly done
            $completed   = true;
            $next_offset = 0;
        }
    }

    return array(
        'success'              => true,
        'processed'            => (int) $processed,
        'processed_ids'        => array_values(array_map('intval', (array) $processed_ids)),
        'processed_details'    => $processed_details, // NEW (id,title,alt,url)
        'errors'               => (int) $errors,
        'failed'               => $failed,
        'error_messages'       => $error_messages,
        'offset'               => (int) $next_offset,
        'total_images'         => (int) $total_images,
        'progress'             => 0,         // UI computes %
        'completed'            => (bool) $completed,
        'latest_image'         => $latest_image ?: null,
        'insufficient_tokens'  => !empty($out_of_tokens),
        'top_up_url'           => admin_url('admin.php?page=aialttext-token'),
    );
}

/* ---------------------------------------
 * Debug and test endpoints
 * --------------------------------------- */

// Add debug endpoint for testing context detection
add_action('wp_ajax_aialttext_token_test_context', function() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'Permission denied'));
    }
    
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }
    
    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
    
    if (!$post_id) {
        wp_send_json_error(array('message' => 'Post ID required'));
    }
    
    $results = array();
    
    // Test basic context detection
    if ($attachment_id) {
        $detected_context = aialttext_token_detect_upload_context($attachment_id);
        $results['detected_context'] = $detected_context;
    }
    
    // Test post context building
    $post_context = '';
    if (function_exists('aialttext_token_build_context_for_post')) {
        $post_context = aialttext_token_build_context_for_post($post_id);
    }
    $results['post_context'] = $post_context;
    
    // Test SEO context
    $seo_context = '';
    if (function_exists('aialttext_token_get_seo_context')) {
        $seo_context = aialttext_token_get_seo_context($post_id);
    }
    $results['seo_context'] = $seo_context;
    
    // Test WooCommerce context
    $woo_context = '';
    if (function_exists('aialttext_token_get_woocommerce_context')) {
        $woo_context = aialttext_token_get_woocommerce_context($post_id);
    }
    $results['woo_context'] = $woo_context;
    
    // Get post details
    $post = get_post($post_id);
    if ($post) {
        $results['post_details'] = array(
            'ID' => $post->ID,
            'post_type' => $post->post_type,
            'post_title' => $post->post_title,
            'post_status' => $post->post_status
        );
    } else {
        $results['post_details'] = array('error' => 'Post not found');
    }
    
    // Check settings
    $results['settings'] = array(
        'woocommerce_integration' => get_option('aialttext_token_woocommerce_integration', 1),
        'seo_integration' => get_option('aialttext_token_seo_integration', 1),
        'auto_generate' => get_option('aialttext_token_auto_generate', 0),
        'woocommerce_active' => class_exists('WooCommerce'),
        'yoast_active' => defined('WPSEO_VERSION')
    );
    
    // Check if Yoast has focus keyphrase for this post
    if (defined('WPSEO_VERSION')) {
        $yoast_focus = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        $results['yoast_focus_keyphrase'] = $yoast_focus;
    }
    
    wp_send_json_success($results);
});

/* ---------------------------------------
 * Enhanced Auto-generate on upload (respects setting)
 * Improved context detection for SEO and WooCommerce integration
 * --------------------------------------- */
add_action('add_attachment', function($attachment_id){
    if (!get_option('aialttext_token_auto_generate', 0)) return;
    if (get_post_type($attachment_id) !== 'attachment') return;
    $mime = get_post_mime_type($attachment_id);
    if (strpos((string) $mime, 'image/') !== 0) return;

    // Enhanced context detection for better SEO and WooCommerce integration
    $context_post_id = 0;
if (function_exists('aialttext_token_detect_current_post_id')) {
    $context_post_id = (int) aialttext_token_detect_current_post_id($attachment_id);
}
if ($context_post_id <= 0) {
    $context_post_id = aialttext_token_detect_upload_context($attachment_id);
}

    
    // Enhanced debugging for upload context
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] UPLOAD EVENT - attachment: ' . $attachment_id . ', detected context: ' . ($context_post_id ?: 'none'));
        error_log('[AI Alt Text] UPLOAD $_REQUEST data: ' . print_r($_REQUEST, true));
        error_log('[AI Alt Text] UPLOAD HTTP_REFERER: ' . (isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'none'));
    }

    $args = array('mime' => $mime);
    if ($context_post_id > 0) {
        $args['post_id'] = $context_post_id;
        
        // Log context detection for debugging
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log(sprintf('AI Alt Text: Detected context post %d for attachment %d', $context_post_id, $attachment_id));
            
            // Test if we can build context for this post
            if (function_exists('aialttext_token_build_context_for_post')) {
                $test_context = aialttext_token_build_context_for_post($context_post_id);
                error_log('[AI Alt Text] UPLOAD Test context build result: ' . $test_context);
            }
        }
    } else {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] UPLOAD WARNING: No context detected for attachment ' . $attachment_id);
        }
    }
    // If an alt exists but we're now on a product, allow overwrite to add product context
if (!empty(get_post_meta($attachment_id, '_wp_attachment_image_alt', true))) {
    if ($context_post_id > 0 && get_post_type($context_post_id) === 'product') {
        delete_post_meta($attachment_id, '_wp_attachment_image_alt');
    }
}

    // Best-effort; ignore failures to avoid blocking uploads
    try {
        
        AiAltText_Token_Image_Processor::process_attachment($attachment_id, $args);
    } catch (\Throwable $e) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('AI Alt Text auto-generation error: ' . $e->getMessage());
        }
    }
}, 10, 1);

/* ---------------------------------------
 * Enhanced Auto-generate on REST media upload (Gutenberg, Woo block editor)
 * --------------------------------------- */
add_action('rest_after_insert_attachment', function($attachment, $request, $creating){
    // Only new uploads, respect the setting
    $auto = get_option('aialttext_token_auto_generate', 0);
    $enabled = in_array($auto, array(1, '1', true, 'true', 'on', 'yes'), true);
    if (!$creating || !$enabled) return;

    $attachment_id = (int) $attachment->ID;
    if (get_post_type($attachment_id) !== 'attachment') return;

    $mime = get_post_mime_type($attachment_id);
    if (strpos((string) $mime, 'image/') !== 0) return;

    // --- Detect context FIRST (so we can decide whether to overwrite existing alt) ---
    $context_post_id = 0;

    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] REST UPLOAD EVENT - attachment: ' . $attachment_id);
        if ($request) {
            $params = $request->get_params();
            error_log('[AI Alt Text] REST request params: ' . print_r($params, true));
        }
        error_log('[AI Alt Text] REST HTTP_REFERER: ' . (isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'none'));
    }

    if ($request) {
        foreach (array('post','parent','uploadedTo','uploaded_to','context_post_id') as $key) {
            $val = (int) $request->get_param($key);
            if ($val > 0) {
                $context_post_id = $val;
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log('[AI Alt Text] REST found context from request param "' . $key . '": ' . $context_post_id);
                }
                break;
            }
        }
        if (!$context_post_id) {
            $ctx = $request->get_param('context');
            if (is_array($ctx)) {
                foreach (array('post','postId','parent','productId') as $key) {
                    if (!empty($ctx[$key]) && (int) $ctx[$key] > 0) {
                        $context_post_id = (int) $ctx[$key];
                        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                            error_log('[AI Alt Text] REST found context from nested context.' . $key . ': ' . $context_post_id);
                        }
                        break;
                    }
                }
            }
        }
    }

    // Fallbacks
    if ($context_post_id <= 0 && function_exists('aialttext_token_detect_current_post_id')) {
        $context_post_id = (int) aialttext_token_detect_current_post_id($attachment_id);
    }
    if ($context_post_id <= 0) {
        $context_post_id = aialttext_token_detect_upload_context($attachment_id);
    }
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] REST fallback context detection result: ' . ($context_post_id ?: 'none'));
    }

    // --- Decide whether to keep or overwrite existing alt text ---
    $existing = (string) get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if ($existing !== '') {
        $allow_overwrite = false;

        // Always overwrite on product pages to inject product context
        if ($context_post_id > 0 && get_post_type($context_post_id) === 'product') {
            $allow_overwrite = true;
        }

        // Or overwrite if it's just a placeholder (filename/title/generic)
        if (!$allow_overwrite && function_exists('aialttext_token_is_placeholder_alt')) {
            $allow_overwrite = aialttext_token_is_placeholder_alt($attachment_id, $existing);
        }

        if ($allow_overwrite) {
            delete_post_meta($attachment_id, '_wp_attachment_image_alt');
        } else {
            // Respect user-provided alt
            return;
        }
    }

    // Build args for processor
    $args = array('mime' => $mime);
    if ($context_post_id > 0) {
        $args['post_id'] = $context_post_id;
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log(sprintf('AI Alt Text REST: Detected context post %d for attachment %d', $context_post_id, $attachment_id));
        }
    }

    try {
        AiAltText_Token_Image_Processor::process_attachment($attachment_id, $args);
    } catch (\Throwable $e) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('AI Alt Text REST auto-generation error: ' . $e->getMessage());
        }
    }
}, 10, 3);


/* ---------------------------------------
 * Delayed context retry for missed uploads
 * --------------------------------------- */
add_action('wp_generate_attachment_metadata', function($metadata, $attachment_id) {
    // This runs later in the upload process, good for catching missed uploads
    if (!get_option('aialttext_token_auto_generate', 0)) return $metadata;
    
    $mime = get_post_mime_type($attachment_id);
    if (strpos((string) $mime, 'image/') !== 0) return $metadata;
    
    // Check if we already have alt text
    $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if (!empty($existing_alt)) return $metadata;
    
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log('[AI Alt Text] DELAYED RETRY for attachment ' . $attachment_id);
    }
    
    // Try context detection again
    $context_post_id = aialttext_token_detect_upload_context($attachment_id);
    
    if ($context_post_id > 0) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] DELAYED RETRY found context: ' . $context_post_id);
        }
        
        $args = array('mime' => $mime, 'post_id' => $context_post_id);
        
        try {
            AiAltText_Token_Image_Processor::process_attachment($attachment_id, $args);
        } catch (\Throwable $e) {
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log('[AI Alt Text] DELAYED RETRY error: ' . $e->getMessage());
            }
        }
    }
    
    return $metadata;
}, 10, 2);

/* ---------------------------------------
 * AJAX: single generate (used by admin.js / attachment.js)
 * Enhanced with better context detection
 * --------------------------------------- */
add_action('wp_ajax_aialttext_token_generate_single', function () {
    if (!current_user_can('upload_files') || !aialttext_token_user_can('generate')) {
        wp_send_json_error(array('message' => 'Permission denied'));
    }

    check_ajax_referer('aialttext_token_nonce', 'nonce');

    $id      = isset($_POST['attachment_id']) ? absint($_POST['attachment_id']) : 0;
    $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;

    if (!$id) {
        wp_send_json_error(array('message' => 'Missing attachment_id'));
    }

    $mime = get_post_mime_type($id);
    $args = array('mime' => $mime);
    if ($post_id > 0) {
        $args['post_id'] = $post_id;
    }

    $result = AiAltText_Token_Image_Processor::process_attachment($id, $args);

    if (is_wp_error($result)) {
        $msg  = $result->get_error_message();
        $code = $result->get_error_code();

        // Uniformly flag insufficient tokens (covers HTTP 402 text too)
        $is_out =
            $code === 'insufficient_tokens' ||
            preg_match('/\b402\b/', (string) $msg) ||
            (stripos($msg, 'insufficient') !== false && stripos($msg, 'token') !== false);

        if ($is_out) {
            wp_send_json_error(array(
                'message'              => __('You have no tokens remaining. AI alt text can’t be generated.', 'aialttext-token'),
                'insufficient_tokens'  => true,
                'top_up_url'           => admin_url('admin.php?page=aialttext-token'),
            ));
        }

        wp_send_json_error(array(
            'message'         => $msg,
            'technical_issue' => true,
            'support_email'   => 'support@arcticfoxdevelopments.com',
            'help_url'        => admin_url('admin.php?page=aialttext-token'),
        ));
    }

    wp_send_json_success(array(
        'attachment_id' => $id, // allow the client to de-race responses safely
        'alt_text'      => isset($result['alt_text']) ? $result['alt_text'] : (isset($result['alt']) ? $result['alt'] : ''),
        'model'         => isset($result['model']) ? $result['model'] : null,
        'debug_logs'    => isset($result['debug_logs']) ? $result['debug_logs'] : null,
    ));
});
