<?php
if (!defined('ABSPATH')) exit;

class AiAltText_Token_API_Client {
    private $api_base; 
    private $access_key;
    
    public function __construct(){
        $this->api_base = defined('AIALTTEXT_TOKEN_API_BASE') ? AIALTTEXT_TOKEN_API_BASE : admin_url('admin-ajax.php');
        $this->access_key = trim(get_option('aialttext_token_access_key',''));
    }
    
    private function request($action, $method = 'POST', $data = array()) {
        $url  = $this->api_base;
    
        // Enforce HTTPS for remote endpoints (local admin-ajax remains allowed).
        $parsed = wp_parse_url($url);
        if ( ! empty($parsed['host']) ) {
            $is_local_admin = (untrailingslashit($url) === untrailingslashit(admin_url('admin-ajax.php')));
            if ( ! $is_local_admin && (!isset($parsed['scheme']) || strtolower($parsed['scheme']) !== 'https') ) {
                return array('success' => false, 'error' => array('code' => 'insecure_endpoint', 'message' => 'Refusing non-HTTPS API base.'));
            }
        }
    
        $body = array_merge((array) $data, array('action' => $action));
    
        // Send access key both as header (preferred) and in body (for backward compat)
        $headers = array('Accept' => 'application/json', 'Cache-Control' => 'no-cache');
        if ( ! empty($this->access_key) ) {
            $headers['Authorization'] = 'Bearer ' . $this->access_key;
            $body['access_key'] = $this->access_key;
        }
    
        $args = array(
            'method'      => $method,
            'timeout'     => 20,
            'redirection' => 3,
            'headers'     => $headers,
            'sslverify'   => true,
            'user-agent'  => 'IMG-Alt-Gen-Pro/' . (defined('AIALTTEXT_TOKEN_VERSION') ? AIALTTEXT_TOKEN_VERSION : 'unknown'),
        );
    
        if (strtoupper($method) !== 'GET') {
            $args['body'] = $body;
        } else {
            $url = add_query_arg($body, $url);
        }
    
        // Simple bounded retry on transient 429/5xx
        $attempts = 0; $max = 4; $last_error = '';
        while ($attempts++ < $max) {
            $res = wp_remote_request($url, $args);
            if (is_wp_error($res)) {
                $last_error = $res->get_error_message();
            } else {
                $code = wp_remote_retrieve_response_code($res);
                if ($code >= 200 && $code < 300) {
                    $json = json_decode(wp_remote_retrieve_body($res), true);
                    return is_array($json) ? $json : array('success' => false, 'error' => array('code' => 'bad_json', 'message' => 'Bad JSON'));
                }
                // transient?
                if ($code == 429 || ($code >= 500 && $code <= 599)) {
                    usleep(200000 * $attempts); // linear backoff 200ms, 400ms, ...
                    continue;
                }
                $last_error = 'HTTP ' . $code;
            }
            usleep(150000); // brief pause before retry
        }
    
        return array('success' => false, 'error' => array('code' => 'request_failed', 'message' => $last_error ?: 'Unknown error'));
    }
    
    
    public function link_account($email, $domain) { 
        return $this->request('aialttext_account_link', 'POST', compact('email', 'domain')); 
    }
    
    public function status() { 
        return $this->request('aialttext_account_status', 'POST'); 
    }
    
    public function pricing() { 
        return $this->request('aialttext_account_pricing', 'GET'); 
    }
    
    public function purchase($package_id, $success_url, $cancel_url, $domain) { 
        return $this->request('aialttext_account_purchase', 'POST', compact('package_id', 'success_url', 'cancel_url', 'domain')); 
    }
    
    public function confirm($session_id) { 
        return $this->request('aialttext_confirm_payment', 'POST', compact('session_id')); 
    }
    
    public function generate($image_url, $image_title = '', $context = '', $image_b64 = null, $image_mime = null) {
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] API Client - generate() called with URL: ' . $image_url);
        }
    
        $payload = array(
            'image_url'       => $image_url,
            'image_title'     => $image_title,
            'context'         => $context,
            'site_name'       => get_bloginfo('name'), // for admin-side logs & usage views
            // NEW: pass the chosen language through to the account server
            'output_language' => get_option('aialttext_token_output_language', 'English'),
        );
        if (is_string($image_b64) && $image_b64 !== '' && is_string($image_mime) && $image_mime !== '') {
            $payload['image_b64']  = $image_b64;
            $payload['image_mime'] = $image_mime;
        }
    
        $result = $this->request('aialttext_account_generate', 'POST', $payload);
    
        if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
            error_log('[AI Alt Text] API Client - response: ' . print_r($result, true));
        }
        return $result;
    }
    
    public function usage() {
        return $this->request('aialttext_account_usage', 'POST');
    }
}

function aialttext_token_cached_pricing(){
    $cached = get_transient('aialttext_token_pricing'); 
    if ($cached) return $cached;
    
    $api = new AiAltText_Token_API_Client(); 
    $result = $api->pricing();
    
    if (!empty($result['success'])) {
        set_transient('aialttext_token_pricing', $result, 15 * MINUTE_IN_SECONDS);
    }
    
    return $result;
}

function aialttext_token_check_balance() {
    $api = new AiAltText_Token_API_Client();
    $result = $api->status();
    
    if (!empty($result['success'])) {
        return intval($result['token_balance'] ?? 0);
    }
    
    return 0;
}

function aialttext_token_has_sufficient_balance($required_tokens = 1) {
    $balance = aialttext_token_check_balance();
    return $balance >= $required_tokens;
}