<?php
/*
Plugin Name: IMG Alt Gen Pro
Plugin URI: https://imgaltgen.com/
Description: Analyzes images in the media library and adds alt text using a managed token service. Full-featured version with token integration.
Version: 2.8.3
Author: Arctic Fox Developments
Author URI: https://arcticfoxdevelopments.com/
Text Domain: aialttext-token
Domain Path: /languages
*/

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}



// Define plugin constants
define('AIALTTEXT_TOKEN_PATH', plugin_dir_path(__FILE__));
define('AIALTTEXT_TOKEN_URL', plugin_dir_url(__FILE__));
define('AIALTTEXT_TOKEN_VERSION', '2.8.3'); // cache-bust after Woo fixes
// Flag so companion plugins know not to enqueue overlapping admin assets
if ( ! defined('AIALTTEXT_TOKEN_ACTIVE') ) {
    define('AIALTTEXT_TOKEN_ACTIVE', true);
}
define('AIALTTEXT_TOKEN_BASENAME', plugin_basename(__FILE__));

// Debug mode
define('AIALTTEXT_TOKEN_DEBUG', true); // Set to true to enable debug logging

// API endpoint for token management
if (!defined('AIALTTEXT_TOKEN_API_BASE')) {
    define('AIALTTEXT_TOKEN_API_BASE', 'https://arcticfoxdevelopments.com/wp-admin/admin-ajax.php');
}

// Ensure includes directory exists
if (!file_exists(AIALTTEXT_TOKEN_PATH . 'includes/')) {
    mkdir(AIALTTEXT_TOKEN_PATH . 'includes/', 0755, true);
}
if (!file_exists(AIALTTEXT_TOKEN_PATH . 'admin/')) {
    mkdir(AIALTTEXT_TOKEN_PATH . 'admin/', 0755, true);
}

function aialttext_token_debug_log($message) {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        if (is_array($message) || is_object($message)) {
            error_log('AI Alt Text Token Debug: ' . print_r($message, true));
        } else {
            error_log('AI Alt Text Token Debug: ' . $message);
        }
    }
}

// Include necessary files
require_once AIALTTEXT_TOKEN_PATH . 'includes/settings-page.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/api-client.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/image-processor.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/integrations.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/admin-pages.php';

// Add admin notices for bulk action results
add_action('admin_notices', 'aialttext_token_bulk_action_notices');

// Show a banner on the Media Library when auto-generate is on and balance is 0
add_action('load-upload.php', function () {
    add_action('admin_notices', function () {
        if (!current_user_can('upload_files')) {
            return;
        }

        // Only show on the Media Library screen
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || $screen->base !== 'upload') {
            return;
        }

        // Respect the setting
        if (intval(get_option('aialttext_token_auto_generate', 0)) !== 1) {
            return;
        }

        // Lightweight balance check (the helper already exists in includes/api-client.php)
        if (!function_exists('aialttext_token_check_balance')) {
            return;
        }

        $balance = (int) aialttext_token_check_balance();
        if ($balance > 0) {
            return;
        }

        $url = esc_url(admin_url('admin.php?page=aialttext-token'));
        echo '<div class="notice notice-error"><p>'
            . '<strong>' . esc_html__('AI Alt Text', 'aialttext-token') . ':</strong> '
            . esc_html__('Alt text was not generated for recent uploads because your token balance is 0.', 'aialttext-token')
            . ' <a class="button button-primary" href="' . $url . '">'
            . esc_html__('Top up tokens', 'aialttext-token') . '</a>'
            . '</p></div>';
    });
});

function aialttext_token_bulk_action_notices() {
    $screen = get_current_screen();
    if (!$screen || $screen->base !== 'upload') {
        return;
    }
    
    if (isset($_GET['bulk_ai_alt_text_token'])) {
        $action_result = $_GET['bulk_ai_alt_text_token'];
        $processed = isset($_GET['bulk_ai_alt_text_processed']) ? intval($_GET['bulk_ai_alt_text_processed']) : 0;
        $errors = isset($_GET['bulk_ai_alt_text_errors']) ? intval($_GET['bulk_ai_alt_text_errors']) : 0;
        
        if ($action_result === 'success' && $processed > 0) {
            echo '<div class="notice notice-success is-dismissible">';
            printf(
                '<p><strong>%1$s</strong> %2$s',
                esc_html__('AI Alt Text Generator:', 'aialttext-token'),
                esc_html( sprintf(
                    _n(
                        'Successfully generated alt text for %s image.',
                        'Successfully generated alt text for %s images.',
                        (int) $processed,
                        'aialttext-token'
                    ),
                    number_format_i18n( (int) $processed )
                ))
            );
            if ($errors > 0) {
                echo ' ' . $errors . ' image(s) had errors.';
            }
            echo '</p></div>';
        } elseif ($action_result === 'no_images') {
            echo '<div class="notice notice-warning is-dismissible">';
            printf(
                '<p><strong>%s</strong> %s</p>',
                esc_html__('AI Alt Text Generator:', 'aialttext-token'),
                esc_html__('No images were selected for processing.', 'aialttext-token')
            );
            echo '</div>';
        } elseif ($errors > 0) {
            echo '<div class="notice notice-error is-dismissible">';
            printf(
                '<p><strong>%s</strong> %s</p>',
                esc_html__('AI Alt Text Generator:', 'aialttext-token'),
                esc_html(
                    sprintf(
                        _n(
                            'Failed to process %d image. This may be due to insufficient tokens or API errors.',
                            'Failed to process %d images. This may be due to insufficient tokens or API errors.',
                            (int) $errors,
                            'aialttext-token'
                        ),
                        (int) $errors
                    )
                )
            );
            echo '</div>';
        }
    }
}

// Hook for adding admin menus
add_action('admin_menu', 'aialttext_token_add_menu');

function aialttext_token_add_menu() {
    $level = aialttext_token_get_user_access_level();

    // Only show a top-level menu for users with at least Partial access
    if ($level === 'none' || $level === 'generate') {
        return;
    }

    // Choose a sensible landing page per level
    $parent_slug = 'aialttext-token';
    $parent_cb   = ($level === 'full') ? 'aialttext_token_account_page' : 'aialttext_token_process_page';

    add_menu_page(
        __('IMG Alt Gen Pro', 'aialttext-token'),
        __('IMG Alt Gen Pro', 'aialttext-token'),
        'read',
        $parent_slug,
        $parent_cb,
        'dashicons-images-alt2',
        30
    );

    // Full access: all pages including Account & Settings
    if (aialttext_token_user_can('account')) {
        add_submenu_page($parent_slug, __('Account & Tokens', 'aialttext-token'), __('Account & Tokens', 'aialttext-token'), 'read', 'aialttext-token', 'aialttext_token_account_page');
        add_submenu_page($parent_slug, __('Settings', 'aialttext-token'),         __('Settings', 'aialttext-token'),         'read', 'aialttext-token-settings', 'aialttext_token_settings_page');
    }

    // Partial and Full: Bulk and Help
    if (aialttext_token_user_can('bulk')) {
        add_submenu_page($parent_slug, __('Bulk Process Images', 'aialttext-token'), __('Bulk Process Images', 'aialttext-token'), 'read', 'aialttext-token-process', 'aialttext_token_process_page');
    }
    add_submenu_page($parent_slug, __('Help & Docs', 'aialttext-token'), __('Help & Docs', 'aialttext-token'), 'read', 'aialttext-token-help', 'aialttext_token_help_page');
}

// Hook for registering plugin settings
add_action('admin_init', 'aialttext_token_register_settings');

function aialttext_token_register_settings() {
    register_setting('aialttext_token_options', 'aialttext_token_batch_size');
    register_setting('aialttext_token_options', 'aialttext_token_max_images');
    register_setting('aialttext_token_options', 'aialttext_token_prompt_template');
    register_setting('aialttext_token_options', 'aialttext_token_woocommerce_integration');
    register_setting('aialttext_token_options', 'aialttext_token_auto_generate');
    register_setting('aialttext_token_options', 'aialttext_token_seo_integration');
    register_setting('aialttext_token_options', 'aialttext_token_show_indicators');

    // NEW: default output language for generation
    register_setting('aialttext_token_options', 'aialttext_token_output_language');

    // Token-specific settings
    register_setting('aialttext_token_options', 'aialttext_token_account_email');
    register_setting('aialttext_token_options', 'aialttext_token_access_key');
}

// Hook for adding plugin action links
add_filter('plugin_action_links_' . AIALTTEXT_TOKEN_BASENAME, 'aialttext_token_add_action_links');

// Load translations so WP can auto-detect UI language when .mo files exist.
add_action('plugins_loaded', function () {
    load_plugin_textdomain(
        'aialttext-token',
        false,
        dirname(AIALTTEXT_TOKEN_BASENAME) . '/languages'
    );
});

function aialttext_token_add_action_links($links) {
    $account_link  = '<a href="' . esc_url(admin_url('admin.php?page=aialttext-token')) . '">' . esc_html__('Account', 'aialttext-token') . '</a>';
$settings_link = '<a href="' . esc_url(admin_url('admin.php?page=aialttext-token-settings')) . '">' . esc_html__('Settings', 'aialttext-token') . '</a>';
    array_unshift($links, $account_link, $settings_link);
    return $links;
}

// Already present:
register_activation_hook(__FILE__, 'aialttext_token_activate');

// Redirect new installs to Account & Tokens (single site, non-bulk)
add_action('admin_init', function () {
    if (!current_user_can('manage_options')) return;
    if (is_network_admin() || isset($_GET['activate-multi'])) return;

    if (get_option('aialttext_token_do_activation_redirect')) {
        delete_option('aialttext_token_do_activation_redirect');
        if (function_exists('wp_safe_redirect')) {
            wp_safe_redirect(admin_url('admin.php?page=aialttext-token'));
            exit;
        }
    }
});

function aialttext_token_activate() {
    // Set default options
    add_option('aialttext_token_auto_generate', 0);
    add_option('aialttext_token_show_indicators', 1);
    add_option('aialttext_token_batch_size', 25);
    add_option('aialttext_token_max_images', 0);
    add_option('aialttext_token_prompt_template', "Describe the image clearly for accessibility. Title: {title}. Context: {context}. Return only the alt text.");
    add_option('aialttext_token_woocommerce_integration', 1);
    add_option('aialttext_token_seo_integration', 1);

    // NEW: flag so we can redirect once after activation
add_option('aialttext_token_do_activation_redirect', 1);

// Seed default role access map if not present: admins full, others none
if (false === get_option('aialttext_token_role_access', false) && function_exists('get_editable_roles')) {
    $map = array();
    foreach (get_editable_roles() as $role_key => $role) {
        $map[$role_key] = ($role_key === 'administrator') ? 'full' : 'none';
    }
    add_option('aialttext_token_role_access', $map);
}

    if (defined('WP_DEBUG') && WP_DEBUG) {
        @file_put_contents(WP_CONTENT_DIR . '/aialttext-token-debug.log', '[' . date('c') . '] Activation complete' . PHP_EOL, FILE_APPEND);
    }

    // Ensure WP reflects activation promptly without purging sitewide caches
if (function_exists('wp_clean_plugins_cache')) { wp_clean_plugins_cache(true); }
// Intentionally NOT calling wp_cache_flush() to avoid sitewide cache purges.
}

// Hook for deactivation
register_deactivation_hook(__FILE__, 'aialttext_token_deactivate');

function aialttext_token_deactivate() {
    // Deactivation tasks if needed
}

// Register a filter to prevent alt text being copied to caption/description on save
add_filter('wp_update_attachment_metadata', 'aialttext_token_sanitize_attachment_metadata', 10, 2);

function aialttext_token_sanitize_attachment_metadata($data, $attachment_id) {
    // Check if this is an image attachment
    if (!wp_attachment_is_image($attachment_id)) {
        return $data;
    }
    
    // Get the current alt text
    $alt_text = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if (empty($alt_text)) {
        return $data;
    }
    
    // Get the attachment post object
    $attachment = get_post($attachment_id);
    if (!$attachment) {
        return $data;
    }
    
    // Check if caption is the same as alt text and clear it
    if ($attachment->post_excerpt === $alt_text) {
        wp_update_post(array(
            'ID' => $attachment_id,
            'post_excerpt' => ''
        ));
    }
    
    // Check if description is the same as alt text and clear it
    if ($attachment->post_content === $alt_text) {
        wp_update_post(array(
            'ID' => $attachment_id,
            'post_content' => ''
        ));
    }
    
    return $data;
}

// Enqueue admin scripts and styles
add_action('admin_enqueue_scripts', 'aialttext_token_enqueue_admin_scripts');

function aialttext_token_enqueue_admin_scripts($hook) {
    // List of hooks where we need our scripts
    $allowed_hooks = array(
        'upload.php',           // Media Library
        'media.php',           // Media Edit
        'post.php',            // Post Edit
        'post-new.php',        // New Post
        'attachment.php',      // Attachment Edit
    );
    
    // Check if we're on our plugin pages or allowed hooks
    $is_plugin_page = strpos($hook, 'aialttext-token') !== false;
    $is_allowed_hook = in_array($hook, $allowed_hooks, true);
    
    if (!$is_plugin_page && !$is_allowed_hook) {
        return;
    }

    wp_enqueue_script(
        'aialttext-token-admin-js',
        AIALTTEXT_TOKEN_URL . 'admin/js/admin.js',
        array('jquery', 'wp-i18n'),
        AIALTTEXT_TOKEN_VERSION,
        true
    );
    
    if ( function_exists('wp_set_script_translations') ) {
        wp_set_script_translations(
            'aialttext-token-admin-js',
            'aialttext-token',
            AIALTTEXT_TOKEN_PATH . 'languages'
        );
    }
    
    // Make JS strings translatable (requires /languages/*.json)
    if ( function_exists('wp_set_script_translations') ) {
        wp_set_script_translations(
            'aialttext-token-admin-js',
            'aialttext-token',
            AIALTTEXT_TOKEN_PATH . 'languages'
        );
    }

    wp_enqueue_style('aialttext-token-admin-css', AIALTTEXT_TOKEN_URL . 'admin/css/admin.css', array(), AIALTTEXT_TOKEN_VERSION);

    wp_localize_script('aialttext-token-admin-js', 'aialttext_token_ajax', array(
        'ajax_url'         => admin_url('admin-ajax.php'),
        'nonce'            => wp_create_nonce('aialttext_token_nonce'),
        // Force single-image requests for robustness; bulk limit is controlled by max_images.
        'batch_size'       => 1,
        'max_images'       => max(0, intval(get_option('aialttext_token_max_images', 0))),
        'upload_item_url'  => admin_url('upload.php?item='),
        'post_edit_base'   => admin_url('post.php'),
        'help_url'         => admin_url('admin.php?page=aialttext-token'),
        'first_run'       => (int) (empty(get_option('aialttext_token_access_key', '')))
    ));
}

add_action('admin_enqueue_scripts', 'aialttext_token_enqueue_attachment_assets');

function aialttext_token_enqueue_attachment_assets( $hook ) {
    // Prevent duplicate enqueue from any other place.
    if ( wp_script_is( 'aialttext-token-attachment-js', 'enqueued' ) ) {
        return;
    }

    // Screens we explicitly support
    $allowed = array(
        'upload.php', 'media.php', 'post.php', 'post-new.php', 'attachment.php',
        // WooCommerce SPA (wc-admin)
        'woocommerce_page_wc-admin', 'toplevel_page_wc-admin', 'admin.php'
    );
    if ( ! in_array( $hook, $allowed, true ) ) {
        return;
    }

    // If we're on admin.php ensure it's the Woo app
    if ( $hook === 'admin.php' ) {
        $is_wc_admin = isset($_GET['page']) && $_GET['page'] === 'wc-admin';
        if ( ! $is_wc_admin ) return;

        // If wc-admin route present, only load on product routes
        if ( ! empty($_GET['path']) ) {
            $decoded = rawurldecode( (string) $_GET['path'] );
            if ( strpos( $decoded, '/product/' ) === false ) {
                return;
            }
        }
    }

    // (Optional) keep REST API handy for some media screens
    wp_enqueue_script( 'wp-api' );

    wp_enqueue_script(
        'aialttext-token-attachment-js',
        AIALTTEXT_TOKEN_URL . 'admin/js/attachment.js',
        array('jquery'),
        AIALTTEXT_TOKEN_VERSION,
        true
    );
    
    if ( function_exists('wp_set_script_translations') ) {
        wp_set_script_translations(
            'aialttext-token-attachment-js',
            'aialttext-token',
            AIALTTEXT_TOKEN_PATH . 'languages'
        );
    }

    // Determine current post/product id when possible
$current_post_id = 0;
if ( ! empty( $_GET['post'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    $current_post_id = absint( $_GET['post'] );
} else {
    // Try current screen/globals
    if ( function_exists( 'get_current_screen' ) ) {
        $screen = get_current_screen();
        if ( $screen ) {
            global $post;
            if ( $post && isset( $post->ID ) ) {
                $current_post_id = absint( $post->ID );
            }
        }
    }
}


wp_localize_script(
    'aialttext-token-attachment-js',
    'aialttext_token_attachment',
    array(
        'ajax_url'         => admin_url( 'admin-ajax.php' ),
        'nonce'            => wp_create_nonce( 'aialttext_token_nonce' ),
        'generating'       => __( 'Generating...', 'aialttext-token' ),
        'success'          => __( 'Alt text generated successfully!', 'aialttext-token' ),
        'error'            => __( 'Error generating alt text.', 'aialttext-token' ),
        'current_post_id'  => $current_post_id,
        'help_url'         => admin_url( 'admin.php?page=aialttext-token' ),
    )
);
}


// Special AJAX handler to force update alt text and return current value
add_action('wp_ajax_aialttext_token_force_update_and_get_alt', 'aialttext_token_force_update_and_get_alt');

function aialttext_token_force_update_and_get_alt() {
    // Verify nonce
    // Verify nonce (accept both media and attachment nonces)
$nonce = isset($_POST['nonce']) ? $_POST['nonce'] : '';
if (
    !wp_verify_nonce($nonce, 'aialttext_token_media_update') &&
    !wp_verify_nonce($nonce, 'aialttext_token_nonce')
) {
    wp_send_json_error(array('message' => 'Security check failed'));
    return;
}

    
    // Check permissions
    if (!current_user_can('upload_files')) {
        wp_send_json_error(array('message' => 'Permission denied'));
        return;
    }
    
    // Get image ID and alt text
    $image_id = isset($_POST['image_id']) ? intval($_POST['image_id']) : 0;
    $alt_text = isset($_POST['alt_text']) ? sanitize_text_field($_POST['alt_text']) : '';
    
    if (!$image_id) {
        wp_send_json_error(array('message' => 'Invalid image ID'));
        return;
    }
    
    // If alt text is provided, update it using multiple methods
    if (!empty($alt_text)) {
        // Update only if the value actually changes
$prev_alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
if ($prev_alt !== $alt_text) {
    // Use WP API (adds or updates as needed)
    update_post_meta($image_id, '_wp_attachment_image_alt', $alt_text);

    // Clear attachment-level caches only (never global)
    clean_post_cache($image_id);
    if (function_exists('clean_attachment_cache')) {
        clean_attachment_cache($image_id);
    }
}
    }
    
    // Get the current alt text value
    $current_alt_text = get_post_meta($image_id, '_wp_attachment_image_alt', true);
    
    // Return the result
    wp_send_json_success(array(
        'alt_text' => $current_alt_text,
        'updated' => (!empty($alt_text)),
        'image_id' => $image_id
    ));
}

// Ajax handler for scanning images
add_action('wp_ajax_aialttext_token_scan_images_ajax', 'aialttext_token_scan_images_ajax');

function aialttext_token_scan_images_ajax() {
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
        return;
    }

    // Gate by the plugin's role map (Partial or Full → bulk allowed)
    if ( ! aialttext_token_user_can('bulk') ) {
        wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
        return;
    }

    $total_images = aialttext_token_get_total_images_without_alt();

    wp_send_json(array(
        'success' => true,
        'total_images' => $total_images
    ));
}

// Ajax handler for processing images
add_action('wp_ajax_aialttext_token_process_images_ajax', 'aialttext_token_process_images_ajax');

function aialttext_token_process_images_ajax() {
    // Security
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
        return;
    }
    if (!aialttext_token_user_can('bulk')) {
        wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
        return;
    }

    // Keep the worker alive for slow images/networks
    if (function_exists('ignore_user_abort')) ignore_user_abort(true);
    if (function_exists('set_time_limit')) @set_time_limit(0);

    // Inputs
// Inputs
$offset = isset($_POST['offset']) ? max(0, intval($_POST['offset'])) : 0;

// Force a single-image slice per request for stability and predictable load.
// (Client controls the overall run length via max_images.)
$batch_size = 1;

$max_images = isset($_POST['max_images'])
    ? max(0, intval($_POST['max_images']))
    : max(0, intval(get_option('aialttext_token_max_images', 0)));

$exclude_ids = isset($_POST['exclude_ids']) ? array_map('intval', (array) $_POST['exclude_ids']) : array();
// Keep exclude list bounded to avoid giant NOT IN() lists on big runs
if (count($exclude_ids) > 1000) {
    $exclude_ids = array_slice(array_values(array_unique($exclude_ids)), -1000);
}

$total_processed = isset($_POST['total_processed']) ? max(0, intval($_POST['total_processed'])) : 0;


    $t0 = microtime(true);

    try {
        // ✅ IMPORTANT: pass exclude_ids so we don’t revisit images in this run
        $result = aialttext_token_process_images_with_params($offset, $batch_size, $max_images, $exclude_ids);

        // Enrich response
        $result['total_processed'] = $total_processed + (int)($result['processed'] ?? 0);

        // Very rough ETA (minutes)
        $elapsed = max(0.1, microtime(true) - $t0);
        if (!empty($result['processed'])) {
            $per_image = $elapsed / max(1, (int)$result['processed']);
            $remaining = max(0, (int)$result['total_images'] - (int)$result['total_processed']);
            $result['time_remaining'] = (int) round(($remaining * $per_image) / 60);
        } else {
            $result['time_remaining'] = 'Unknown';
        }

        wp_send_json(array_merge(array('success' => true), $result));
    } catch (Throwable $e) {
        // Soft failure so the client can continue or stop gracefully
        wp_send_json(array(
            'success'              => false,
            'processed'            => 0,
            'processed_ids'        => array(),
            'errors'               => 1,
            'failed'               => array(),
            'error_messages'       => array('Error during batch processing: ' . $e->getMessage()),
            'offset'               => $offset,
            'total_images'         => aialttext_token_get_total_images_without_alt(),
            'completed'            => false,
            'latest_image'         => null,
            'insufficient_tokens'  => false,
            'technical_issue'      => true,
            'time_remaining'       => 'Unknown',
        ));
    }
}


// Add action to media edit page
add_action('attachment_submitbox_misc_actions', 'aialttext_token_add_generate_button', 10, 1);

function aialttext_token_add_generate_button($post) {
    // Only show for image attachments and when user can generate
    if (!wp_attachment_is_image($post->ID) || !aialttext_token_user_can('generate')) {
        return;
    }
    
    ?>
    <div class="misc-pub-section aialttext-token-generate-section">
        <span class="aialttext-token-generate-label"><?php _e('AI Alt Text', 'aialttext-token'); ?></span>
        <button
  type="button"
  id="aialttext-token-generate-single"
  class="button button-primary aialttext-token-generate"
  data-attachment-id="<?php echo esc_attr($post->ID); ?>"
  style="background: linear-gradient(135deg, #673FC5 0%, #4B2AA3 100%); color: #fff;">
  <?php _e('✨ Generate AI Alt Text', 'aialttext-token'); ?>
</button>
<div id="aialttext-token-single-status" class="aialttext-token-result-container" style="margin-top:6px;"></div>


        <span class="spinner" id="aialttext_token_spinner" style="float: none; margin-top: 0;"></span>
        <div id="aialttext_token_result_container" class="aialttext-token-result-container" style="display: none;"></div>
    </div>
    <?php
}

// Add button to media modal
add_filter('attachment_fields_to_edit', 'aialttext_token_add_generate_field', 10, 2);


function aialttext_token_add_generate_field($form_fields, $post) {
    // Only add button for image attachments and when user can generate
    if (!aialttext_token_user_can('generate')) return $form_fields;
    // Only add button for image attachments
    if (!wp_attachment_is_image($post->ID)) {
        return $form_fields;
    }

    /**
     * Prevent duplicate buttons on the single attachment edit screen.
     * We already add a sidebar button via `attachment_submitbox_misc_actions`.
     * Keep THIS form-field button for the Media Modal / AJAX context only.
     */
    $is_attachment_edit_screen = false;

    // 1) Reliable early indicator
    $pagenow = isset($GLOBALS['pagenow']) ? $GLOBALS['pagenow'] : '';
    if ($pagenow === 'attachment.php' || $pagenow === 'media.php') {
        $is_attachment_edit_screen = true;
    }

    // 2) Fallback to current screen, when available
    if (!$is_attachment_edit_screen && function_exists('get_current_screen')) {
        $screen = get_current_screen();
        if ($screen && ( $screen->id === 'attachment' || $screen->base === 'attachment' || $screen->base === 'media' )) {
            $is_attachment_edit_screen = true;
        }
    }

    // If we're on the single attachment edit screen (not an AJAX modal),
    // skip rendering this field to avoid a duplicate button.
    if ($is_attachment_edit_screen && !wp_doing_ajax()) {
        if (function_exists('aialttext_token_console_debug')) {
            aialttext_token_console_debug('Skipping modal/form-field button on single attachment screen to avoid duplicates.');
        }
        return $form_fields;
    }

    // Add custom field for generate button (used in Media Modal)
    $form_fields['aialttext_generate'] = array(
        'label' => '',
        'input' => 'html',
        'html'  => '<button type="button" 
                    class="button button-primary aialttext-token-generate" 
                    data-attachment-id="' . esc_attr($post->ID) . '"
                    style="background: linear-gradient(135deg, #673FC5 0%, #4B2AA3 100%); color: #fff; border: none;">
                    ' . __('✨ Generate AI Alt Text', 'aialttext-token') . '
                    </button>
                    <span class="spinner" style="float: none; margin-top: 0; display: none;"></span>
                    <div class="aialttext-token-result" style="margin-top: 5px; display: none;"></div>',
        'helps' => __('Generate AI-powered alt text for this image', 'aialttext-token')
    );

    return $form_fields;
}


// AJAX handler for applying generated alt text
add_action('wp_ajax_aialttext_token_apply_alt_text_ajax', 'aialttext_token_apply_alt_text_ajax');

function aialttext_token_apply_alt_text_ajax() {
    check_ajax_referer('aialttext_token_nonce', 'nonce');

    $image_id = isset($_POST['image_id']) ? absint($_POST['image_id']) : 0;
    $alt_text = isset($_POST['alt_text']) ? sanitize_text_field(wp_unslash($_POST['alt_text'])) : '';

    if ( ! $image_id || $alt_text === '' ) {
        wp_send_json_error(array('message' => __('Invalid image or alt text.', 'aialttext-token')));
    }

    // Require both general media capability and per-object edit cap
    if ( ! current_user_can('upload_files') || ! current_user_can('edit_post', $image_id) ) {
        wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'aialttext-token')), 403);
    }

    $updated = update_post_meta($image_id, '_wp_attachment_image_alt', $alt_text);

    if ($updated) {
        wp_send_json_success(array('message' => __('Alt text applied successfully!', 'aialttext-token')));
    } else {
        wp_send_json_error(array('message' => __('Error applying alt text.', 'aialttext-token')));
    }
}

// Simple balance check used by the Media Library bulk action interceptor
add_action('wp_ajax_aialttext_token_check_balance_ajax', function () {
    if (!current_user_can('upload_files')) {
        wp_send_json_error(array('message' => 'Permission denied'));
    }
    check_ajax_referer('aialttext_token_nonce', 'nonce');

    $balance = function_exists('aialttext_token_check_balance')
        ? (int) aialttext_token_check_balance()
        : 0;

    wp_send_json_success(array(
        'balance'   => $balance,
        'top_up_url'=> admin_url('admin.php?page=aialttext-token'),
    ));
});

// Add bulk action to media library
add_filter('bulk_actions-upload', 'aialttext_token_register_bulk_action');

function aialttext_token_register_bulk_action($bulk_actions) {
    $bulk_actions['generate_alt_text_token'] = __('Generate AI Alt Text (Selected Images)', 'aialttext-token');
    return $bulk_actions;
}

// Handle bulk action
add_filter('handle_bulk_actions-upload', 'aialttext_token_handle_bulk_action', 10, 3);

function aialttext_token_handle_bulk_action($redirect_to, $action, $post_ids) {
    if ($action !== 'generate_alt_text_token') {
        return $redirect_to;
    }
    
    // Filter out non-image attachments
    $image_ids = array();
    foreach ($post_ids as $post_id) {
        if (wp_attachment_is_image($post_id)) {
            $image_ids[] = $post_id;
        }
    }
    
    if (empty($image_ids)) {
        return add_query_arg('bulk_ai_alt_text_token', 'no_images', $redirect_to);
    }
    
    // Process images immediately instead of redirecting
    $success_count = 0;
    $error_count = 0;
    $errors = array();
    
    foreach ($image_ids as $image_id) {
        try {
            // Check if image already has alt text
            $existing_alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
            if (!empty($existing_alt)) {
                continue; // Skip images that already have alt text
            }
            
            $result = AiAltText_Token_Image_Processor::process_attachment($image_id);
            
            if (!is_wp_error($result)) {
                $success_count++;
            } else {
                $error_count++;
                $errors[] = $result->get_error_message();
                
                // Stop processing if we hit token limits
                if ($result->get_error_code() === 'insufficient_tokens') {
                    break;
                }
            }
        } catch (Exception $e) {
            $error_count++;
            $errors[] = $e->getMessage();
        }
    }
    
    // Build result message
    if ($success_count > 0) {
        $redirect_to = add_query_arg('bulk_ai_alt_text_token', 'success', $redirect_to);
        $redirect_to = add_query_arg('bulk_ai_alt_text_processed', $success_count, $redirect_to);
    }
    
    if ($error_count > 0) {
        $redirect_to = add_query_arg('bulk_ai_alt_text_errors', $error_count, $redirect_to);
    }
    
    return $redirect_to;
}

add_action('add_attachment', 'aialttext_token_auto_generate_alt', 20);
add_action('wp_generate_attachment_metadata', 'aialttext_token_auto_generate_on_metadata', 10, 2);



// Hook when attachments are created via the REST API (Gutenberg uploads)
// Disabled: unified auto-generation now runs in includes/image-processor.php
// add_action( 'rest_after_insert_attachment', function( $attachment, $request, $create ) {
//     if ( ! $create ) { return; }
//     if ( intval( get_option( 'aialttext_token_auto_generate', 0 ) ) !== 1 ) { return; }
//     if ( ! wp_attachment_is_image( $attachment->ID ) ) { return; }
//     aialttext_token_process_delayed_generation( $attachment->ID );
// }, 10, 3 );


// NEW: treat trivial/placeholder alts as "empty" so we can overwrite on product pages
if (!function_exists('aialttext_token_is_placeholder_alt')) {
    function aialttext_token_is_placeholder_alt(int $attachment_id, string $alt): bool {
        $alt = trim(wp_strip_all_tags($alt));
        if ($alt === '') return true;

        // Normalizer
        $norm = function($s) {
            $s = wp_strip_all_tags((string)$s);
            if (function_exists('remove_accents')) $s = remove_accents($s);
            $s = strtolower($s);
            $s = preg_replace('/\.[a-z0-9]+$/i', '', $s); // drop extension if present
            return preg_replace('/[^a-z0-9]+/', '', $s);
        };

        $a = $norm($alt);

        // 1) filename
        $url  = wp_get_attachment_url($attachment_id);
        $file = $url ? wp_basename(parse_url($url, PHP_URL_PATH)) : '';
        if ($file && $a === $norm($file)) return true;

        // 2) attachment title
        $title = get_the_title($attachment_id);
        if ($title && $a === $norm($title)) return true;

        // 3) parent/post/product title
        $parent_id = (int) wp_get_post_parent_id($attachment_id);
        if ($parent_id) {
            $parent = get_post($parent_id);
            if ($parent && $parent->post_title && $a === $norm($parent->post_title)) return true;

            if (get_post_type($parent_id) === 'product' && function_exists('wc_get_product')) {
                $product = wc_get_product($parent_id);
                if ($product && $product->get_name() && $a === $norm($product->get_name())) return true;
            }
        }

        // 4) common placeholders
        $placeholders = array('image','photo','productimage','productphoto','woocommerceproductimage','wpimage','defaultimage');
        if (in_array($a, $placeholders, true)) return true;

        return false;
    }
}


function aialttext_token_auto_generate_alt($attachment_id) {
    aialttext_token_debug_log("Auto-generate triggered for attachment ID: $attachment_id");
    
    // Check if auto-generation is enabled
    if (get_option('aialttext_token_auto_generate', 0) != 1) {
        aialttext_token_debug_log("Auto-generate is disabled in settings");
        return;
    }
    
    // Check if the attachment is an image
    if (!wp_attachment_is_image($attachment_id)) {
        aialttext_token_debug_log("Attachment $attachment_id is not an image");
        return;
    }
    
    // Check existing alt; allow overwrite if it's just a placeholder (e.g., product title/filename)
$existing_alt = (string) get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
if ($existing_alt !== '') {
    if (function_exists('aialttext_token_is_placeholder_alt') && aialttext_token_is_placeholder_alt($attachment_id, $existing_alt)) {
        aialttext_token_debug_log("Attachment $attachment_id has placeholder alt ('$existing_alt'); generating contextual alt…");
        // fall through to generation
    } else {
        aialttext_token_debug_log("Attachment $attachment_id already has non-placeholder alt; skipping.");
        return;
    }
}
    
    // Process immediately, don't wait
    aialttext_token_process_delayed_generation($attachment_id);
}

// Function to handle metadata-based generation
function aialttext_token_auto_generate_on_metadata($metadata, $attachment_id) {
    // This is another entry point that might work better in some WordPress setups
    aialttext_token_auto_generate_alt($attachment_id);
    return $metadata;
}

// Add custom action for delayed processing
add_action('aialttext_token_delayed_generate_alt', 'aialttext_token_process_delayed_generation');

function aialttext_token_process_delayed_generation($attachment_id) {
    aialttext_token_debug_log("Starting delayed alt text generation for attachment $attachment_id");
    
    // Check if auto-generation is still enabled
    if (get_option('aialttext_token_auto_generate', 0) != 1) {
        return;
    }
    
    // Double-check that the attachment is still an image
    if (!wp_attachment_is_image($attachment_id)) {
        return;
    }
    
    // Check if alt text was added in the meantime; only skip if it's non-placeholder
$existing_alt = (string) get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
if ($existing_alt !== '') {
    if (function_exists('aialttext_token_is_placeholder_alt') && aialttext_token_is_placeholder_alt($attachment_id, $existing_alt)) {
        aialttext_token_debug_log("Delayed run: placeholder alt detected ('$existing_alt'); will overwrite.");
        // continue
    } else {
        aialttext_token_debug_log("Delayed run: non-placeholder alt present; skipping.");
        return;
    }
}

    
    // Get image data - ensure we have a valid URL before proceeding
    $image_url = wp_get_attachment_url($attachment_id);
    if (empty($image_url)) {
        aialttext_token_debug_log("Empty image URL for attachment $attachment_id");
        return;
    }
    
    $image_title = get_the_title($attachment_id);
    if (empty($image_title)) {
        $image_title = basename($image_url);
    }
    
    // Get additional context (robust detection across Classic + Gutenberg + Woo screens)
    $post_id = aialttext_token_detect_current_post_id( $attachment_id );
    $context = aialttext_token_get_image_context( $attachment_id, $post_id );

    
    try {
        // Use the token-based image processor
        $result = AiAltText_Token_Image_Processor::process_attachment($attachment_id, array('context' => $context));
        
        if (!is_wp_error($result)) {
            // Log the auto-generation
            aialttext_token_debug_log(sprintf('AI Alt Text Token: Successfully auto-generated for image ID %d: %s', $attachment_id, $result['alt']));
        } else {
            aialttext_token_debug_log(sprintf('AI Alt Text Token: Failed to auto-generate for image ID %d: %s', $attachment_id, $result->get_error_message()));
        }
    } catch (Exception $e) {
        aialttext_token_debug_log(sprintf('AI Alt Text Token: Exception when auto-generating for image ID %d: %s', $attachment_id, $e->getMessage()));
    }
}

// Add indicator to media library grid view
add_filter('attachment_fields_to_edit', 'aialttext_token_add_indicator', 10, 2);

function aialttext_token_add_indicator($form_fields, $post) {
    if (wp_attachment_is_image($post->ID)) {
        $alt_text = get_post_meta($post->ID, '_wp_attachment_image_alt', true);
        
        if (!empty($alt_text)) {
            // Add a small indicator that this image has alt text
            $alt_preview = esc_html(substr($alt_text, 0, 40)) . (strlen($alt_text) > 40 ? '...' : '');
            $form_fields['alt']['input'] = $form_fields['alt']['input'] . 
                '<div class="aialttext-token-status" style="margin-top:5px;font-size:11px;color:#666;">' .
                '<span style="color:green;font-size:15px;">✓</span> ' . esc_html__( 'Alt text:', 'aialttext-token' ) . ' ' . $alt_preview . '</div>';
        } else {
            $form_fields['alt']['input'] = $form_fields['alt']['input'] . 
                '<div class="aialttext-token-status" style="margin-top:5px;font-size:11px;color:#666;">' .
                '<span style="color:#d63638;font-size:15px;">⚠</span> ' . esc_html__( 'No alt text', 'aialttext-token' ) . '</div>';
        }
    }
    
    return $form_fields;
}

// Gutenberg integration
add_action('enqueue_block_editor_assets', 'aialttext_token_gutenberg_assets');

function aialttext_token_gutenberg_assets() {
    // Get current post ID for context
    global $post;
    $current_post_id = 0;
    if ($post && isset($post->ID)) {
        $current_post_id = $post->ID;
    } elseif (isset($_GET['post'])) {
        $current_post_id = intval($_GET['post']);
    }

    wp_enqueue_script(
    'aialttext-token-gutenberg',
    AIALTTEXT_TOKEN_URL . 'admin/js/gutenberg.js',
    array('wp-blocks', 'wp-i18n', 'wp-element', 'wp-editor', 'wp-block-editor', 'wp-components', 'wp-data', 'wp-compose', 'wp-hooks', 'jquery'),
    AIALTTEXT_TOKEN_VERSION,
    true
);

if ( function_exists('wp_set_script_translations') ) {
    wp_set_script_translations(
        'aialttext-token-gutenberg',
        'aialttext-token',
        AIALTTEXT_TOKEN_PATH . 'languages'
    );
}
    
    wp_localize_script('aialttext-token-gutenberg', 'aialttext_token_gutenberg', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('aialttext_token_nonce'),
        'generating' => __('Generating...', 'aialttext-token'),
        'generate_text' => __('✨ Generate AI Alt Text', 'aialttext-token'),
        'success' => __('Alt text generated successfully!', 'aialttext-token'),
        'error' => __('Error generating alt text.', 'aialttext-token'),
        'current_post_id' => $current_post_id
    ));
    
    // Add inline script to set global context
    if ($current_post_id > 0) {
        wp_add_inline_script('aialttext-token-gutenberg', 
            "window.aialttext_current_post_id = " . intval($current_post_id) . ";", 
            'before'
        );
    }
}

// Add visual indicators to the media grid view
add_action('admin_head', 'aialttext_token_add_grid_indicators');

function aialttext_token_add_grid_indicators() {
    // Only on media library page and if indicators are enabled
    $screen = get_current_screen();
    if (!$screen || $screen->base !== 'upload' || get_option('aialttext_token_show_indicators', 1) != 1) {
        return;
    }
    
    ?>
    <style>
    .attachment-preview .has-alt-text:after {
        content: "ALT";
        position: absolute;
        bottom: 0;
        right: 0;
        background: rgba(0,170,0,0.8);
        color: white;
        font-size: 10px;
        padding: 2px 4px;
        border-radius: 3px 0 0 0;
    }
    .attachment-preview .no-alt-text:after {
        content: "NO ALT";
        position: absolute;
        bottom: 0;
        right: 0;
        background: rgba(220,53,69,0.8);
        color: white;
        font-size: 10px;
        padding: 2px 4px;
        border-radius: 3px 0 0 0;
    }
    </style>
    <script>
    jQuery(document).ready(function($) {
        // Add indicators to attachments in grid view
        function addGridIndicators() {
            $('.attachment').each(function() {
                var $this = $(this);
                var id = $this.data('id');
                
                // Skip if already processed
                if ($this.hasClass('alt-text-checked')) {
                    return;
                }
                
                $this.addClass('alt-text-checked');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aialttext_token_check_alt_text',
                        id: id,
                        nonce: '<?php echo wp_create_nonce('aialttext_token_check_alt'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            if (response.data.has_alt_text) {
                                $this.find('.attachment-preview').addClass('has-alt-text');
                            } else {
                                $this.find('.attachment-preview').addClass('no-alt-text');
                            }
                        }
                    }
                });
            });
        }
        
        // Run on page load
        addGridIndicators();
        
        // Run again when media is loaded via AJAX
        $(document).on('DOMNodeInserted', '.attachment', function() {
            addGridIndicators();
        });
    });
    </script>
    <?php
}

// AJAX handler to check for alt text
add_action('wp_ajax_aialttext_token_check_alt_text', 'aialttext_token_check_alt_text_ajax');

function aialttext_token_check_alt_text_ajax() {
    // Verify nonce - accept multiple nonce types for compatibility
    $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
    $valid_nonce = wp_verify_nonce($nonce, 'aialttext_token_check_alt') || 
                  wp_verify_nonce($nonce, 'aialttext_token_media_update') || 
                  wp_verify_nonce($nonce, 'aialttext_token_nonce');
                  
    if (!$valid_nonce) {
        wp_send_json_error(array('message' => 'Invalid security token'));
        return;
    }
    
    // Get attachment ID
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    
    if (!$id) {
        wp_send_json_error(array('message' => 'Invalid attachment ID'));
        return;
    }
    
    // Use DIRECT database query for maximum reliability
    global $wpdb;
    $alt_text = $wpdb->get_var($wpdb->prepare(
        "SELECT meta_value FROM $wpdb->postmeta 
         WHERE post_id = %d AND meta_key = '_wp_attachment_image_alt' 
         LIMIT 1",
        $id
    ));
    
    // If still not found, try traditional method as backup
    if ($alt_text === null) {
        $alt_text = get_post_meta($id, '_wp_attachment_image_alt', true);
    }
    
    wp_send_json_success(array(
        'has_alt_text' => !empty($alt_text),
        'alt_text' => $alt_text,
        'id' => $id
    ));
}

// Add filter to ensure alt text only applies to the alt field
add_filter('attachment_fields_to_save', 'aialttext_token_prevent_alttext_in_other_fields', 10, 2);

function aialttext_token_prevent_alttext_in_other_fields($post, $attachment) {
    // Don't do anything for non-image attachments
    if (!wp_attachment_is_image($post['ID'])) {
        return $post;
    }
    
    // Only check if alt text is being set
    if (isset($attachment['alt'])) {
        $alt_text = $attachment['alt'];
        
        // Store only in the alt text field and not elsewhere
        update_post_meta($post['ID'], '_wp_attachment_image_alt', $alt_text);
        
        // Ensure it doesn't contaminate other fields like caption or description
        if (isset($post['post_excerpt']) && $post['post_excerpt'] === $alt_text) {
            $post['post_excerpt'] = '';
        }
        if (isset($post['post_content']) && $post['post_content'] === $alt_text) {
            $post['post_content'] = '';
        }
    }
    
    return $post;
}

// Add clean-up on post save to ensure alt text doesn't leak into other fields
add_action('edit_attachment', 'aialttext_token_clean_attachment_fields');

function aialttext_token_clean_attachment_fields($post_id) {
    // Only process image attachments
    if (!wp_attachment_is_image($post_id)) {
        return;
    }
    
    // Get the alt text value
    $alt_text = get_post_meta($post_id, '_wp_attachment_image_alt', true);
    if (empty($alt_text)) {
        return;
    }
    
    // Get post data
    $post = get_post($post_id);
    $updated = false;
    
    // Check if caption or description is same as alt text (indicating it leaked)
    if ($post->post_excerpt === $alt_text) {
        // Clear caption
        wp_update_post(array(
            'ID' => $post_id,
            'post_excerpt' => ''
        ));
        $updated = true;
    }
    
    if ($post->post_content === $alt_text) {
        // Clear description
        wp_update_post(array(
            'ID' => $post_id,
            'post_content' => ''
        ));
        $updated = true;
    }
    
    if ($updated && defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log(sprintf('AI Alt Text Token: Cleaned caption/description fields for attachment %d', $post_id));
    }
}

// Add simple visual indicators to the media library
add_filter('manage_media_columns', 'aialttext_token_add_alt_column', 999);
add_action('manage_media_custom_column', 'aialttext_token_alt_column_content', 10, 2);

function aialttext_token_add_alt_column($columns) {
    // Only add if indicators are enabled
    if (get_option('aialttext_token_show_indicators', 1) != 1) {
        return $columns;
    }
    
    // Remove ANY existing alt text related columns first to avoid ALL duplicates
    // This includes our own previous column to prevent duplicates
    $columns_to_remove = array(
        'alt', 
        'alt_text', 
        'alt_text_status', 
        'image_alt', 
        'attachment_alt',
        'aialttext_alt_status',  // Our own column
        'alt_status',            // Another possible variant
        'has_alt'               // Another possible variant
    );
    
    foreach ($columns as $key => $value) {
        // Remove any column that contains 'alt' in its key (case-insensitive)
        if (stripos($key, 'alt') !== false) {
            unset($columns[$key]);
        }
    }
    
    // Now remove specific ones that might not contain 'alt'
    foreach ($columns_to_remove as $col) {
        if (isset($columns[$col])) {
            unset($columns[$col]);
        }
    }
    
    // Add our column after title
    $new_columns = array();
    $added = false;
    foreach ($columns as $key => $value) {
        $new_columns[$key] = $value;
        
        // Add our column after 'title' and make sure we only add it once
        if ($key === 'title' && !$added) {
            $new_columns['aialttext_alt_status'] = __('Alt Text', 'aialttext-token');
            $added = true;
        }
    }
    
    // If we didn't add it yet (no title column), add it at the end
    if (!$added) {
        $new_columns['aialttext_alt_status'] = __('Alt Text', 'aialttext-token');
    }
    
    return $new_columns;
}

function aialttext_token_alt_column_content($column_name, $post_id) {
    // Updated to match our unique column name
    if ($column_name !== 'aialttext_alt_status' || !wp_attachment_is_image($post_id)) {
        return;
    }
    
    $alt_text = get_post_meta($post_id, '_wp_attachment_image_alt', true);
    
    if (!empty($alt_text)) {
        echo '<div class="aialttext-alt-indicator" style="color:#46b450;font-weight:bold;">✓ ' . __('Yes', 'aialttext-token') . '</div>';
    } else {
        // Generate a unique ID for this button
        $button_id = 'aialttext-gen-' . $post_id;
        
        echo '<div class="aialttext-alt-indicator" style="color:#dc3232;font-weight:bold;">✗ ' . __('No', 'aialttext-token') . ' ';
        echo '<button type="button" ';
        echo 'id="' . esc_attr($button_id) . '" ';
        echo 'class="button button-small aialttext-generate-table" ';
        echo 'data-image-id="' . esc_attr($post_id) . '" ';
        echo 'data-attachment-id="' . esc_attr($post_id) . '" ';
        echo 'style="margin-left:5px;background:#673FC5;color:#fff;border-color:#673FC5;cursor:pointer;">';
        echo __('Generate', 'aialttext-token');
        echo '</button></div>';
        
        // Add inline script to ensure button works
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Ensure the button is bound immediately
            $('#<?php echo esc_js($button_id); ?>').off('click.aialttext').on('click.aialttext', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var $btn = $(this);
                if ($btn.prop('disabled')) return;
                
                var imageId = <?php echo intval($post_id); ?>;
                var originalText = $btn.text();
                
                $btn.prop('disabled', true).text('Generating...').css('opacity', '0.6');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    dataType: 'json',
                    data: {
                        action: 'aialttext_token_generate_single',
                        nonce: '<?php echo wp_create_nonce('aialttext_token_nonce'); ?>',
                        attachment_id: imageId,
                        post_id: 0
                    },
                    success: function(res) {
                        if (res && res.success && res.data && (res.data.alt || res.data.alt_text)) {
                            $btn.parent().html('<div style="color:#46b450;font-weight:bold;">✓ <?php echo esc_js(__('Yes', 'aialttext-token')); ?></div>');
                        } else {
                            $btn.prop('disabled', false).text(originalText).css('opacity', '1');
                            alert('Error: ' + ((res && res.data && res.data.message) || 'Failed to generate'));
                        }
                    },
                    error: function() {
                        $btn.prop('disabled', false).text(originalText).css('opacity', '1');
                        alert('Network error occurred. Please try again.');
                    }
                });
            });
        });
        </script>
        <?php
    }
}

// Add version upgrade check
add_action('admin_init', 'aialttext_token_check_version_upgrade');

function aialttext_token_check_version_upgrade() {
    $current_version = get_option('aialttext_token_version', '0');
    
    // If this is a first install or an upgrade
    if (version_compare($current_version, AIALTTEXT_TOKEN_VERSION, '<')) {
        // Perform any database upgrades if needed
        
        // Update stored version number
        update_option('aialttext_token_version', AIALTTEXT_TOKEN_VERSION);
        
        // Set debug mode on upgrade to help troubleshoot if needed
        if (defined('AIALTTEXT_TOKEN_DEBUG') && !AIALTTEXT_TOKEN_DEBUG) {
            update_option('aialttext_token_debug_upgrade', true);
        }
        
        // No global cache flush; option updates don't require it.
// If you want to ensure fresh pricing for this plugin only:
delete_transient('aialttext_token_pricing');
    }
}