/**
 * AI Alt Text Generator - Gutenberg Block Editor Integration
 * Adds "Generate AI Alt Text" button to Image block settings panel
 */

const { __, _n, sprintf } = wp.i18n;

(function(wp, $) {
    'use strict';

    if (!wp || !wp.blocks || !wp.element || !wp.components || !wp.data) {
        console.warn('[AI Alt Text] Required WordPress packages not available');
        return;
    }

    const { createElement: el, Fragment, useState, useEffect } = wp.element;
    const { InspectorControls } = wp.blockEditor || wp.editor;
    const { PanelBody, Button, Notice } = wp.components;
    const { addFilter } = wp.hooks;
    const { select, dispatch } = wp.data;
    const { __ } = wp.i18n;

    /**
     * Get current post/product ID for context
     */
    function getCurrentPostId() {
        try {
            const editorSelect = select('core/editor');
            if (editorSelect) {
                const postId = editorSelect.getCurrentPostId();
                if (postId) return parseInt(postId, 10);
            }
        } catch (e) {
            console.debug('[AI Alt Text] Could not get post ID from editor:', e);
        }
        
        // Fallback to global
        if (window.aialttext_current_post_id) {
            return parseInt(window.aialttext_current_post_id, 10);
        }
        
        // Fallback to URL parameter
        const urlParams = new URLSearchParams(window.location.search);
        const postParam = urlParams.get('post');
        if (postParam) {
            return parseInt(postParam, 10);
        }
        
        return 0;
    }

    /**
     * Build inline context from current editor state
     */
    function buildInlineContext() {
        try {
            const editorSelect = select('core/editor');
            if (!editorSelect) return '';

            const title = editorSelect.getEditedPostAttribute('title') || '';
            const excerpt = editorSelect.getEditedPostAttribute('excerpt') || '';
            const content = editorSelect.getEditedPostAttribute('content') || '';

            let context = '';
            if (title) {
                context += sprintf( __('This image appears on: "%s". ', 'aialttext-token'), title );
            }
            
            const desc = excerpt || content.substring(0, 500);
            if (desc) {
                // Strip HTML tags for cleaner context
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = desc;
                const cleanDesc = (tempDiv.textContent || tempDiv.innerText || '').trim();
                if (cleanDesc) {
                    context += `Content: ${cleanDesc}. `;
                }
            }

            return context.trim();
        } catch (e) {
            console.debug('[AI Alt Text] Error building context:', e);
            return '';
        }
    }

    /**
     * Generate AI Alt Text Component
     */
    function GenerateAltTextButton({ clientId, imageId, currentAlt }) {
        const [isGenerating, setIsGenerating] = useState(false);
        const [message, setMessage] = useState(null);
        const [messageType, setMessageType] = useState('info');

        const handleGenerate = async () => {
            if (!imageId) {
                setMessage(__('No image selected', 'aialttext-token'));
                setMessageType('error');
                return;
            }

            setIsGenerating(true);
            setMessage(null);

            try {
                const postId = getCurrentPostId();
                const context = buildInlineContext();

                // Call the AJAX endpoint
                const response = await $.ajax({
                    url: window.aialttext_token_gutenberg?.ajax_url || ajaxurl,
                    type: 'POST',
                    dataType: 'json',
                    data: {
                        action: 'aialttext_token_generate_single',
                        nonce: window.aialttext_token_gutenberg?.nonce || '',
                        attachment_id: imageId,
                        post_id: postId,
                        context: context
                    }
                });

                if (response.success && response.data) {
                    const altText = response.data.alt_text || response.data.alt || '';
                    
                    if (altText) {
                        // Update the block's alt attribute
                        dispatch('core/block-editor').updateBlockAttributes(clientId, {
                            alt: altText
                        });

                        // Also update the media library alt text
                        await $.ajax({
                            url: window.aialttext_token_gutenberg?.ajax_url || ajaxurl,
                            type: 'POST',
                            data: {
                                action: 'aialttext_token_force_update_and_get_alt',
                                nonce: window.aialttext_token_gutenberg?.nonce || '',
                                image_id: imageId,
                                alt_text: altText
                            }
                        });

                        setMessage('✓ Alt text generated successfully!');
                        setMessageType('success');

                        // Log success
                        console.log('[AI Alt Text] Generated:', altText);
                        
                        // If there are debug logs, show them
                        if (response.data.debug_logs) {
                            console.group('[AI Alt Text] Debug Logs');
                            response.data.debug_logs.forEach(log => {
                                console.log(log.message, log.data || '');
                            });
                            console.groupEnd();
                        }
                    } else {
                        throw new Error('No alt text returned');
                    }
                } else {
                    // Handle specific error cases
                    if (response.data && response.data.insufficient_tokens) {
                        setMessage('⚠ Out of tokens. Please top up.');
                        setMessageType('error');
                        
                        // Show modal if available
                        if (window.aialttextShowTopUpModal) {
                            setTimeout(() => {
                                window.aialttextShowTopUpModal(response.data.top_up_url);
                            }, 1000);
                        }
                    } else {
                        const errorMsg = response.data?.message || 'Failed to generate alt text';
                        setMessage(`✗ ${errorMsg}`);
                        setMessageType('error');
                    }
                }
            } catch (error) {
                console.error('[AI Alt Text] Generation error:', error);
                
                // Check for 402 status (insufficient tokens)
                if (error.status === 402) {
                    setMessage('⚠ Out of tokens. Please top up.');
                    setMessageType('error');
                    
                    if (window.aialttextShowTopUpModal) {
                        setTimeout(() => {
                            window.aialttextShowTopUpModal();
                        }, 1000);
                    }
                } else {
                    setMessage('✗ Error generating alt text. Please try again.');
                    setMessageType('error');
                }
            } finally {
                setIsGenerating(false);
                
                // Clear message after 5 seconds
                setTimeout(() => {
                    setMessage(null);
                }, 5000);
            }
        };

        return el(Fragment, {},
            el(Button, {
                variant: 'primary',
                onClick: handleGenerate,
                disabled: isGenerating || !imageId,
                style: {
                    background: 'linear-gradient(135deg, #673FC5 0%, #4B2AA3 100%)',
                    border: 'none',
                    color: '#fff',
                    width: '100%',
                    justifyContent: 'center',
                    marginTop: '8px'
                }
            }, isGenerating ? __('Generating...', 'aialttext-token') : __('✨ Generate AI Alt Text', 'aialttext-token')),
            
            message && el(Notice, {
                status: messageType,
                isDismissible: false,
                style: { marginTop: '8px', marginBottom: '0' }
            }, message)
        );
    }

    /**
     * Add our button to the Image block's inspector controls
     */
    const addAltTextGeneratorControl = wp.compose.createHigherOrderComponent((BlockEdit) => {
        return (props) => {
            const { name, attributes, clientId } = props;

            // Only add to core/image blocks
            if (name !== 'core/image') {
                return el(BlockEdit, props);
            }

            const imageId = attributes.id;
            const currentAlt = attributes.alt || '';

            return el(Fragment, {},
                el(BlockEdit, props),
                el(InspectorControls, {},
                    el(PanelBody, {
                        title: __('AI Alt Text Generator', 'aialttext-token'),
                        initialOpen: false
                    },
                        el('div', { 
                            style: { padding: '0' },
                            className: 'aialttext-gutenberg-panel'
                        },
                            el('p', { 
                                style: { 
                                    fontSize: '12px', 
                                    color: '#757575',
                                    marginTop: '0',
                                    marginBottom: '8px'
                                }
                            }, imageId 
                                ? __('Generate AI-powered alt text for this image with SEO and product context.', 'aialttext-token')
                                : __('Select an image first to generate alt text.', 'aialttext-token')
                            ),
                            el(GenerateAltTextButton, {
                                clientId: clientId,
                                imageId: imageId,
                                currentAlt: currentAlt
                            })
                        )
                    )
                )
            );
        };
    }, 'addAltTextGeneratorControl');

    // Register the filter
    addFilter(
        'editor.BlockEdit',
        'aialttext-token/add-alt-text-generator',
        addAltTextGeneratorControl
    );

    console.log('[AI Alt Text] Gutenberg integration loaded');

})(window.wp, jQuery);