<?php
// This file handles the settings page functionality for the token-based plugin
// It's included in the main plugin file

if (!defined('ABSPATH')) exit;

// The settings page is handled in admin-pages.php
// This file can contain additional settings-related functions if needed

/**
 * Get default prompt template
 */
function aialttext_token_get_default_prompt_template() {
    return "Describe the image clearly for accessibility. Title: {title}. Context: {context}. Return only the alt text.";
}

/**
 * Sanitize prompt template
 */
function aialttext_token_sanitize_prompt_template($template) {
    // Allow basic HTML tags and placeholders
    $allowed_tags = array();
    return wp_kses($template, $allowed_tags);
}

/**
 * Validate batch size
 */
function aialttext_token_validate_batch_size($size) {
    $size = intval($size);
    if ($size < 1) $size = 1;
    if ($size > 200) $size = 200;
    return $size;
}

/**
 * Validate max images
 */
function aialttext_token_validate_max_images($max) {
    $max = intval($max);
    if ($max < 0) $max = 0;
    return $max;
}

/**
 * Get available integrations
 */
function aialttext_token_get_available_integrations() {
    $integrations = array();
    
    if (class_exists('WooCommerce')) {
        $integrations['woocommerce'] = array(
            'name' => 'WooCommerce',
            'description' => 'Include product information in alt text generation',
            'active' => get_option('aialttext_token_woocommerce_integration', 1)
        );
    }
    
    if (defined('WPSEO_VERSION') || class_exists('RankMath') || class_exists('AIOSEO\Plugin\Common\Main') || defined('SEOPRESS_VERSION')) {
        $integrations['seo'] = array(
            'name' => 'SEO Plugins',
            'description' => 'Include SEO keywords and meta descriptions in context',
            'active' => get_option('aialttext_token_seo_integration', 1)
        );
    }
    
    return $integrations;
}

/**
 * Check if token system is properly configured
 */
function aialttext_token_is_configured() {
    $email = get_option('aialttext_token_account_email', '');
    $access_key = get_option('aialttext_token_access_key', '');
    
    return !empty($email) && !empty($access_key);
}

/**
 * Get configuration status for admin notices
 */
function aialttext_token_get_config_status() {
    if (!aialttext_token_is_configured()) {
        return array(
            'status' => 'incomplete',
            'message' => 'Please link your account in the Account & Tokens page to start using the service.',
            'action_url' => admin_url('admin.php?page=aialttext-token'),
            'action_text' => 'Link Account'
        );
    }
    
    // Check balance
    $balance = aialttext_token_check_balance();
    if ($balance <= 0) {
        return array(
            'status' => 'no_tokens',
            'message' => 'You have no tokens remaining. Purchase more tokens to continue generating alt text.',
            'action_url' => admin_url('admin.php?page=aialttext-token'),
            'action_text' => 'Purchase Tokens'
        );
    }
    
    if ($balance <= 5) {
        return array(
            'status' => 'low_tokens',
            'message' => "You have {$balance} tokens remaining. Consider purchasing more tokens soon.",
            'action_url' => admin_url('admin.php?page=aialttext-token'),
            'action_text' => 'Purchase Tokens'
        );
    }
    
    return array(
        'status' => 'configured',
        'message' => "Token system is configured. You have {$balance} tokens available.",
        'balance' => $balance
    );
}

// Add admin notices for configuration status
add_action('admin_notices', 'aialttext_token_admin_notices');

function aialttext_token_admin_notices() {
    // Only show on our plugin pages
    $screen = get_current_screen();
    if (!$screen || strpos($screen->id, 'aialttext-token') === false) {
        return;
    }
    
    $status = aialttext_token_get_config_status();
    
    if ($status['status'] === 'incomplete') {
        echo '<div class="notice notice-warning">';
        echo '<p><strong>AI Alt Text Generator:</strong> ' . esc_html($status['message']);
        echo ' <a href="' . esc_url($status['action_url']) . '" class="button button-secondary">' . esc_html($status['action_text']) . '</a></p>';
        echo '</div>';
    } elseif ($status['status'] === 'no_tokens') {
        echo '<div class="notice notice-error">';
        echo '<p><strong>AI Alt Text Generator:</strong> ' . esc_html($status['message']);
        echo ' <a href="' . esc_url($status['action_url']) . '" class="button button-primary">' . esc_html($status['action_text']) . '</a></p>';
        echo '</div>';
    } elseif ($status['status'] === 'low_tokens') {
        echo '<div class="notice notice-warning">';
        echo '<p><strong>AI Alt Text Generator:</strong> ' . esc_html($status['message']);
        echo ' <a href="' . esc_url($status['action_url']) . '" class="button button-secondary">' . esc_html($status['action_text']) . '</a></p>';
        echo '</div>';
    }
}

// Add settings link to plugins page
add_filter('plugin_action_links_' . AIALTTEXT_TOKEN_BASENAME, 'aialttext_token_plugin_action_links');

function aialttext_token_plugin_action_links($links) {
    $status = aialttext_token_get_config_status();
    
    if ($status['status'] === 'incomplete') {
        $setup_link = '<a href="' . admin_url('admin.php?page=aialttext-token') . '" style="color: #d63638; font-weight: bold;">' . __('Setup Required', 'aialttext-token') . '</a>';
        array_unshift($links, $setup_link);
    } elseif ($status['status'] === 'no_tokens') {
        $tokens_link = '<a href="' . admin_url('admin.php?page=aialttext-token') . '" style="color: #d63638; font-weight: bold;">' . __('No Tokens', 'aialttext-token') . '</a>';
        array_unshift($links, $tokens_link);
    } elseif (isset($status['balance'])) {
        $balance_link = '<span style="color: #46b450; font-weight: bold;">' . sprintf(__('%d Tokens', 'aialttext-token'), $status['balance']) . '</span>';
        array_unshift($links, $balance_link);
    }
    
    return $links;
}