<?php
if (!defined('ABSPATH')) exit;

// Build contextual hints (WooCommerce, Yoast, RankMath)
function aialttext_token_build_context_for_attachment($attachment_id){
    $parts = array();
    
    if (class_exists('WC_Product')){
        $parent = wp_get_post_parent_id($attachment_id);
        if ($parent && 'product' === get_post_type($parent)){
            $parts[] = 'Product: '.get_the_title($parent);
        }
    }
    
    if (defined('WPSEO_VERSION')){
        $parent = wp_get_post_parent_id($attachment_id);
        if ($parent){
            $kw = get_post_meta($parent, '_yoast_wpseo_focuskw', true);
            if ($kw) $parts[] = 'Keywords: '.$kw;
        }
    }
    
    if (defined('RANK_MATH_VERSION')){
        $parent = wp_get_post_parent_id($attachment_id);
        if ($parent){
            $kw = get_post_meta($parent, 'rank_math_focus_keyword', true);
            if ($kw) $parts[] = 'Keywords: '.$kw;
        }
    }
    
    return implode(' | ', $parts);
}

/**
 * Get context information for an image
 * 
 * @param int $image_id The image ID
 * @param int $post_id Optional post ID for context
 * @return string Context information
 */
function aialttext_token_get_image_context($image_id, $post_id = 0) {
    $context = '';
    
    // If no post ID provided but image has a parent post
    if (!$post_id) {
        $attachment = get_post($image_id);
        if ($attachment && $attachment->post_parent) {
            $post_id = $attachment->post_parent;
        }
    }
    
    // Get SEO context if enabled
    if (get_option('aialttext_token_seo_integration', 0) == 1) {
        $seo_context = aialttext_token_get_seo_context($post_id);
        if (!empty($seo_context)) {
            $context .= ' ' . $seo_context;
        }
    }
    
    // Always check for WooCommerce context regardless of setting
    // This ensures we detect product pages automatically
    if (class_exists('WooCommerce')) {
        $woo_context = aialttext_token_get_woocommerce_context($post_id);
        if (!empty($woo_context)) {
            $context .= ' ' . $woo_context;
            // Force enable WooCommerce integration for this request
            update_option('aialttext_token_woocommerce_integration', 1);
        }
    }
    
    return trim($context);
}

/**
 * Get SEO context from popular SEO plugins
 * 
 * @param int $post_id The post ID
 * @return string SEO context
 */
function aialttext_token_get_seo_context($post_id) {
    if (!$post_id) {
        return '';
    }
    
    $context = '';
    
    // Check for Yoast SEO
    if (defined('WPSEO_VERSION')) {
        // Get focus keyphrase
        $focus_keyphrase = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        if (!empty($focus_keyphrase)) {
            $context .= sprintf('The main SEO keyphrase for this page is "%s".', $focus_keyphrase);
        }
        
        // Get meta description
        $meta_description = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
        if (!empty($meta_description)) {
            $context .= sprintf(' The page meta description is: "%s".', $meta_description);
        }
    }
    
    // Check for Rank Math
    if (class_exists('RankMath')) {
        // Get focus keyphrase
        $focus_keyphrase = get_post_meta($post_id, 'rank_math_focus_keyword', true);
        if (!empty($focus_keyphrase)) {
            $context .= sprintf('The main SEO keyphrase for this page is "%s".', $focus_keyphrase);
        }
        
        // Get meta description
        $meta_description = get_post_meta($post_id, 'rank_math_description', true);
        if (!empty($meta_description)) {
            $context .= sprintf(' The page meta description is: "%s".', $meta_description);
        }
    }
    
    // Check for All in One SEO
    if (class_exists('AIOSEO\Plugin\Common\Main')) {
        // Get focus keyphrase
        $focus_keyphrase = get_post_meta($post_id, '_aioseo_focus_keyphrase', true);
        if (!empty($focus_keyphrase)) {
            // Parse JSON if necessary
            if (strpos($focus_keyphrase, '{') === 0) {
                $keyphrase_data = json_decode($focus_keyphrase, true);
                if (isset($keyphrase_data['keyphrase'])) {
                    $focus_keyphrase = $keyphrase_data['keyphrase'];
                }
            }
            
            if (!empty($focus_keyphrase)) {
                $context .= sprintf('The main SEO keyphrase for this page is "%s".', $focus_keyphrase);
            }
        }
        
        // Get meta description
        $meta_description = get_post_meta($post_id, '_aioseo_description', true);
        if (!empty($meta_description)) {
            $context .= sprintf(' The page meta description is: "%s".', $meta_description);
        }
    }
    
    // Check for SEOPress
    if (defined('SEOPRESS_VERSION')) {
        // Get target keywords
        $target_kw = get_post_meta($post_id, '_seopress_analysis_target_kw', true);
        if (!empty($target_kw)) {
            $context .= sprintf('The main SEO keyphrase for this page is "%s".', $target_kw);
        }
        
        // Get meta description
        $meta_description = get_post_meta($post_id, '_seopress_titles_desc', true);
        if (!empty($meta_description)) {
            $context .= sprintf(' The page meta description is: "%s".', $meta_description);
        }
    }
    
    // Add post title and excerpt if no SEO plugins are active
    if (empty($context)) {
        $post = get_post($post_id);
        if ($post) {
            $context .= sprintf('The page title is "%s".', $post->post_title);
            
            if (!empty($post->post_excerpt)) {
                $context .= sprintf(' The page excerpt is: "%s".', $post->post_excerpt);
            }
        }
    }
    
    return $context;
}

/**
 * Get WooCommerce product context
 * 
 * @param int $post_id The post ID
 * @return string WooCommerce context
 */
function aialttext_token_get_woocommerce_context($post_id) {
    if (!$post_id || !class_exists('WooCommerce')) {
        return '';
    }
    
    // Check if this is a product
    $product = wc_get_product($post_id);
    if (!$product) {
        return '';
    }
    
    $context = '';
    
    // Add product name
    $context .= sprintf('This is a product image for "%s".', $product->get_name());
    
    // Add product categories
    $categories = wp_get_post_terms($post_id, 'product_cat', array('fields' => 'names'));
    if (!empty($categories) && !is_wp_error($categories)) {
        $context .= sprintf(' Product categories: %s.', implode(', ', $categories));
    }
    
    // Add product tags
    $tags = wp_get_post_terms($post_id, 'product_tag', array('fields' => 'names'));
    if (!empty($tags) && !is_wp_error($tags)) {
        $context .= sprintf(' Product tags: %s.', implode(', ', $tags));
    }
    
    // Add product short description
    $short_description = $product->get_short_description();
    if (!empty($short_description)) {
        $short_description = wp_strip_all_tags($short_description);
        $context .= sprintf(' Product description: %s.', $short_description);
    }
    
    // Add product attributes if available
    $attributes = $product->get_attributes();
    if (!empty($attributes)) {
        $attr_text = array();
        foreach ($attributes as $attribute) {
            if ($attribute->get_visible()) {
                $name = wc_attribute_label($attribute->get_name());
                
                if ($attribute->is_taxonomy()) {
                    $terms = wp_get_post_terms($product->get_id(), $attribute->get_name(), array('fields' => 'names'));
                    if (!is_wp_error($terms)) {
                        $attr_text[] = $name . ': ' . implode(', ', $terms);
                    }
                } else {
                    $options = $attribute->get_options();
                    if (!empty($options)) {
                        $attr_text[] = $name . ': ' . $options[0];
                    }
                }
            }
        }
        
        if (!empty($attr_text)) {
            $context .= sprintf(' Product attributes: %s.', implode('; ', $attr_text));
        }
    }
    
    return $context;
}

// Media Library column indicator
add_filter('manage_upload_columns', function($cols){
    if (get_option('aialttext_token_show_indicators', 1)){
        $before = array();
        foreach($cols as $key => $label){
            $before[$key] = $label;
            if ($key === 'title'){ 
                $before['aialttext_token_alt_status'] = __('Alt Text','aialttext-token'); 
            }
        }
        return $before;
    }
    return $cols;
});

add_action('manage_media_custom_column', function($column_name, $post_id){
    if ($column_name !== 'aialttext_token_alt_status' || !wp_attachment_is_image($post_id)) return;
    $alt = get_post_meta($post_id, '_wp_attachment_image_alt', true);
    if (!empty($alt)){
        echo '<span class="aialttext-token-dot ok" title="Has alt text"></span>';
    } else {
        echo '<span class="aialttext-token-dot missing" title="Missing alt text"></span>';
    }
}, 10, 2);