<?php
if (!defined('ABSPATH')) exit;

class AiAltText_Token_Image_Processor {
    public static function process_attachment($attachment_id, $context_data=array()){
        $url = wp_get_attachment_url($attachment_id);
        if (!$url) {
            return new WP_Error('no_url', 'Missing attachment URL');
        }
        
        $title = get_the_title($attachment_id);
        $context = isset($context_data['context']) ? $context_data['context'] : aialttext_token_build_context_for_attachment($attachment_id);
        
        $api = new AiAltText_Token_API_Client();
        $result = $api->generate($url, $title, $context);
        
        if (empty($result['success'])){
            $error_message = isset($result['error']['message']) ? $result['error']['message'] : 'API error';
            return new WP_Error('api_error', $error_message);
        }
        
        $alt = trim($result['alt_text']);
        update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
        
        return array('alt' => $alt);
    }
}

function aialttext_token_scan_media_library($batch_size = 10, $offset = 0) {
    global $wpdb;

    // More reliable query that handles cases where the meta row doesn't exist at all
    $query = $wpdb->prepare(
        "SELECT p.ID, p.post_title, p.guid, p.post_mime_type
        FROM $wpdb->posts p
        WHERE p.post_type = 'attachment'
        AND p.post_mime_type LIKE 'image/%'
        AND p.ID NOT IN (
            SELECT post_id FROM $wpdb->postmeta 
            WHERE meta_key = '_wp_attachment_image_alt' 
            AND meta_value != ''
        )
        ORDER BY p.ID DESC
        LIMIT %d OFFSET %d",
        $batch_size,
        $offset
    );

    $results = $wpdb->get_results($query);
    
    // Log what we found for debugging
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log("Found " . count($results) . " images without alt text (batch size: $batch_size, offset: $offset)");
        
        // Log image IDs for debugging
        if ($results) {
            $ids = array_map(function($item) { return $item->ID; }, $results);
            error_log("Image IDs in this batch: " . implode(', ', $ids));
        }
    }
    
    return $results;
}

function aialttext_token_update_alt_text($image_id, $alt_text) {
    return update_post_meta($image_id, '_wp_attachment_image_alt', $alt_text);
}

function aialttext_token_process_images($offset, $batch_size) {
    $images = aialttext_token_scan_media_library($batch_size, 0);
    $processed = 0;
    $errors = 0;
    $error_messages = array();

    // If no images found, try to recover by returning a neutral result
    if (empty($images)) {
        $total_images = aialttext_token_get_total_images_without_alt();
        
        // If we still have images to process but none were returned in this batch,
        // it might be a database issue - just move to the next batch
        if ($total_images > 0 && $offset < $total_images) {
            return array(
                'success' => true,
                'processed' => 0,
                'errors' => 1,
                'error_messages' => array("No images found in this batch, moving to next batch"),
                'offset' => $offset + $batch_size,
                'total_images' => $total_images,
                'progress' => min(($offset + $batch_size) / max(1, $total_images) * 100, 100),
                'completed' => ($offset + $batch_size) >= $total_images || $total_images == 0,
            );
        }
    }

    foreach ($images as $image) {
        try {
            $image_id = $image->ID;
            $image_url = wp_get_attachment_url($image_id);
            $image_title = get_the_title($image_id);

            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log("Processing image ID: $image_id, URL: $image_url, Title: $image_title");
            }

            // Use the token-based processor
            $result = AiAltText_Token_Image_Processor::process_attachment($image_id);
            
            if (!is_wp_error($result)) {
                $processed++;
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log("Successfully updated alt text for image ID: $image_id");
                }
            } else {
                $errors++;
                $error_message = "Failed to generate alt text for image $image_id: " . $result->get_error_message();
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log($error_message);
                }
                $error_messages[] = $error_message;
            }
            
            // Small delay to prevent API rate limiting
            usleep(100000); // 100ms delay
            
        } catch (Exception $e) {
            $errors++;
            $error_message = "Exception processing image {$image->ID}: " . $e->getMessage();
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log($error_message);
            }
            $error_messages[] = $error_message;
            // Continue to next image
            continue;
        }
    }

    $total_images = aialttext_token_get_total_images_without_alt();
    $progress = min(($offset + $batch_size) / max(1, $total_images) * 100, 100);
    
    // Ensure we don't divide by zero
    if ($total_images == 0) {
        $progress = 100;
    }

    return array(
        'success' => true,
        'processed' => $processed,
        'errors' => $errors,
        'error_messages' => $error_messages,
        'offset' => $offset + $batch_size,
        'total_images' => $total_images,
        'progress' => $progress,
        'completed' => ($offset + $batch_size) >= $total_images || $total_images == 0,
    );
}

function aialttext_token_get_total_images_without_alt() {
    global $wpdb;
    return $wpdb->get_var(
        "SELECT COUNT(p.ID) FROM $wpdb->posts p
        LEFT JOIN $wpdb->postmeta pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
        WHERE p.post_type = 'attachment'
        AND p.post_mime_type LIKE 'image/%'
        AND (pm.meta_value IS NULL OR pm.meta_value = '')"
    );
}

/**
 * Modified process_images function to accept batch size parameter and use tokens
 */
function aialttext_token_process_images_with_params($offset, $batch_size, $max_images = 0) {
    // Use the custom batch size if provided
    $batch_size = absint($batch_size);
    if ($batch_size <= 0) {
        $batch_size = 25; // Default
    }
    
    // Adjust batch size if max_images is set
    if ($max_images > 0) {
        $total_processed = $offset;
        $remaining = $max_images - $total_processed;
        if ($remaining < $batch_size) {
            $batch_size = $remaining;
        }
        
        // If no more to process, return completed
        if ($batch_size <= 0) {
            return array(
                'success' => true,
                'processed' => 0,
                'errors' => 0,
                'error_messages' => array(),
                'offset' => $offset,
                'total_images' => $max_images,
                'total_processed' => $offset,
                'progress' => 100,
                'completed' => true
            );
        }
    }
    
    $images = aialttext_token_scan_media_library($batch_size, 0);
    $processed = 0;
    $errors = 0;
    $error_messages = array();
    $latest_image = null;

    // If no images found, try to recover by returning a neutral result
    if (empty($images)) {
        $total_images = aialttext_token_get_total_images_without_alt();
        
        // If max_images is set, adjust total
        if ($max_images > 0 && $total_images > $max_images) {
            $total_images = $max_images;
        }
        
        // If we still have images to process but none were returned in this batch,
        // it might be a database issue - just move to the next batch
        if ($total_images > 0 && $offset < $total_images) {
            return array(
                'success' => true,
                'processed' => 0,
                'errors' => 1,
                'error_messages' => array("No images found in this batch, moving to next batch"),
                'offset' => $offset + $batch_size,
                'total_images' => $total_images,
                'total_processed' => $offset,
                'progress' => min(($offset + $batch_size) / max(1, $total_images) * 100, 100),
                'completed' => ($offset + $batch_size) >= $total_images || $total_images == 0,
            );
        }
    }

    foreach ($images as $image) {
        try {
            $image_id = $image->ID;
            $image_url = wp_get_attachment_url($image_id);
            $image_title = get_the_title($image_id);

            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log("Processing image ID: $image_id, URL: $image_url, Title: $image_title");
            }

            // Get additional context
            $context = aialttext_token_get_image_context($image_id);

            // Use the token-based processor
            $result = AiAltText_Token_Image_Processor::process_attachment($image_id, array('context' => $context));
            
            if (!is_wp_error($result)) {
                $processed++;
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log("Successfully updated alt text for image ID: $image_id");
                }
                
                // Store latest successfully processed image
                $latest_image = array(
                    'id' => $image_id,
                    'url' => wp_get_attachment_thumb_url($image_id),
                    'title' => $image_title,
                    'alt_text' => $result['alt']
                );
            } else {
                $errors++;
                $error_message = "Failed to generate alt text for image $image_id: " . $result->get_error_message();
                if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                    error_log($error_message);
                }
                $error_messages[] = $error_message;
            }
            
            // Small delay to prevent API rate limiting
            usleep(100000); // 100ms delay
            
        } catch (Exception $e) {
            $errors++;
            $error_message = "Exception processing image {$image->ID}: " . $e->getMessage();
            if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
                error_log($error_message);
            }
            $error_messages[] = $error_message;
            // Continue to next image
            continue;
        }
    }

    $total_images = aialttext_token_get_total_images_without_alt();
    
    // If max_images is set, adjust total
    if ($max_images > 0 && $total_images > $max_images) {
        $total_images = $max_images;
    }
    
    $progress = min(($offset + $batch_size) / max(1, $total_images) * 100, 100);
    
    // Ensure we don't divide by zero
    if ($total_images == 0) {
        $progress = 100;
    }

    return array(
        'success' => true,
        'processed' => $processed,
        'errors' => $errors,
        'error_messages' => $error_messages,
        'offset' => $offset + $batch_size,
        'total_images' => $total_images,
        'total_processed' => $offset + $processed,
        'progress' => $progress,
        'completed' => aialttext_token_get_total_images_without_alt() === 0,
        'latest_image' => $latest_image
    );
}

// Auto-generate on upload
add_action('add_attachment', function($post_id){
    if (!get_option('aialttext_token_auto_generate', 0)) return;
    if (get_post_type($post_id) !== 'attachment') return;
    $mime = get_post_mime_type($post_id); 
    if (strpos($mime, 'image/') !== 0) return;
    AiAltText_Token_Image_Processor::process_attachment($post_id);
}, 10, 1);

// Attachment screen button
add_action('attachment_submitbox_misc_actions', function($post){
    if ($post->post_mime_type && strpos($post->post_mime_type, 'image/') === 0){
        echo '<div class="misc-pub-section"><button type="button" class="button" id="aialttext-token-generate-single" data-id="'.esc_attr($post->ID).'">Generate AI Alt Text</button> <span id="aialttext-token-single-status"></span></div>';
    }
});

// AJAX generate single
add_action('wp_ajax_aialttext_token_generate_single', function(){
    if (!current_user_can('upload_files')) {
        wp_send_json_error(['message' => 'Permission denied']);
        return;
    }
    if (false === check_ajax_referer('aialttext_token_nonce', 'nonce', false)) {
        wp_send_json_error(['message' => 'Security check failed']);
        return;
    }
    $id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
    if (!$id) {
        wp_send_json_error(array('message' => 'Missing attachment'));
    }
    
    $result = AiAltText_Token_Image_Processor::process_attachment($id);
    if (is_wp_error($result)) {
        wp_send_json_error(array('message' => $result->get_error_message()));
    }
    
    wp_send_json_success(array('alt' => $result['alt']));
});