<?php
if (!defined('ABSPATH')) exit;

function aialttext_token_account_page(){
    if (!current_user_can('manage_options')) return;
    
    $email = get_option('aialttext_token_account_email', get_option('admin_email'));
    $access_key = get_option('aialttext_token_access_key', '');
    $masked_key = $access_key ? substr($access_key, 0, 6) . '••••••' . substr($access_key, -4) : '—';
    $show_success = (!empty($_GET['payment']) && $_GET['payment'] === 'success' && !empty($_GET['session_id']));
    
    ?>
    <div class="wrap aialttext-token-wrap">
        <h1>AI Alt Text Generator - Account & Tokens</h1>
        <?php if ($show_success): ?>
            <div class="notice notice-warning is-dismissible">
                <p>Thank you for your purchase! We're confirming your payment and updating your balance…</p>
            </div>
        <?php endif; ?>
        
        <div class="aialttext-token-grid">
            <div class="aialttext-token-card">
                <h2>Account</h2>
                <table class="form-table">
                    <tr>
                        <th>Email</th>
                        <td>
                            <input type="email" id="aialttext_token_account_email" value="<?php echo esc_attr($email); ?>" class="regular-text" /> 
                            <button id="aialttext_token_link_account_btn" class="button button-secondary">Link / Relink</button>
                        </td>
                    </tr>
                    <tr>
                        <th>Token Balance</th>
                        <td>
                            <span id="aialttext-token-balance">—</span> 
                            <a href="#" id="aialttext_token_refresh_balance">Refresh</a>
                        </td>
                    </tr>
                    <tr>
                        <th>Access Key</th>
                        <td><code><?php echo esc_html($masked_key); ?></code></td>
                    </tr>
                </table>
            </div>
            
            <div class="aialttext-token-card">
                <h2>Purchase More Tokens</h2>
                <div id="aialttext_token_pricing" class="aialttext-token-pricing"></div>
            </div>
        </div>
        
        <div class="aialttext-token-card">
            <h2>Usage Statistics</h2>
            <div id="aialttext_token_usage_stats">
                <button id="aialttext_token_load_usage" class="button button-secondary">Load Usage History</button>
                <div id="aialttext_token_usage_list"></div>
            </div>
        </div>
    </div>
    <?php
}

function aialttext_token_settings_page(){
    if (!current_user_can('manage_options')) return;
    
    if (!empty($_POST['aialttext_token_save']) && check_admin_referer('aialttext_token_save')){
        update_option('aialttext_token_auto_generate', !empty($_POST['auto_generate']) ? 1 : 0);
        update_option('aialttext_token_show_indicators', !empty($_POST['show_indicators']) ? 1 : 0);
        update_option('aialttext_token_batch_size', max(1, intval(isset($_POST['batch_size']) ? $_POST['batch_size'] : 25)));
        update_option('aialttext_token_max_images', max(0, intval(isset($_POST['max_images']) ? $_POST['max_images'] : 0)));
        update_option('aialttext_token_prompt_template', sanitize_textarea_field(isset($_POST['prompt_template']) ? $_POST['prompt_template'] : ''));
        update_option('aialttext_token_woocommerce_integration', !empty($_POST['woocommerce_integration']) ? 1 : 0);
        update_option('aialttext_token_seo_integration', !empty($_POST['seo_integration']) ? 1 : 0);
        
        echo '<div class="notice notice-success is-dismissible"><p>Settings saved.</p></div>';
    }
    
    $auto = get_option('aialttext_token_auto_generate', 0);
    $ind = get_option('aialttext_token_show_indicators', 1);
    $batch = get_option('aialttext_token_batch_size', 25);
    $max = get_option('aialttext_token_max_images', 0);
    $tmpl = get_option('aialttext_token_prompt_template', 'Describe the image clearly for accessibility. Title: {title}. Context: {context}. Return only the alt text.');
    $woo = get_option('aialttext_token_woocommerce_integration', 1);
    $seo = get_option('aialttext_token_seo_integration', 1);
    
    ?>
    <div class="wrap aialttext-token-wrap">
        <h1>AI Alt Text Generator - Settings</h1>
        <div class="aialttext-token-card">
            <form method="post">
                <?php wp_nonce_field('aialttext_token_save'); ?>
                <table class="form-table">
                    <tr>
                        <th>Auto-generate on upload</th>
                        <td>
                            <label>
                                <input type="checkbox" name="auto_generate" value="1" <?php checked($auto, 1); ?> /> 
                                Enable automatic alt text generation when images are uploaded
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th>Show indicators in Media Library</th>
                        <td>
                            <label>
                                <input type="checkbox" name="show_indicators" value="1" <?php checked($ind, 1); ?> /> 
                                Show visual indicators for images with/without alt text
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th>WooCommerce Integration</th>
                        <td>
                            <label>
                                <input type="checkbox" name="woocommerce_integration" value="1" <?php checked($woo, 1); ?> /> 
                                Include product information in alt text generation
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th>SEO Integration</th>
                        <td>
                            <label>
                                <input type="checkbox" name="seo_integration" value="1" <?php checked($seo, 1); ?> /> 
                                Include SEO keywords and meta descriptions in context
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th>Bulk batch size</th>
                        <td>
                            <input type="number" min="1" max="200" name="batch_size" value="<?php echo esc_attr($batch); ?>" /> 
                            images per batch
                        </td>
                    </tr>
                    <tr>
                        <th>Max images per session</th>
                        <td>
                            <input type="number" min="0" name="max_images" value="<?php echo esc_attr($max); ?>" /> 
                            <em>0 = unlimited</em>
                        </td>
                    </tr>
                    <tr>
                        <th>Prompt template</th>
                        <td>
                            <textarea name="prompt_template" rows="4" class="large-text"><?php echo esc_textarea($tmpl); ?></textarea>
                            <p class="description">Variables: {title}, {context}</p>
                        </td>
                    </tr>
                </table>
                <p>
                    <button class="button button-primary">Save Settings</button>
                    <input type="hidden" name="aialttext_token_save" value="1" />
                </p>
            </form>
        </div>
    </div>
    <?php
}

function aialttext_token_process_page(){
    if (!current_user_can('manage_options')) return;
    
    ?>
    <div class="wrap aialttext-token-wrap">
        <h1>Process Images</h1>
        <div class="aialttext-token-card">
            <p>This tool scans your Media Library for images that are missing alt text and generates AI alt text at a rate of <strong>1 token per image</strong>.</p>
            
            <div class="aialttext-token-controls">
                <button class="button button-primary" id="aialttext_token_process_button">Scan Images</button>
                <button class="button button-secondary" id="aialttext_token_stop_button" style="display:none;">Stop Processing</button>
            </div>
            
            <div id="aialttext_token_progress_container" class="aialttext-token-progress-container">
                <div class="aialttext-token-progress-stats">
                    <div class="aialttext-token-stat-box">
                        <span class="aialttext-token-stat-label">Total Images:</span>
                        <span id="aialttext_token_total_images" class="aialttext-token-stat-value">0</span>
                    </div>
                    <div class="aialttext-token-stat-box">
                        <span class="aialttext-token-stat-label">Processed:</span>
                        <span id="aialttext_token_processed_count" class="aialttext-token-stat-value">0</span>
                    </div>
                    <div class="aialttext-token-stat-box">
                        <span class="aialttext-token-stat-label">Errors:</span>
                        <span id="aialttext_token_error_count" class="aialttext-token-stat-value">0</span>
                    </div>
                    <div class="aialttext-token-stat-box">
                        <span class="aialttext-token-stat-label">Time Remaining:</span>
                        <span id="aialttext_token_time_remaining" class="aialttext-token-stat-value">Unknown</span>
                    </div>
                </div>
                
                <div id="aialttext_token_progress_bar" class="aialttext-token-progress-bar" style="display:none;">
                    <div id="aialttext_token_progress" class="aialttext-token-progress"></div>
                </div>
                
                <div id="aialttext_token_process_status" class="aialttext-token-process-status"></div>
                <div id="aialttext_token_error_details" class="aialttext-token-error-details"></div>
                
                <div id="aialttext_token_latest_image" class="aialttext-token-latest-image" style="display:none;">
                    <h3>Latest Processed Image</h3>
                    <div class="aialttext-token-image-preview">
                        <img id="aialttext_token_preview_image" src="" alt="">
                    </div>
                    <div class="aialttext-token-image-details">
                        <p><strong>Title:</strong> <span id="aialttext_token_preview_title"></span></p>
                        <p><strong>Generated Alt Text:</strong> <span id="aialttext_token_preview_alt"></span></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
}

function aialttext_token_help_page(){
    ?>
    <div class="wrap aialttext-token-wrap">
        <h1>AI Alt Text Generator - Help & Documentation</h1>
        
        <div class="aialttext-token-card">
            <h2>Help & Documentation</h2>
            
            <div class="aialttext-token-help-section">
                <h3>Getting Started</h3>
                <ol>
                    <li>Go to the <a href="<?php echo admin_url('admin.php?page=aialttext-token'); ?>">Account & Tokens</a> page and link your account with your email address.</li>
                    <li>You'll receive 10 free tokens to test the service.</li>
                    <li>Configure your settings in the <a href="<?php echo admin_url('admin.php?page=aialttext-token-settings'); ?>">Settings</a> page.</li>
                    <li>Go to the <a href="<?php echo admin_url('admin.php?page=aialttext-token-process'); ?>">Process Images</a> page to start generating alt text.</li>
                </ol>
            </div>
            
            <div class="aialttext-token-help-section">
                <h3>Token-Based System</h3>
                <p>This plugin uses a token-based system where:</p>
                <ul>
                    <li><strong>1 token = 1 image</strong> alt text generation</li>
                    <li>New accounts receive 10 free tokens to test the service</li>
                    <li>Additional tokens can be purchased through the Account page</li>
                    <li>All processing is done on our secure servers using advanced AI models</li>
                </ul>
            </div>
            
            <div class="aialttext-token-help-section">
                <h3>Features</h3>
                <ul>
                    <li><strong>Automatic Generation:</strong> Enable auto-generation for newly uploaded images</li>
                    <li><strong>Bulk Processing:</strong> Process multiple images at once</li>
                    <li><strong>WooCommerce Integration:</strong> Product-specific alt text using product details</li>
                    <li><strong>SEO Integration:</strong> Uses keywords and meta descriptions for context</li>
                    <li><strong>Visual Indicators:</strong> See which images have alt text in your media library</li>
                    <li><strong>Gutenberg Support:</strong> Generate alt text directly in the block editor</li>
                </ul>
            </div>
            
            <div class="aialttext-token-help-section">
                <h3>Frequently Asked Questions</h3>
                
                <div class="aialttext-token-faq">
                    <h4>What is alt text and why is it important?</h4>
                    <div class="aialttext-token-faq-answer">
                        <p>Alt text (alternative text) provides a text description of an image for screen readers, improving accessibility for visually impaired users. It also helps search engines understand image content, potentially improving SEO.</p>
                    </div>
                </div>
                
                <div class="aialttext-token-faq">
                    <h4>How does the token system work?</h4>
                    <div class="aialttext-token-faq-answer">
                        <p>Each image processed uses exactly 1 token. New accounts receive 10 free tokens. Additional tokens can be purchased in various packages through the Account page.</p>
                    </div>
                </div>
                
                <div class="aialttext-token-faq">
                    <h4>Will this affect images that already have alt text?</h4>
                    <div class="aialttext-token-faq-answer">
                        <p>No. The plugin only processes images that don't already have alt text, so you won't waste tokens on images that are already optimized.</p>
                    </div>
                </div>
                
                <div class="aialttext-token-faq">
                    <h4>What happens if I run out of tokens?</h4>
                    <div class="aialttext-token-faq-answer">
                        <p>The plugin will stop processing new images when you run out of tokens. You can purchase more tokens through the Account page to continue using the service.</p>
                    </div>
                </div>
                
                <div class="aialttext-token-faq">
                    <h4>Is my data secure?</h4>
                    <div class="aialttext-token-faq-answer">
                        <p>Yes. Images are processed securely on our servers and are not stored permanently. Only the generated alt text is returned to your site.</p>
                    </div>
                </div>
            </div>
            
            <div class="aialttext-token-help-section">
                <h3>Support</h3>
                <p>If you need additional help, please contact our support team:</p>
                <ul>
                    <li>Email: <a href="mailto:contact@arcticfoxdevelopments.com">contact@arcticfoxdevelopments.com</a></li>
                    <li>Documentation: <a href="https://imgaltgen.com/docs" target="_blank">https://imgaltgen.com/docs</a></li>
                </ul>
            </div>
        </div>
    </div>
    <?php
}

// Ajax: account UI
add_action('wp_ajax_aialttext_token_link_account', 'aialttext_token_link_account');
add_action('wp_ajax_aialttext_token_refresh_balance', 'aialttext_token_refresh_balance');
add_action('wp_ajax_aialttext_token_get_pricing', 'aialttext_token_get_pricing');
add_action('wp_ajax_aialttext_token_purchase_tokens', 'aialttext_token_purchase_tokens');
add_action('wp_ajax_aialttext_token_confirm_payment', 'aialttext_token_confirm_payment');
add_action('wp_ajax_aialttext_token_load_usage', 'aialttext_token_load_usage');

// Ajax: processing
add_action('wp_ajax_aialttext_token_list_missing', 'aialttext_token_list_missing');

function aialttext_token_check_nonce(){ 
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed')); 
    }
}

function aialttext_token_link_account(){
    aialttext_token_check_nonce();
    $email = sanitize_email(isset($_POST['email']) ? $_POST['email'] : '');
    if (!$email) wp_send_json_error(array('message' => 'Email required'));
    
    update_option('aialttext_token_account_email', $email);
    $api = new AiAltText_Token_API_Client();
    $result = $api->link_account($email, parse_url(get_site_url(), PHP_URL_HOST));
    
    if (!empty($result['success'])){
        if (!empty($result['access_key'])) {
            update_option('aialttext_token_access_key', $result['access_key']);
        }
        wp_send_json_success(array(
            'message' => isset($result['message']) ? $result['message'] : 'Linked',
            'balance' => isset($result['token_balance']) ? $result['token_balance'] : 0
        ));
    }
    
    wp_send_json_error(array('message' => isset($result['error']['message']) ? $result['error']['message'] : 'Failed to link'));
}

function aialttext_token_refresh_balance(){
    aialttext_token_check_nonce();
    $api = new AiAltText_Token_API_Client(); 
    $result = $api->status();
    
    if (!empty($result['success'])) {
        wp_send_json_success(array('balance' => $result['token_balance']));
    }
    
    wp_send_json_error(array('message' => isset($result['error']['message']) ? $result['error']['message'] : 'Unable to fetch balance'));
}

function aialttext_token_get_pricing(){ 
    aialttext_token_check_nonce(); 
    $result = aialttext_token_cached_pricing(); 
    
    if (!empty($result['success'])) {
        wp_send_json_success($result); 
    }
    
    wp_send_json_error(array('message' => isset($result['error']['message']) ? $result['error']['message'] : 'Pricing not available')); 
}

function aialttext_token_purchase_tokens(){
    aialttext_token_check_nonce();
    $pkg = sanitize_text_field(isset($_POST['package_id']) ? $_POST['package_id'] : '');
    if (!$pkg) wp_send_json_error(array('message' => 'Missing package'));
    
    $base = admin_url('admin.php?page=aialttext-token');
    $success = $base . (strpos($base, '?') !== false ? '&' : '?') . 'payment=success&session_id={CHECKOUT_SESSION_ID}';
    $cancel = $base . (strpos($base, '?') !== false ? '&' : '?') . 'payment=cancelled';
    
    $api = new AiAltText_Token_API_Client();
    $result = $api->purchase($pkg, $success, $cancel, parse_url(get_site_url(), PHP_URL_HOST));
    
    if (!empty($result['success']) && !empty($result['checkout_url'])) {
        wp_send_json_success(array(
            'checkout_url' => $result['checkout_url'],
            'session_id' => $result['session_id']
        ));
    }
    
    wp_send_json_error(array('message' => isset($result['error']['message']) ? $result['error']['message'] : 'Payment could not be initiated'));
}

function aialttext_token_confirm_payment(){
    aialttext_token_check_nonce();
    $session_id = sanitize_text_field(isset($_POST['session_id']) ? $_POST['session_id'] : '');
    if (!$session_id) wp_send_json_error(array('message' => 'Missing session_id'));
    
    $api = new AiAltText_Token_API_Client();
    $result = $api->confirm($session_id);
    
    if (!empty($result['success'])){
        $status_result = $api->status();
        $balance = !empty($status_result['success']) ? (isset($status_result['token_balance']) ? $status_result['token_balance'] : null) : null;
        $added = isset($result['tokens_added']) ? intval($result['tokens_added']) : 0;
        $pkg = isset($result['package_tokens']) ? intval($result['package_tokens']) : $added;
        
        wp_send_json_success(array(
            'message' => ($added > 0 ? 'Payment confirmed. ' . $added . ' tokens added.' : 'Payment confirmed. ' . $pkg . ' tokens added.'),
            'tokens_added' => $added,
            'package_tokens' => $pkg,
            'balance' => $balance
        ));
    }
    
    wp_send_json_error(array('message' => isset($result['error']['message']) ? $result['error']['message'] : 'Payment confirmation failed'));
}

function aialttext_token_load_usage(){
    aialttext_token_check_nonce();
    $api = new AiAltText_Token_API_Client();
    $result = $api->usage();
    
    if (!empty($result['success'])) {
        wp_send_json_success($result);
    }
    
    wp_send_json_error(array('message' => isset($result['error']['message']) ? $result['error']['message'] : 'Unable to load usage data'));
}

function aialttext_token_list_missing(){
    aialttext_token_check_nonce();
    global $wpdb;
    
    $max = max(0, intval(get_option('aialttext_token_max_images', 0)));
    $limit_sql = $max > 0 ? $wpdb->prepare("LIMIT %d", $max) : "";
    
    $ids = $wpdb->get_col("
        SELECT p.ID
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} m ON m.post_id=p.ID AND m.meta_key='_wp_attachment_image_alt'
        WHERE p.post_type='attachment' AND p.post_mime_type LIKE 'image/%'
        AND (m.meta_value IS NULL OR m.meta_value='')
        ORDER BY p.ID DESC
        $limit_sql
    ");
    
    wp_send_json_success(array('ids' => array_map('intval', $ids)));
}