<?php
/*
Plugin Name: AI Alt Image Text Generator (Token-Based)
Description: Analyzes images in the media library and adds alt text using a managed token service. Full-featured version with token integration.
Version: 2.1.3
Author: Arctic Fox Developments | Lead Developer - Jessica L Dale
*/

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('AIALTTEXT_TOKEN_PATH', plugin_dir_path(__FILE__));
define('AIALTTEXT_TOKEN_URL', plugin_dir_url(__FILE__));
define('AIALTTEXT_TOKEN_VERSION', '2.1.3');
define('AIALTTEXT_TOKEN_BASENAME', plugin_basename(__FILE__));

// Debug mode
define('AIALTTEXT_TOKEN_DEBUG', false); // Set to true to enable debug logging

// API endpoint for token management
if (!defined('AIALTTEXT_TOKEN_API_BASE')) {
    define('AIALTTEXT_TOKEN_API_BASE', 'https://arcticfoxdevelopments.com/wp-admin/admin-ajax.php');
}

// Ensure includes directory exists
if (!file_exists(AIALTTEXT_TOKEN_PATH . 'includes/')) {
    mkdir(AIALTTEXT_TOKEN_PATH . 'includes/', 0755, true);
}
if (!file_exists(AIALTTEXT_TOKEN_PATH . 'admin/')) {
    mkdir(AIALTTEXT_TOKEN_PATH . 'admin/', 0755, true);
}

function aialttext_token_debug_log($message) {
    if (defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        if (is_array($message) || is_object($message)) {
            error_log('AI Alt Text Token Debug: ' . print_r($message, true));
        } else {
            error_log('AI Alt Text Token Debug: ' . $message);
        }
    }
}

// Include necessary files
require_once AIALTTEXT_TOKEN_PATH . 'includes/settings-page.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/api-client.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/image-processor.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/integrations.php';
require_once AIALTTEXT_TOKEN_PATH . 'includes/admin-pages.php';

// Hook for adding admin menus
add_action('admin_menu', 'aialttext_token_add_menu');

function aialttext_token_add_menu() {
    // Add main menu page in the sidebar
    add_menu_page(
        'AI Alt Text Generator',           // Page title
        'AI Alt Text',                     // Menu title
        'manage_options',                  // Capability
        'aialttext-token',                       // Menu slug
        'aialttext_token_account_page',         // Callback function
        'dashicons-images-alt2',           // Icon
        30                                 // Position
    );
    
    // Add submenu pages
    add_submenu_page(
        'aialttext-token',                       // Parent slug
        'Account & Tokens',                // Page title
        'Account & Tokens',                // Menu title
        'manage_options',                  // Capability
        'aialttext-token',                       // Menu slug (same as parent for first item)
        'aialttext_token_account_page'          // Callback function
    );
    
    add_submenu_page(
        'aialttext-token',                       // Parent slug
        'Settings',                        // Page title
        'Settings',                        // Menu title
        'manage_options',                  // Capability
        'aialttext-token-settings',               // Menu slug
        'aialttext_token_settings_page'          // Callback function
    );
    
    add_submenu_page(
        'aialttext-token',                       // Parent slug
        'Process Images',                  // Page title
        'Process Images',                  // Menu title
        'manage_options',                  // Capability
        'aialttext-token-process',               // Menu slug
        'aialttext_token_process_page'           // Callback function
    );
    
    add_submenu_page(
        'aialttext-token',                       // Parent slug
        'Help & Documentation',            // Page title
        'Help',                           // Menu title
        'manage_options',                  // Capability
        'aialttext-token-help',                  // Menu slug
        'aialttext_token_help_page'              // Callback function
    );
}

// Hook for registering plugin settings
add_action('admin_init', 'aialttext_token_register_settings');

function aialttext_token_register_settings() {
    register_setting('aialttext_token_options', 'aialttext_token_batch_size');
    register_setting('aialttext_token_options', 'aialttext_token_max_images');
    register_setting('aialttext_token_options', 'aialttext_token_prompt_template');
    register_setting('aialttext_token_options', 'aialttext_token_woocommerce_integration');
    register_setting('aialttext_token_options', 'aialttext_token_auto_generate');
    register_setting('aialttext_token_options', 'aialttext_token_seo_integration');
    register_setting('aialttext_token_options', 'aialttext_token_show_indicators');
    
    // Token-specific settings
    register_setting('aialttext_token_options', 'aialttext_token_account_email');
    register_setting('aialttext_token_options', 'aialttext_token_access_key');
}

// Hook for adding plugin action links
add_filter('plugin_action_links_' . AIALTTEXT_TOKEN_BASENAME, 'aialttext_token_add_action_links');

function aialttext_token_add_action_links($links) {
    $account_link = '<a href="' . admin_url('admin.php?page=aialttext-token') . '">' . __('Account') . '</a>';
    $settings_link = '<a href="' . admin_url('admin.php?page=aialttext-token-settings') . '">' . __('Settings') . '</a>';
    array_unshift($links, $account_link, $settings_link);
    return $links;
}

// Add hook for activation
register_activation_hook(__FILE__, 'aialttext_token_activate');

function aialttext_token_activate() {
    // Set default options
    add_option('aialttext_token_auto_generate', 0);
    add_option('aialttext_token_show_indicators', 1);
    add_option('aialttext_token_batch_size', 25);
    add_option('aialttext_token_max_images', 0);
    add_option('aialttext_token_prompt_template', "Describe the image clearly for accessibility. Title: {title}. Context: {context}. Return only the alt text.");
    add_option('aialttext_token_woocommerce_integration', 1);
    add_option('aialttext_token_seo_integration', 1);
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        @file_put_contents(WP_CONTENT_DIR.'/aialttext-token-debug.log','['.date('c').'] Activation complete'.PHP_EOL, FILE_APPEND);
    }
}

// Hook for deactivation
register_deactivation_hook(__FILE__, 'aialttext_token_deactivate');

function aialttext_token_deactivate() {
    // Deactivation tasks if needed
}

// Register a filter to prevent alt text being copied to caption/description on save
add_filter('wp_update_attachment_metadata', 'aialttext_token_sanitize_attachment_metadata', 10, 2);

function aialttext_token_sanitize_attachment_metadata($data, $attachment_id) {
    // Check if this is an image attachment
    if (!wp_attachment_is_image($attachment_id)) {
        return $data;
    }
    
    // Get the current alt text
    $alt_text = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if (empty($alt_text)) {
        return $data;
    }
    
    // Get the attachment post object
    $attachment = get_post($attachment_id);
    if (!$attachment) {
        return $data;
    }
    
    // Check if caption is the same as alt text and clear it
    if ($attachment->post_excerpt === $alt_text) {
        wp_update_post(array(
            'ID' => $attachment_id,
            'post_excerpt' => ''
        ));
    }
    
    // Check if description is the same as alt text and clear it
    if ($attachment->post_content === $alt_text) {
        wp_update_post(array(
            'ID' => $attachment_id,
            'post_content' => ''
        ));
    }
    
    return $data;
}

// Enqueue admin scripts and styles
add_action('admin_enqueue_scripts', 'aialttext_token_enqueue_admin_scripts');

function aialttext_token_enqueue_admin_scripts($hook) {
    // Check if we're on our plugin pages
    if (strpos($hook, 'aialttext-token') === false && $hook !== 'upload.php' && $hook !== 'post.php' && $hook !== 'attachment.php') {
        return;
    }

    wp_enqueue_script('aialttext-token-admin-js', AIALTTEXT_TOKEN_URL . 'admin/js/admin.js', array('jquery'), AIALTTEXT_TOKEN_VERSION, true);
    wp_enqueue_style('aialttext-token-admin-css', AIALTTEXT_TOKEN_URL . 'admin/css/admin.css', array(), AIALTTEXT_TOKEN_VERSION);

    wp_localize_script('aialttext-token-admin-js', 'aialttext_token_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('aialttext_token_nonce'),
        'batch_size' => get_option('aialttext_token_batch_size', 25),
        'max_images' => get_option('aialttext_token_max_images', 0)
    ));
}

// Add this new function to enqueue our media uploader integration script
function aialttext_token_enqueue_media_scripts() {
    // Only load on media pages or if media modal could be used
    if (wp_script_is('media-editor') || wp_script_is('media-upload')) {
        wp_enqueue_script('aialttext-token-media-js', AIALTTEXT_TOKEN_URL . 'admin/js/media.js', array('jquery', 'media-editor'), AIALTTEXT_TOKEN_VERSION, true);
        wp_localize_script('aialttext-token-media-js', 'aialttext_token_media', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aialttext_token_media_update'),
            'auto_generate' => get_option('aialttext_token_auto_generate', 0)
        ));
    }
}
add_action('admin_enqueue_scripts', 'aialttext_token_enqueue_media_scripts');

// Special AJAX handler to force update alt text and return current value
add_action('wp_ajax_aialttext_token_force_update_and_get_alt', 'aialttext_token_force_update_and_get_alt');

function aialttext_token_force_update_and_get_alt() {
    // Verify nonce
    if (false === check_ajax_referer('aialttext_token_media_update', 'nonce', false)) {
        wp_send_json_error(['message' => 'Security check failed']);
        return;
    }
    
    // Check permissions
    if (!current_user_can('upload_files')) {
        wp_send_json_error(array('message' => 'Permission denied'));
        return;
    }
    
    // Get image ID and alt text
    $image_id = isset($_POST['image_id']) ? intval($_POST['image_id']) : 0;
    $alt_text = isset($_POST['alt_text']) ? sanitize_text_field($_POST['alt_text']) : '';
    
    if (!$image_id) {
        wp_send_json_error(array('message' => 'Invalid image ID'));
        return;
    }
    
    // If alt text is provided, update it using multiple methods
    if (!empty($alt_text)) {
        // 1. Direct database update
        global $wpdb;
        $wpdb->delete($wpdb->postmeta, array('post_id' => $image_id, 'meta_key' => '_wp_attachment_image_alt'));
        $wpdb->insert($wpdb->postmeta, array(
            'post_id' => $image_id,
            'meta_key' => '_wp_attachment_image_alt',
            'meta_value' => $alt_text
        ));
        
        // 2. WordPress API
        update_post_meta($image_id, '_wp_attachment_image_alt', $alt_text);
        
        // 3. Clear caches
        clean_post_cache($image_id);
        if (function_exists('clean_attachment_cache')) {
            clean_attachment_cache($image_id);
        }
    }
    
    // Get the current alt text value
    $current_alt_text = get_post_meta($image_id, '_wp_attachment_image_alt', true);
    
    // Return the result
    wp_send_json_success(array(
        'alt_text' => $current_alt_text,
        'updated' => (!empty($alt_text)),
        'image_id' => $image_id
    ));
}

// Ajax handler for scanning images
add_action('wp_ajax_aialttext_token_scan_images_ajax', 'aialttext_token_scan_images_ajax');

function aialttext_token_scan_images_ajax() {
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
        return;
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
    }

    $total_images = aialttext_token_get_total_images_without_alt();

    wp_send_json(array(
        'success' => true,
        'total_images' => $total_images
    ));
}

// Ajax handler for processing images
add_action('wp_ajax_aialttext_token_process_images_ajax', 'aialttext_token_process_images_ajax');

function aialttext_token_process_images_ajax() {
    if (!wp_verify_nonce(isset($_POST['nonce']) ? $_POST['nonce'] : '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
        return;
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
    }

    $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
    $total_processed = isset($_POST['total_processed']) ? intval($_POST['total_processed']) : 0;
    
    // Get settings
    $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : get_option('aialttext_token_batch_size', 25);
    $max_images = isset($_POST['max_images']) ? intval($_POST['max_images']) : get_option('aialttext_token_max_images', 0);

    $max_execution_time = 45; // Maximum execution time in seconds
    $start_time = time();
    
    // Set a longer PHP execution time for this request
    set_time_limit(120);

    try {
        $result = aialttext_token_process_images_with_params($offset, $batch_size, $max_images);
        
        // Add total processed info to the response
        $result['total_processed'] = isset($_POST['total_processed']) ? intval($_POST['total_processed']) + $result['processed'] : $result['processed'];
        $result['time_remaining'] = ($result['total_images'] > 0 && $result['processed'] > 0) ? 
            round((($result['total_images'] - $result['total_processed']) / $result['processed']) * (time() - $start_time) / 60) : 'Unknown';

        wp_send_json($result);
    } catch (Exception $e) {
        // If there's an exception, return a partial success so processing can continue
        $result = array(
            'success' => true,
            'processed' => 0,
            'errors' => 1, 
            'error_messages' => array("Error during batch processing: " . $e->getMessage()),
            'offset' => $offset + $batch_size, // Still increment offset to move forward
            'total_images' => aialttext_token_get_total_images_without_alt(),
            'progress' => 0,
            'completed' => false // Force it to continue
        );
        
        $result['total_processed'] = isset($_POST['total_processed']) ? intval($_POST['total_processed']) : 0;
        
        wp_send_json($result);
    }
}

// Add action to media edit page
add_action('attachment_submitbox_misc_actions', 'aialttext_token_add_generate_button', 10, 1);

function aialttext_token_add_generate_button($post) {
    // Only show for image attachments
    if (!wp_attachment_is_image($post->ID)) {
        return;
    }
    
    ?>
    <div class="misc-pub-section aialttext-token-generate-section">
        <span class="aialttext-token-generate-label"><?php _e('AI Alt Text', 'aialttext-token'); ?></span>
        <button type="button" id="aialttext_token_nonce" class="button" data-image-id="<?php echo esc_attr($post->ID); ?>" style="background: linear-gradient(135deg, #9868E0 0%, #6740C5 100%) !important; color: white !important; border: none !important; border-radius: 4px !important; padding: 6px 12px 6px 28px !important; position: relative !important; box-shadow: 0 2px 4px rgba(103, 64, 197, 0.3) !important;">
            <?php _e('Generate Alt Text', 'aialttext-token'); ?>
        </button>
        <span class="spinner" id="aialttext_token_spinner" style="float: none; margin-top: 0;"></span>
        <div id="aialttext_token_result_container" class="aialttext-token-result-container" style="display: none;"></div>
    </div>
    <?php
}

// Add JS to the media edit page
add_action('admin_enqueue_scripts', 'aialttext_token_enqueue_attachment_scripts');

// Add button to media modal
add_filter('attachment_fields_to_edit', 'aialttext_token_add_generate_field', 10, 2);

function aialttext_token_add_generate_field($form_fields, $post) {
    // Only show for image attachments
    if (!wp_attachment_is_image($post->ID)) {
        return $form_fields;
    }
    
    $button_html = '<button type="button" class="button aialttext_token_nonce" data-image-id="' . esc_attr($post->ID) . '" style="background: linear-gradient(135deg, #9868E0 0%, #6740C5 100%) !important; color: white !important; border: none !important; border-radius: 4px !important; padding: 6px 12px 6px 28px !important; position: relative !important; box-shadow: 0 2px 4px rgba(103, 64, 197, 0.3) !important;">';
    $button_html .= __('Generate AI Alt Text', 'aialttext-token');
    $button_html .= '</button>';
    $button_html .= '<span class="spinner aialttext_token_spinner" style="float: none; margin: 0 10px;"></span>';
    $button_html .= '<div class="aialttext_token_result_container" style="display: none;"></div>';
    
    $form_fields['aialttext_token_generate'] = array(
        'label' => __('AI Alt Text', 'aialttext-token'),
        'input' => 'html',
        'html' => $button_html,
    );
    
    return $form_fields;
}

function aialttext_token_enqueue_attachment_scripts($hook) {
    // Check if we're on an attachment/media edit page
    global $post;
    
    // Enqueue on media edit screen, upload.php or media modal
    if (($hook == 'post.php' && isset($post) && $post->post_type == 'attachment') || 
        $hook == 'upload.php' || 
        $hook == 'media-upload.php' ||
        $hook == 'attachment.php') {
        
        // Ensure the WordPress REST API is available for our script
        wp_enqueue_script('wp-api');
            
        wp_enqueue_script('aialttext-token-attachment-js', AIALTTEXT_TOKEN_URL . 'admin/js/attachment.js', array('jquery'), AIALTTEXT_TOKEN_VERSION, true);
        wp_localize_script('aialttext-token-attachment-js', 'aialttext_token_attachment', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aialttext_token_nonce'),
            'generating' => __('Generating...', 'aialttext-token'),
            'success' => __('Alt text generated successfully!', 'aialttext-token'),
            'error' => __('Error generating alt text.', 'aialttext-token')
        ));
            
        wp_add_inline_style('wp-admin', '
/* Hide the duplicate compatibility field */
tr.compat-field-alt {
    display: none !important;
}

.aialttext_token_nonce, 
.aialttext-token-gutenberg-generate,
#aialttext_token_nonce,
.aialttext_token_copy_text,
#aialttext_token_apply_text {
    display: inline-block !important;
    background: linear-gradient(135deg, #9868E0 0%, #6740C5 100%) !important;
    border: none !important;
    color: white !important;
    text-decoration: none !important;
    text-shadow: none !important;
    box-shadow: 0 2px 4px rgba(103, 64, 197, 0.3) !important;
    position: relative !important;
    padding: 6px 12px 6px 28px !important;
    font-weight: 500 !important;
    border-radius: 4px !important;
    transition: all 0.2s ease !important;
    margin-right: 10px !important;
}

.aialttext_token_nonce:before,
.aialttext-token-gutenberg-generate:before,
#aialttext_token_nonce:before {
    content: "✨" !important;
    position: absolute !important;
    left: 10px !important;
    top: 50% !important;
    transform: translateY(-50%) !important;
}

.aialttext_token_nonce:hover,
.aialttext-token-gutenberg-generate:hover,
#aialttext_token_nonce:hover,
.aialttext_token_copy_text:hover,
#aialttext_token_apply_text:hover {
    background: linear-gradient(135deg, #8050D0 0%, #5730B5 100%) !important;
    color: white !important;
    transform: translateY(-1px) !important;
    box-shadow: 0 4px 8px rgba(103, 64, 197, 0.4) !important;
}

.aialttext_token_nonce:active,
.aialttext-token-gutenberg-generate:active,
#aialttext_token_nonce:active,
.aialttext_token_copy_text:active,
#aialttext_token_apply_text:active {
    transform: translateY(1px) !important;
    box-shadow: 0 1px 3px rgba(103, 64, 197, 0.3) !important;
}

.aialttext_token_result_container {
    background-color: #f8f8f8 !important;
    border-left: 4px solid #46b450 !important;
    padding: 10px !important;
    margin-top: 10px !important;
    border-radius: 4px !important;
    clear: both !important;
}

.aialttext-token-status {
    display: block !important;
    margin-top: 5px !important;
    padding: 3px 0 !important;
    font-size: 12px !important;
}

/* Highlight effect for alt text field when updated */
@keyframes highlightField {
    0% { background-color: #e7f7e3; }
    100% { background-color: transparent; }
}
.field-highlight {
    animation: highlightField 1.5s ease-out;
}

/* Force styling for WordPress admin context */
.wp-core-ui .button.aialttext_token_nonce,
.wp-core-ui .button.aialttext-token-gutenberg-generate,
.wp-core-ui .button#aialttext_token_nonce,
.wp-core-ui .button.aialttext_token_copy_text,
.wp-core-ui .button#aialttext_token_apply_text {
    background: linear-gradient(135deg, #9868E0 0%, #6740C5 100%) !important;
    color: white !important;
    border: none !important;
    text-shadow: none !important;
}
');
    }
}

// AJAX handler for applying generated alt text
add_action('wp_ajax_aialttext_token_apply_alt_text_ajax', 'aialttext_token_apply_alt_text_ajax');

function aialttext_token_apply_alt_text_ajax() {
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'aialttext_token_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
        return;
    }
    
    if (!current_user_can('upload_files')) {
        wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'aialttext-token')));
    }
    
    $image_id = isset($_POST['image_id']) ? intval($_POST['image_id']) : 0;
    $alt_text = isset($_POST['alt_text']) ? sanitize_text_field($_POST['alt_text']) : '';
    
    if (!$image_id || empty($alt_text)) {
        wp_send_json_error(array('message' => __('Invalid image or alt text.', 'aialttext-token')));
    }
    
    $result = update_post_meta($image_id, '_wp_attachment_image_alt', $alt_text);
    
    if ($result) {
        wp_send_json_success(array('message' => __('Alt text applied successfully!', 'aialttext-token')));
    } else {
        wp_send_json_error(array('message' => __('Error applying alt text.', 'aialttext-token')));
    }
}

// Add bulk action to media library
add_filter('bulk_actions-upload', 'aialttext_token_register_bulk_action');

function aialttext_token_register_bulk_action($bulk_actions) {
    $bulk_actions['generate_alt_text_token'] = __('Generate AI Alt Text (Tokens)', 'aialttext-token');
    return $bulk_actions;
}

// Handle bulk action
add_filter('handle_bulk_actions-upload', 'aialttext_token_handle_bulk_action', 10, 3);

function aialttext_token_handle_bulk_action($redirect_to, $action, $post_ids) {
    if ($action !== 'generate_alt_text_token') {
        return $redirect_to;
    }
    
    // Filter out non-image attachments
    $image_ids = array();
    foreach ($post_ids as $post_id) {
        if (wp_attachment_is_image($post_id)) {
            $image_ids[] = $post_id;
        }
    }
    
    if (empty($image_ids)) {
        return add_query_arg('bulk_ai_alt_text_token', 'no_images', $redirect_to);
    }
    
    // Store image IDs in a transient
    set_transient('aialttext_token_bulk_images', $image_ids, 3600); // 1 hour expiration
    
    // Redirect to processing page
    $process_url = admin_url('upload.php?page=aialttext-token-bulk-process');
    return $process_url;
}

// Hook into media upload - use a later priority to ensure all image data is processed
add_action('add_attachment', 'aialttext_token_auto_generate_alt', 20);
// Also add hook for when metadata is added
add_action('wp_generate_attachment_metadata', 'aialttext_token_auto_generate_on_metadata', 10, 2);

function aialttext_token_auto_generate_alt($attachment_id) {
    aialttext_token_debug_log("Auto-generate triggered for attachment ID: $attachment_id");
    
    // Check if auto-generation is enabled
    if (get_option('aialttext_token_auto_generate', 0) != 1) {
        aialttext_token_debug_log("Auto-generate is disabled in settings");
        return;
    }
    
    // Check if the attachment is an image
    if (!wp_attachment_is_image($attachment_id)) {
        aialttext_token_debug_log("Attachment $attachment_id is not an image");
        return;
    }
    
    // Check if the image already has alt text
    $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if (!empty($existing_alt)) {
        aialttext_token_debug_log("Attachment $attachment_id already has alt text: $existing_alt");
        return;
    }
    
    // Process immediately, don't wait
    aialttext_token_process_delayed_generation($attachment_id);
}

// Function to handle metadata-based generation
function aialttext_token_auto_generate_on_metadata($metadata, $attachment_id) {
    // This is another entry point that might work better in some WordPress setups
    aialttext_token_auto_generate_alt($attachment_id);
    return $metadata;
}

// Add custom action for delayed processing
add_action('aialttext_token_delayed_generate_alt', 'aialttext_token_process_delayed_generation');

function aialttext_token_process_delayed_generation($attachment_id) {
    aialttext_token_debug_log("Starting delayed alt text generation for attachment $attachment_id");
    
    // Check if auto-generation is still enabled
    if (get_option('aialttext_token_auto_generate', 0) != 1) {
        return;
    }
    
    // Double-check that the attachment is still an image
    if (!wp_attachment_is_image($attachment_id)) {
        return;
    }
    
    // Check if alt text was added in the meantime
    $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
    if (!empty($existing_alt)) {
        return;
    }
    
    // Get image data - ensure we have a valid URL before proceeding
    $image_url = wp_get_attachment_url($attachment_id);
    if (empty($image_url)) {
        aialttext_token_debug_log("Empty image URL for attachment $attachment_id");
        return;
    }
    
    $image_title = get_the_title($attachment_id);
    if (empty($image_title)) {
        $image_title = basename($image_url);
    }
    
    // Get additional context
    $post_id = wp_get_post_parent_id($attachment_id);
    $context = aialttext_token_get_image_context($attachment_id, $post_id);
    
    try {
        // Use the token-based image processor
        $result = AiAltText_Token_Image_Processor::process_attachment($attachment_id, array('context' => $context));
        
        if (!is_wp_error($result)) {
            // Log the auto-generation
            aialttext_token_debug_log(sprintf('AI Alt Text Token: Successfully auto-generated for image ID %d: %s', $attachment_id, $result['alt']));
        } else {
            aialttext_token_debug_log(sprintf('AI Alt Text Token: Failed to auto-generate for image ID %d: %s', $attachment_id, $result->get_error_message()));
        }
    } catch (Exception $e) {
        aialttext_token_debug_log(sprintf('AI Alt Text Token: Exception when auto-generating for image ID %d: %s', $attachment_id, $e->getMessage()));
    }
}

// Add indicator to media library grid view
add_filter('attachment_fields_to_edit', 'aialttext_token_add_indicator', 10, 2);

function aialttext_token_add_indicator($form_fields, $post) {
    if (wp_attachment_is_image($post->ID)) {
        $alt_text = get_post_meta($post->ID, '_wp_attachment_image_alt', true);
        
        if (!empty($alt_text)) {
            // Add a small indicator that this image has alt text
            $alt_preview = esc_html(substr($alt_text, 0, 40)) . (strlen($alt_text) > 40 ? '...' : '');
            $form_fields['alt']['input'] = $form_fields['alt']['input'] . 
                '<div class="aialttext-token-status" style="margin-top:5px;font-size:11px;color:#666;">' .
                '<span style="color:green;font-size:15px;">✓</span> Alt text: ' . $alt_preview . '</div>';
        } else {
            $form_fields['alt']['input'] = $form_fields['alt']['input'] . 
                '<div class="aialttext-token-status" style="margin-top:5px;font-size:11px;color:#666;">' .
                '<span style="color:#d63638;font-size:15px;">⚠</span> No alt text</div>';
        }
    }
    
    return $form_fields;
}

// Gutenberg integration
add_action('enqueue_block_editor_assets', 'aialttext_token_gutenberg_assets');

function aialttext_token_gutenberg_assets() {
    wp_enqueue_script(
        'aialttext-token-gutenberg',
        AIALTTEXT_TOKEN_URL . 'admin/js/gutenberg.js',
        array('wp-blocks', 'wp-i18n', 'wp-element', 'wp-editor', 'wp-components', 'wp-data', 'jquery'),
        AIALTTEXT_TOKEN_VERSION,
        true
    );
    
    wp_localize_script('aialttext-token-gutenberg', 'aialttext_token_gutenberg', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('aialttext_token_nonce'),
        'generating' => __('Generating...', 'aialttext-token'),
        'generate_text' => __('Generate AI Alt Text', 'aialttext-token'),
        'success' => __('Alt text generated successfully!', 'aialttext-token'),
        'error' => __('Error generating alt text.', 'aialttext-token')
    ));
}

// Add visual indicators to the media grid view
add_action('admin_head', 'aialttext_token_add_grid_indicators');

function aialttext_token_add_grid_indicators() {
    // Only on media library page and if indicators are enabled
    $screen = get_current_screen();
    if (!$screen || $screen->base !== 'upload' || get_option('aialttext_token_show_indicators', 1) != 1) {
        return;
    }
    
    ?>
    <style>
    .attachment-preview .has-alt-text:after {
        content: "ALT";
        position: absolute;
        bottom: 0;
        right: 0;
        background: rgba(0,170,0,0.8);
        color: white;
        font-size: 10px;
        padding: 2px 4px;
        border-radius: 3px 0 0 0;
    }
    .attachment-preview .no-alt-text:after {
        content: "NO ALT";
        position: absolute;
        bottom: 0;
        right: 0;
        background: rgba(220,53,69,0.8);
        color: white;
        font-size: 10px;
        padding: 2px 4px;
        border-radius: 3px 0 0 0;
    }
    </style>
    <script>
    jQuery(document).ready(function($) {
        // Add indicators to attachments in grid view
        function addGridIndicators() {
            $('.attachment').each(function() {
                var $this = $(this);
                var id = $this.data('id');
                
                // Skip if already processed
                if ($this.hasClass('alt-text-checked')) {
                    return;
                }
                
                $this.addClass('alt-text-checked');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aialttext_token_check_alt_text',
                        id: id,
                        nonce: '<?php echo wp_create_nonce('aialttext_token_check_alt'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            if (response.data.has_alt_text) {
                                $this.find('.attachment-preview').addClass('has-alt-text');
                            } else {
                                $this.find('.attachment-preview').addClass('no-alt-text');
                            }
                        }
                    }
                });
            });
        }
        
        // Run on page load
        addGridIndicators();
        
        // Run again when media is loaded via AJAX
        $(document).on('DOMNodeInserted', '.attachment', function() {
            addGridIndicators();
        });
    });
    </script>
    <?php
}

// AJAX handler to check for alt text
add_action('wp_ajax_aialttext_token_check_alt_text', 'aialttext_token_check_alt_text_ajax');

function aialttext_token_check_alt_text_ajax() {
    // Verify nonce - accept multiple nonce types for compatibility
    $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
    $valid_nonce = wp_verify_nonce($nonce, 'aialttext_token_check_alt') || 
                  wp_verify_nonce($nonce, 'aialttext_token_media_update') || 
                  wp_verify_nonce($nonce, 'aialttext_token_nonce');
                  
    if (!$valid_nonce) {
        wp_send_json_error(array('message' => 'Invalid security token'));
        return;
    }
    
    // Get attachment ID
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    
    if (!$id) {
        wp_send_json_error(array('message' => 'Invalid attachment ID'));
        return;
    }
    
    // Use DIRECT database query for maximum reliability
    global $wpdb;
    $alt_text = $wpdb->get_var($wpdb->prepare(
        "SELECT meta_value FROM $wpdb->postmeta 
         WHERE post_id = %d AND meta_key = '_wp_attachment_image_alt' 
         LIMIT 1",
        $id
    ));
    
    // If still not found, try traditional method as backup
    if ($alt_text === null) {
        $alt_text = get_post_meta($id, '_wp_attachment_image_alt', true);
    }
    
    wp_send_json_success(array(
        'has_alt_text' => !empty($alt_text),
        'alt_text' => $alt_text,
        'id' => $id
    ));
}

// Add filter to ensure alt text only applies to the alt field
add_filter('attachment_fields_to_save', 'aialttext_token_prevent_alttext_in_other_fields', 10, 2);

function aialttext_token_prevent_alttext_in_other_fields($post, $attachment) {
    // Don't do anything for non-image attachments
    if (!wp_attachment_is_image($post['ID'])) {
        return $post;
    }
    
    // Only check if alt text is being set
    if (isset($attachment['alt'])) {
        $alt_text = $attachment['alt'];
        
        // Store only in the alt text field and not elsewhere
        update_post_meta($post['ID'], '_wp_attachment_image_alt', $alt_text);
        
        // Ensure it doesn't contaminate other fields like caption or description
        if (isset($post['post_excerpt']) && $post['post_excerpt'] === $alt_text) {
            $post['post_excerpt'] = '';
        }
        if (isset($post['post_content']) && $post['post_content'] === $alt_text) {
            $post['post_content'] = '';
        }
    }
    
    return $post;
}

// Add clean-up on post save to ensure alt text doesn't leak into other fields
add_action('edit_attachment', 'aialttext_token_clean_attachment_fields');

function aialttext_token_clean_attachment_fields($post_id) {
    // Only process image attachments
    if (!wp_attachment_is_image($post_id)) {
        return;
    }
    
    // Get the alt text value
    $alt_text = get_post_meta($post_id, '_wp_attachment_image_alt', true);
    if (empty($alt_text)) {
        return;
    }
    
    // Get post data
    $post = get_post($post_id);
    $updated = false;
    
    // Check if caption or description is same as alt text (indicating it leaked)
    if ($post->post_excerpt === $alt_text) {
        // Clear caption
        wp_update_post(array(
            'ID' => $post_id,
            'post_excerpt' => ''
        ));
        $updated = true;
    }
    
    if ($post->post_content === $alt_text) {
        // Clear description
        wp_update_post(array(
            'ID' => $post_id,
            'post_content' => ''
        ));
        $updated = true;
    }
    
    if ($updated && defined('AIALTTEXT_TOKEN_DEBUG') && AIALTTEXT_TOKEN_DEBUG) {
        error_log(sprintf('AI Alt Text Token: Cleaned caption/description fields for attachment %d', $post_id));
    }
}

// Add simple visual indicators to the media library
add_filter('manage_media_columns', 'aialttext_token_add_alt_column');
add_action('manage_media_custom_column', 'aialttext_token_alt_column_content', 10, 2);

function aialttext_token_add_alt_column($columns) {
    // Only add if indicators are enabled
    if (get_option('aialttext_token_show_indicators', 1) != 1) {
        return $columns;
    }
    
    // Add our column after title
    $new_columns = array();
    foreach ($columns as $key => $value) {
        $new_columns[$key] = $value;
        
        if ($key === 'title') {
            $new_columns['alt_text_status'] = __('Alt Text', 'aialttext-token');
        }
    }
    
    return $new_columns;
}

function aialttext_token_alt_column_content($column_name, $post_id) {
    if ($column_name !== 'alt_text_status' || !wp_attachment_is_image($post_id)) {
        return;
    }
    
    $alt_text = get_post_meta($post_id, '_wp_attachment_image_alt', true);
    
    if (!empty($alt_text)) {
        echo '<div style="color:#46b450;font-weight:bold;">✓ ' . __('Yes', 'aialttext-token') . '</div>';
    } else {
        echo '<div style="color:#dc3232;font-weight:bold;">✗ ' . __('No', 'aialttext-token') . ' <button type="button" class="button button-small aialttext_token_nonce" data-image-id="' . esc_attr($post_id) . '">' . __('Generate', 'aialttext-token') . '</button></div>';
    }
}

// Add version upgrade check
add_action('admin_init', 'aialttext_token_check_version_upgrade');

function aialttext_token_check_version_upgrade() {
    $current_version = get_option('aialttext_token_version', '0');
    
    // If this is a first install or an upgrade
    if (version_compare($current_version, AIALTTEXT_TOKEN_VERSION, '<')) {
        // Perform any database upgrades if needed
        
        // Update stored version number
        update_option('aialttext_token_version', AIALTTEXT_TOKEN_VERSION);
        
        // Set debug mode on upgrade to help troubleshoot if needed
        if (defined('AIALTTEXT_TOKEN_DEBUG') && !AIALTTEXT_TOKEN_DEBUG) {
            update_option('aialttext_token_debug_upgrade', true);
        }
        
        // Clear any caches
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
    }
}