jQuery(document).ready(function($) {
    // Basic initialization message
    console.log('AI Alt Text Generator initialized');
    
    // Function to find all alt text fields in the DOM
    function findAltTextFields() {
        // Store all possible alt text fields
        const fields = {
            // Standard field in the media modal
            primary: document.getElementById('attachment-details-two-column-alt-text'),
            
            // Data attribute selector fields
            dataSetting: document.querySelector('.attachment-details [data-setting="alt"] input, .attachment-details [data-setting="alt"] textarea'),
            
            // Fallback to any field that might contain alt text
            fallback: document.querySelector('textarea[aria-describedby="alt-text-description"]')
        };
        
        // Return the first valid field found
        return fields.primary || fields.dataSetting || fields.fallback;
    }
    
    // Function to update all possible alt text fields
    function updateAllAltTextFields(altText) {
        if (!altText) return false;
        
        console.log('Updating all Alt Text fields with:', altText);
        let updated = false;
        
        // 1. Primary field
        const primaryField = document.getElementById('attachment-details-two-column-alt-text');
        if (primaryField) {
            primaryField.value = altText;
            $(primaryField).val(altText).trigger('input').trigger('change');
            
            // Add visual highlight
            $(primaryField).addClass('field-highlight');
            setTimeout(() => $(primaryField).removeClass('field-highlight'), 2000);
            
            console.log('Updated primary alt text field:', altText);
            updated = true;
        }
        
        // 2. Data attribute selector
        const dataSettingField = document.querySelector('.attachment-details [data-setting="alt"] input, .attachment-details [data-setting="alt"] textarea');
        if (dataSettingField && dataSettingField !== primaryField) {
            dataSettingField.value = altText;
            $(dataSettingField).val(altText).trigger('input').trigger('change');
            
            // Add visual highlight
            $(dataSettingField).addClass('field-highlight');
            setTimeout(() => $(dataSettingField).removeClass('field-highlight'), 2000);
            
            console.log('Updated data-setting alt text field:', altText);
            updated = true;
        }
        
        // 3. Special aria field
        const ariaField = document.querySelector('textarea[aria-describedby="alt-text-description"]');
        if (ariaField && ariaField !== primaryField && ariaField !== dataSettingField) {
            ariaField.value = altText;
            $(ariaField).val(altText).trigger('input').trigger('change');
            
            // Add visual highlight
            $(ariaField).addClass('field-highlight');
            setTimeout(() => $(ariaField).removeClass('field-highlight'), 2000);
            
            console.log('Updated aria alt text field:', altText);
            updated = true;
        }
        
        // 4. WordPress media frame update
        if (wp && wp.media && wp.media.frame) {
            // Try to update the media frame directly
            try {
                const selection = wp.media.frame.state().get('selection');
                const attachment = selection.first();
                
                if (attachment) {
                    // Update the model
                    attachment.set('alt', altText);
                    console.log('Updated media frame attachment model');
                    updated = true;
                }
            } catch (e) {
                console.log('Error updating media frame:', e);
            }
        }
        
        console.log('Field update success:', updated);
        return updated;
    }
    
    // Main function to handle alt text generation
    function handleGenerateAltText(button, imageId, postId) {
        console.log('Generate alt text triggered for image ID:', imageId);
        
        const spinner = button.closest('.field, .misc-pub-section').find('.spinner, .aialttext_spinner');
        const resultContainer = button.closest('.field, .misc-pub-section').find('.aialttext_result_container');
        
        // Show spinner and disable button
        spinner.addClass('is-active');
        button.prop('disabled', true);
        button.text(aialttext_attachment.generating || 'Generating...');
        
        // Hide previous results
        resultContainer.hide();
        
        // Find alt text fields before generation
        const altField = findAltTextFields();
        if (altField) {
            console.log('Found existing Alt Text field:', altField.value);
        } else {
            console.log('Alt Text field not found before generation');
        }
        
        // AJAX request to generate alt text
        $.ajax({
            url: aialttext_token_attachment.ajax_url,
            type: 'POST',
            data: {
                action: 'aialttext_token_generate_single',
                nonce: aialttext_token_attachment.nonce,
                image_id: imageId,
                post_id: postId || 0
            },
            success: function(response) {
                // Hide spinner
                spinner.removeClass('is-active');
                
                if (response.success) {
                    const altText = response.data.alt_text;
                    console.log('Successfully generated alt text:', altText);
                    
                    // Step 1: Update the database
                    $.ajax({
                        url: aialttext_token_attachment.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'aialttext_token_force_update_and_get_alt',
                            nonce: aialttext_token_attachment.nonce,
                            image_id: imageId,
                            alt_text: altText
                        },
                        success: function(dbResponse) {
                            console.log('Database update response:', dbResponse);
                            
                            if (dbResponse.success) {
                                console.log('Alt text saved to database:', altText);
                                
                                // Step 2: Update UI fields
                                const fieldUpdateSuccess = updateAllAltTextFields(altText);
                                
                                // Double check after a short delay
                                setTimeout(function() {
                                    // If the first update failed, try again
                                    if (!fieldUpdateSuccess) {
                                        console.log('First update attempt failed, trying again...');
                                        updateAllAltTextFields(altText);
                                    }
                                    
                                    // Verify field values
                                    const altField = findAltTextFields();
                                    if (altField) {
                                        console.log('Final field check - Current value:', altField.value);
                                        console.log('Field update successful:', altField.value === altText);
                                    }
                                }, 500);
                                
                                // Show success message
                                resultContainer.html(`
                                    <div class="notice notice-success inline" style="padding: 8px; margin: 5px 0;">
                                        <p><span style="color:#46b450; font-weight:bold;">✓</span> Alt text generated and saved!</p>
                                        <p style="margin: 4px 0 0; font-size: 12px;">Value: "${altText}"</p>
                                    </div>
                                `);
                                resultContainer.show();
                            } else {
                                // Show database error
                                resultContainer.html(`
                                    <div class="notice notice-error inline" style="padding: 8px; margin: 5px 0;">
                                        <p><span style="color:#d63638;">⚠</span> Error saving alt text to the database.</p>
                                    </div>
                                `);
                                resultContainer.show();
                            }
                        },
                        error: function(error) {
                            console.log('Database update error:', error);
                            
                            // Show database error
                            resultContainer.html(`
                                <div class="notice notice-error inline" style="padding: 8px; margin: 5px 0;">
                                    <p><span style="color:#d63638;">⚠</span> Error connecting to the server.</p>
                                </div>
                            `);
                            resultContainer.show();
                        }
                    });
                } else {
                    console.log('Alt text generation failed:', response);
                    
                    // Show error message
                    resultContainer.html(`
                        <div class="notice notice-error inline" style="padding: 8px; margin: 5px 0;">
                            <p><span style="color:#d63638;">⚠</span> ${response.data?.message || 'Error generating alt text.'}</p>
                        </div>
                    `);
                    resultContainer.show();
                }
                
                // Re-enable button
                button.prop('disabled', false).text('Generate AI Alt Text');
            },
            error: function(error) {
                console.log('AJAX request error:', error);
                
                // Hide spinner
                spinner.removeClass('is-active');
                
                // Show error message
                resultContainer.html(`
                    <div class="notice notice-error inline" style="padding: 8px; margin: 5px 0;">
                        <p><span style="color:#d63638;">⚠</span> ${aialttext_attachment.error || 'Error generating alt text.'}</p>
                    </div>
                `);
                resultContainer.show();
                
                // Re-enable button
                button.prop('disabled', false).text('Generate AI Alt Text');
            }
        });
    }
    
    // Handle button click events
    $(document).on('click', '#aialttext_generate_single, .aialttext_generate_single', function() {
        const imageId = $(this).data('image-id');
        const postId = $('#post_ID').val();
        handleGenerateAltText($(this), imageId, postId);
    });
    
    // Handle button for Gutenberg sidebar
    $(document).on('click', '.aialttext-gutenberg-generate', function() {
        const imageId = $(this).data('image-id');
        handleGenerateAltText($(this), imageId);
    });
    
    // Hide compat fields that might cause confusion
    function hideCompatFields() {
        $('.compat-field-alt').hide();
    }
    
    // Run immediately and after a delay to catch dynamically loaded content
    hideCompatFields();
    setTimeout(hideCompatFields, 500);
    
    // Use modern MutationObserver instead of deprecated DOMNodeInserted
    if (window.MutationObserver) {
        const observer = new MutationObserver(function(mutations) {
            // Hide compat fields when DOM changes
            hideCompatFields();
            
            // Check if the alt field was added to the DOM
            for (const mutation of mutations) {
                if (mutation.addedNodes && mutation.addedNodes.length) {
                    const addedNodes = Array.from(mutation.addedNodes);
                    for (const node of addedNodes) {
                        if (node.querySelector) {
                            const altField = node.querySelector('#attachment-details-two-column-alt-text, [data-setting="alt"] input, [data-setting="alt"] textarea');
                            if (altField) {
                                console.log('Alt text field added to DOM:', altField.value);
                            }
                        }
                    }
                }
            }
        });
        
        // Observe the entire document for changes
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }
    
    // Initial check for alt fields
    setTimeout(function() {
        const altField = findAltTextFields();
        if (altField) {
            console.log('Initial check - Found alt text field:', altField.value);
        } else {
            console.log('Initial check - Alt text field not found');
        }
    }, 1000);
});