jQuery(function($){
    function ajax(action, data){ 
        data = data || {}; 
        data.action = action; 
        data.nonce = aialttext_token_ajax.nonce; 
        return $.ajax({
            url: aialttext_token_ajax.ajax_url,
            type: 'POST',
            data: data
        }); 
    }
    
    function param(name){ 
        var match = new RegExp('[?&]' + name + '=([^&#]*)').exec(location.search); 
        return match ? decodeURIComponent(match[1].replace(/\+/g,' ')) : null; 
    }
    
    function banner(kind, text){ 
        var className = kind === 'success' ? 'notice-success' : (kind === 'warn' ? 'notice-warning' : 'notice-error'); 
        $('<div class="notice ' + className + ' is-dismissible"><p>' + text + '</p></div>').insertBefore('.wrap h1'); 
    }

    function renderPricing(packages){ 
        var $wrapper = $('#aialttext_token_pricing').empty(); 
        $.each(packages, function(key, pkg){ 
            var badge = pkg.popular ? '<div class="badge">Most Popular</div>' : ''; 
            var html = '<div class="aialttext-token-plan">' +
                      '<div class="head"><h3>' + pkg.name + '</h3>' + badge + '</div>' +
                      '<div class="tokens">' + pkg.tokens + ' tokens</div>' +
                      '<div class="price">$' + Number(pkg.price).toFixed(2) + '</div>' +
                      (pkg.discount ? '<div class="save">Save ' + pkg.discount + '%</div>' : '') +
                      '<button class="button button-primary buy" data-id="' + key + '">Purchase</button></div>';
            $wrapper.append(html);
        }); 
    }
    // If the balance widget is present, load it immediately.
if ($('#aialttext-token-balance').length) {
    $('#aialttext_token_refresh_balance').trigger('click');
  }

    // Load pricing on account page
    if ($('#aialttext_token_pricing').length){
        ajax('aialttext_token_get_pricing').done(function(result){ 
            if (result && result.success && result.data && result.data.packages) {
                renderPricing(result.data.packages); 
            }
        });
    }

    // Balance refresh
    $('#aialttext_token_refresh_balance').on('click', function(e){ 
        e.preventDefault(); 
        ajax('aialttext_token_refresh_balance').done(function(result){ 
            if (result.success) {
                $('#aialttext-token-balance').text(result.data.balance); 
            }
        }); 
    });

    // Link account
    $('#aialttext_token_link_account_btn').on('click', function(e){ 
        e.preventDefault(); 
        var email = $('#aialttext_token_account_email').val();
        ajax('aialttext_token_link_account', {email: email}).done(function(result){ 
            if (result.success){ 
                banner('success', 'Account linked successfully.'); 
                $('#aialttext-token-balance').text(result.data.balance || 0);
                setTimeout(function(){ location.reload(); }, 1500);
            } else { 
                banner('error', (result.data && result.data.message) || 'Failed to link account'); 
            } 
        }); 
    });

    // Purchase flow
    $(document).on('click', '.buy', function(e){
        e.preventDefault(); 
        var $button = $(this);
        var packageId = $button.data('id'); 
        $button.prop('disabled', true).text('Redirecting…');
        
        ajax('aialttext_token_purchase_tokens', {package_id: packageId}).done(function(result){
            if (result && result.success && result.data && result.data.checkout_url){ 
                location.href = result.data.checkout_url; 
            } else { 
                $button.prop('disabled', false).text('Purchase'); 
                banner('error', (result && result.data && result.data.message) || 'Unable to start checkout'); 
            }
        }).fail(function(){ 
            $button.prop('disabled', false).text('Purchase'); 
            banner('error', 'Connection error'); 
        });
    });

    function confirmAndRefresh(sessionId, attempts){
        attempts = attempts || 0;
        ajax('aialttext_token_confirm_payment', {session_id: sessionId}).done(function(result){
            if (result && result.success){
                var added = result.data.tokens_added || 0;
                var pkg = result.data.package_tokens || added;
                if (typeof result.data.balance !== 'undefined') {
                    $('#aialttext-token-balance').text(result.data.balance);
                }
                banner('success', 'Payment confirmed. ' + pkg + ' tokens added to your account.');
                if (history.replaceState) {
                    history.replaceState({}, document.title, location.pathname);
                }
            } else {
                if (attempts < 10) {
                    setTimeout(function(){ 
                        confirmAndRefresh(sessionId, attempts + 1); 
                    }, 1000 * (attempts + 1));
                } else {
                    banner('error', 'Payment confirmation failed after multiple attempts: ' + ((result && result.data && result.data.message) || 'Unknown error'));
                }
            }
        }).fail(function(){
            if (attempts < 10) {
                setTimeout(function(){ 
                    confirmAndRefresh(sessionId, attempts + 1); 
                }, 1000 * (attempts + 1));
            } else {
                banner('error', 'Network error while confirming payment.');
            }
        });
    }

    var payment = param('payment');
    var sessionId = param('session_id');
    if (payment === 'success' && sessionId){ 
        banner('warn', 'Thank you for your purchase! We\'re confirming your payment and updating your balance…'); 
        confirmAndRefresh(sessionId, 0); 
    }

    // Load usage statistics
    $('#aialttext_token_load_usage').on('click', function(e){
        e.preventDefault();
        var $button = $(this);
        $button.prop('disabled', true).text('Loading...');
        
        ajax('aialttext_token_load_usage').done(function(result){
            if (result && result.success && result.data && result.data.usage) {
                var $list = $('#aialttext_token_usage_list');
                $list.empty();
                
                if (result.data.usage.length === 0) {
                    $list.html('<p><em>No usage history found.</em></p>');
                } else {
                    var html = '<table class="wp-list-table widefat fixed striped">' +
                              '<thead><tr><th>Date</th><th>Image Title</th><th>Tokens Used</th></tr></thead><tbody>';
                    
                    $.each(result.data.usage, function(index, item){
                        var date = new Date(item.created_at).toLocaleDateString();
                        var title = item.image_title || 'Untitled';
                        html += '<tr><td>' + date + '</td><td>' + title + '</td><td>' + item.tokens_used + '</td></tr>';
                    });
                    
                    html += '</tbody></table>';
                    $list.html(html);
                }
                
                $button.hide();
            } else {
                banner('error', 'Failed to load usage history');
                $button.prop('disabled', false).text('Load Usage History');
            }
        }).fail(function(){
            banner('error', 'Connection error while loading usage history');
            $button.prop('disabled', false).text('Load Usage History');
        });
    });

    // Processing page functionality
    var processing = false;
    var offset = 0;
    var totalProcessed = 0;
    var totalErrors = 0;
    var batchSize = parseInt(aialttext_token_ajax.batch_size) || 25;
    var totalImages = 0;
    var initialTotalImages = 0;
    var timeRemaining = 'Unknown';
    var maxImages = parseInt(aialttext_token_ajax.max_images) || 0;

    function updateProgressBar(progress) {
        progress = Math.min(progress, 100);
        $('#aialttext_token_progress').css('width', progress + '%').text(Math.round(progress) + '%');
    }

    function updateStatsDisplay() {
        $('#aialttext_token_total_images').text(totalImages);
        $('#aialttext_token_processed_count').text(totalProcessed);
        $('#aialttext_token_error_count').text(totalErrors);
        $('#aialttext_token_time_remaining').text(timeRemaining !== 'Unknown' ? timeRemaining + ' min' : 'Unknown');
    }

    function updateStatusMessage() {
        var message = '';
        
        if (totalImages > 0) {
            var denom = Math.max(1, initialTotalImages);
            var percentComplete = Math.min(100, Math.round((totalProcessed / denom) * 100));
            message = 'Progress: ' + percentComplete + '% complete';
            
            if (timeRemaining !== 'Unknown') {
                message += ' • Estimated time remaining: ' + timeRemaining + ' minutes';
            }
        } else {
            message = 'Scanning for images...';
        }
        
        $('#aialttext_token_process_status').html(message);
    }

    function scanForImages() {
        $('#aialttext_token_process_button').prop('disabled', true).text('Scanning...');
        $('#aialttext_token_process_status').html('Scanning your media library for images without alt text...');
        
        ajax('aialttext_token_scan_images_ajax').done(function(response) {
            if (response.success) {
                totalImages = response.total_images;
                initialTotalImages = totalImages; // freeze the original total
                
                // If max_images is set, limit total
                if (maxImages > 0 && totalImages > maxImages) {
                    totalImages = maxImages;
                }
                
                $('#aialttext_token_process_button').prop('disabled', false).text('Start Processing ' + totalImages + ' Images');
                $('#aialttext_token_process_status').html('Found ' + totalImages + ' images without alt text');
                updateStatsDisplay();
            } else {
                $('#aialttext_token_process_status').html('Error: ' + response.message);
                $('#aialttext_token_process_button').prop('disabled', false).text('Retry Scan');
            }
        }).fail(function(jqXHR, textStatus, errorThrown) {
            $('#aialttext_token_process_status').html('An error occurred during scan. Please try again. Details: ' + textStatus + ' ' + errorThrown);
            $('#aialttext_token_process_button').prop('disabled', false).text('Retry Scan');
        });
    }

    function processImages() {
        if (!processing) return;

        // Check if we've hit the max images
        if (maxImages > 0 && totalProcessed >= maxImages) {
            $('#aialttext_token_process_status').prepend('<strong>Processing completed!</strong> Reached maximum image limit of ' + maxImages + '.<br>');
            $('#aialttext_token_process_button').prop('disabled', false).text('Process More Images');
            $('#aialttext_token_stop_button').hide();
            processing = false;
            return;
        }

        // Calculate batch size for final batch if max_images is set
        var currentBatchSize = batchSize;
        if (maxImages > 0 && (totalProcessed + batchSize) > maxImages) {
            currentBatchSize = maxImages - totalProcessed;
        }

        ajax('aialttext_token_process_images_ajax', {
            offset: offset,
            total_processed: totalProcessed,
            batch_size: currentBatchSize,
            max_images: maxImages
        }).done(function(response) {
            if (response.success) {
                totalProcessed += parseInt(response.processed || 0, 10);
                totalErrors += response.errors || 0;
                offset = response.offset || (offset + currentBatchSize);
                timeRemaining = response.time_remaining || 'Unknown';
                
                if (response.total_images !== undefined) {
                    totalImages = maxImages > 0 ? Math.min(response.total_images, maxImages) : response.total_images;
                }
                
                // Update UI elements
                var denom = Math.max(1, initialTotalImages);
                var percent = Math.min(100, Math.round((totalProcessed / denom) * 100));
                updateProgressBar(percent);
                updateStatsDisplay();
                updateStatusMessage();
                
                // Update latest image preview if available
                if (response.latest_image) {
                    $('#aialttext_token_preview_image').attr('src', response.latest_image.url);
                    $('#aialttext_token_preview_title').text(response.latest_image.title);
                    $('#aialttext_token_preview_alt').text(response.latest_image.alt_text);
                    $('#aialttext_token_latest_image').show();
                }
                
                // Add error messages if any
                if (response.error_messages && response.error_messages.length > 0) {
                    var errorList = '<div class="error-details" style="display:none;"><ul><li>' + 
                        response.error_messages.join('</li><li>') + '</li></ul></div>';
                    
                    // Append to error details rather than replace
                    if ($('#aialttext_token_error_details').html()) {
                        $('#aialttext_token_error_details').append('<a href="#" class="toggle-errors">Show error details</a>' + errorList);
                    } else {
                        $('#aialttext_token_error_details').html('<a href="#" class="toggle-errors">Show error details</a>' + errorList);
                    }
                }

                // Check if we should continue processing
                if (!response.completed && processing) {
                    // Add a small delay between batches to prevent overloading
                    setTimeout(function() {
                        processImages();
                    }, 500);
                } else {
                    $('#aialttext_token_process_button').prop('disabled', false).text('Start Processing Images');
                    $('#aialttext_token_stop_button').hide();
                    processing = false;
                    if (response.completed) {
                        $('#aialttext_token_process_status').prepend('<div class="notice notice-success"><p><strong>Processing completed!</strong></p></div>');
                    }
                }
            } else {
                // Even on error, try to recover
                $('#aialttext_token_process_status').append('<p class="error">Error occurred but attempting to continue: ' + (response.message || 'Unknown error') + '</p>');
                
                // Still increment offset to move past problematic images
                offset += currentBatchSize;
                
                // If we're still processing, continue with next batch
                if (processing) {
                    // Scale the delay based on errors to prevent hammering the server when issues occur
                    var delayTime = totalErrors > 0 ? 1500 : 500;
                    setTimeout(function() {
                        processImages();
                    }, delayTime);
                } else {
                    $('#aialttext_token_process_button').prop('disabled', false).text('Resume Processing');
                    $('#aialttext_token_stop_button').hide();
                }
            }
        }).fail(function(jqXHR, textStatus, errorThrown) {
            // Log the error but continue
            $('#aialttext_token_process_status').append('<p class="error">AJAX error occurred: ' + textStatus + ' ' + errorThrown + ' - attempting to continue</p>');
            
            // Increment offset to move forward
            offset += currentBatchSize;
            
            // If still processing, try next batch
            if (processing) {
                setTimeout(function() {
                    processImages();
                }, 3000); // Even longer delay after AJAX error
            } else {
                $('#aialttext_token_process_button').prop('disabled', false).text('Resume Processing');
                $('#aialttext_token_stop_button').hide();
            }
        });
    }

    $('#aialttext_token_process_button').click(function() {
        if (totalImages === 0) {
            scanForImages();
            return;
        }
        
        if (!processing) {
            processing = true;
            // Keep offset if resuming, reset if starting new
            if ($(this).text().indexOf('Resume') === -1) {
                offset = 0;
                totalProcessed = 0;
                totalErrors = 0;
            }
            
            $(this).prop('disabled', true).text('Processing...');
            $('#aialttext_token_stop_button').show();
            $('#aialttext_token_progress_bar').show();
            updateProgressBar(0);
            $('#aialttext_token_process_status').html('');
            $('#aialttext_token_error_details').html('');
            processImages();
        }
    });

    $('#aialttext_token_stop_button').click(function() {
        processing = false;
        $(this).hide();
        $('#aialttext_token_process_button').prop('disabled', false).text('Resume Processing');
    });
    
    $(document).on('click', '.toggle-errors', function(e) {
        e.preventDefault();
        $(this).next('.error-details').toggle();
        var text = $(this).next('.error-details').is(':visible') ? 'Hide error details' : 'Show error details';
        $(this).text(text);
    });
    
    // Scan for images when the process page loads
    if ($('#aialttext_token_process_button').length) {
        scanForImages();
    }

    // If single-generate button is present, check balance and disable if zero
    if ($('#aialttext-token-generate-single').length){
        ajax('aialttext_token_refresh_balance').done(function(result){
            if (result && result.success){
                var balance = parseInt(result.data.balance || 0, 10);
                if (balance <= 0){ 
                    $('#aialttext-token-generate-single').prop('disabled', true).after(' <em>(no tokens)</em>'); 
                }
            }
        });
    }

    // Single generation in attachment edit screen
    $(document).on('click', '#aialttext-token-generate-single', function(){
        var id = $(this).data('id');
        var $status = $('#aialttext-token-single-status'); 
        $status.text('Generating…');
        
        ajax('aialttext_token_generate_single', {attachment_id: id}).done(function(result){
            if (result && result.success){ 
                $status.text('Alt text generated and saved.'); 
            } else { 
                $status.text('Failed: ' + ((result && result.data && result.data.message) || 'Error')); 
            }
        }).fail(function(){ 
            $status.text('Network error'); 
        });
    });

    // Handle single generation from media library buttons
    $(document).on('click', '.aialttext_token_generate_single', function(e){
        e.preventDefault();
        var $button = $(this);
        var imageId = $button.data('image-id');
        
        if (!imageId) return;
        
        $button.prop('disabled', true).text('Generating...');
        
        ajax('aialttext_token_generate_single_ajax', {
            image_id: imageId,
            nonce: aialttext_token_ajax.nonce
        }).done(function(result){
            if (result && result.success) {
                $button.text('Generated!').css('background-color', '#46b450');
                
                // Try to update the alt text field if visible
                var altField = $('input[name="attachments[' + imageId + '][alt]"], #attachment-details-two-column-alt-text');
                if (altField.length) {
                    altField.val(result.data.alt_text).trigger('change');
                    altField.css('background-color', '#e7f7e3');
                    setTimeout(function() {
                        altField.css('background-color', '');
                    }, 2000);
                }
                
                setTimeout(function() {
                    $button.prop('disabled', false).text('Generate').css('background-color', '');
                }, 3000);
            } else {
                $button.prop('disabled', false).text('Error - Retry');
                alert('Error: ' + (result.data && result.data.message || 'Failed to generate alt text'));
            }
        }).fail(function(){
            $button.prop('disabled', false).text('Error - Retry');
            alert('Network error occurred');
        });
    });
});